package plugins.ylemontag.sequencecomparator;

import icy.sequence.Sequence;
import icy.type.DataType;
import icy.type.collection.array.Array1DUtil;

/**
 * 
 * @author Yoann
 *
 * Base class for comparison objects
 */
public abstract class Comparator<T>
{
	/**
	 * Exception thrown when trying to compare sequences with incompatible sizes
	 */
	public static class IncompatibleSizes extends IllegalArgumentException
	{
		private static final long serialVersionUID = 1L;
		
		public IncompatibleSizes()
		{
			super("The compared sequences do not have compatible sizes.");
		}
	}
	
	private String _distanceName;
	
	/**
	 * Constructor
	 */
	protected Comparator(String distanceName)
	{
		_distanceName = distanceName;
	}
	
	/**
	 * Name of the distance measure
	 */
	public String getDistanceName()
	{
		return _distanceName;
	}
	
	@Override
	public String toString() {
		return getDistanceName();
	}
	
	/**
	 * Comparison function
	 */
	public T compare(Sequence ref, Sequence seq)
	{
		T out = allocateResult();
		compare(ref, seq, out);
		return out;
	}
	
	/**
	 * Comparison function with pre-allocated output result
	 */
	public void compare(Sequence ref, Sequence seq, T out)
	{
		try {
			compare(ref, seq, out, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException("Comparison unexpectedly interrupted");
		}
	}
	
	/**
	 * Comparison function
	 */
	public T compare(Sequence ref, Sequence seq, Controller controller)
		throws Controller.CanceledByUser
	{
		T out = allocateResult();
		compare(ref, seq, out, controller);
		return out;
	}
	
	/**
	 * Comparison function with pre-allocated output result
	 */
	public void compare(Sequence ref, Sequence seq, T out, Controller controller)
		throws Controller.CanceledByUser
	{
		if(ref==null) {
			throw new IllegalArgumentException("Argument 'ref' cannot be null");
		}
		if(seq==null) {
			throw new IllegalArgumentException("Argument 'seq' cannot be null");
		}
		if(out==null) {
			throw new IllegalArgumentException("Argument 'out' cannot be null");
		}
		if(controller==null) {
			throw new IllegalArgumentException("Argument 'controller' cannot be null");
		}
		synchronized (controller.getMutex())
		{
			controller.reset();
			implCompare(ref, seq, out, controller);
			controller.checkPoint();
		}
	}
	
	/**
	 * Allocate a new T object
	 */
	protected abstract T allocateResult();
	
	/**
	 * Actual comparison function, to implement
	 */
	protected abstract void implCompare(Sequence ref, Sequence seq,  T out, Controller controller)
		throws Controller.CanceledByUser;
	
	/**
	 * Useful function to extract the pixel values of a sequence in a given XY plane,
	 * with conversion to double if necessary
	 * @warning The returned array must not be written, although this constraint is not checked
	 */
	protected static double[] getReadOnlyDataXYAsDouble(Sequence seq, int t, int z, int c)
	{
		DataType datatype = seq.getDataType_();
		if(datatype==DataType.DOUBLE) {
			return seq.getDataXYAsDouble(t, z, c);
		}
		else {
			return Array1DUtil.arrayToDoubleArray(seq.getDataXY(t, z, c), datatype.isSigned());
		}
	}
	
	/**
	 * Return a human-readable label describing the compared sequences
	 */
	protected static String getComparisonLabel(Sequence ref, Sequence seq)
	{
		return ref.getName() + " \u2260 " + seq.getName();
	}
}
