package plugins.stef.roi.quantify;

import java.awt.Dimension;
import java.awt.Point;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JComponent;

import org.apache.poi.xssf.usermodel.XSSFWorkbook;

import icy.gui.dialog.MessageDialog;
import icy.math.Line3DIterator;
import icy.math.MathUtil;
import icy.roi.BooleanMask2D;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.sequence.Sequence;
import icy.type.geom.Line3D;
import icy.type.point.Point3D;
import icy.type.rectangle.Rectangle5D;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzGUI;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.vars.gui.swing.WorkbookEditor;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.lang.VarWorkbook;
import plugins.adufour.workbooks.IcySpreadSheet;
import plugins.adufour.workbooks.Workbooks;
import plugins.adufour.workbooks.Workbooks.WorkbookFormat;
import plugins.kernel.roi.roi2d.ROI2DShape;
import plugins.kernel.roi.roi3d.ROI3DLine;
import plugins.kernel.roi.roi3d.ROI3DPoint;
import plugins.kernel.roi.roi3d.ROI3DPolyLine;
import plugins.kernel.roi.roi3d.ROI3DShape;

/**
 * Path profiler plugin class.<br>
 * Computes intensity profil along path (contour for area ROI) for the given list of ROI and return result in XLSX workbook format
 * where we have one sheet per ROI.
 * 
 * @author Stephane
 */
public class PathIntensityProfiler extends EzPlug implements Block, EzStoppable
{
    // VAR
    public final VarSequence varSequence;
    public final VarROIArray varRois;
    public final VarWorkbook varWorkbook;

    public PathIntensityProfiler()
    {
        super();

        varSequence = new VarSequence("Sequence", null);
        varRois = new VarROIArray("Roi(s)");
        varWorkbook = new VarWorkbook("WorkBook", Workbooks.createEmptyWorkbook(WorkbookFormat.XLSX));
    }

    @Override
    protected void initialize()
    {
        final WorkbookEditor viewer = (WorkbookEditor) varWorkbook.createVarViewer();

        viewer.setEnabled(true);
        viewer.setFirstRowAsHeader(true);

        final JComponent jc = viewer.getEditorComponent();
        jc.setPreferredSize(new Dimension(400, 300));

        addComponent(jc);
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        inputMap.add("sequence", varSequence);
        inputMap.add("rois", varRois);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("workbook", varWorkbook);
    }

    @Override
    public void clean()
    {
        // set empty workbook to release resources
        varWorkbook.setValue(Workbooks.createEmptyWorkbook(WorkbookFormat.XLSX));
    }

    @Override
    protected void execute()
    {
        // set empty workbook by default (also release resources from previous run)
        varWorkbook.setValue(Workbooks.createEmptyWorkbook(WorkbookFormat.XLSX));

        // interactive mode (not in protocol)
        if (!isHeadLess())
        {
            final Sequence seq = getActiveSequence();

            // set variables
            if (seq != null)
            {
                varSequence.setValue(seq);
                varRois.setValue(seq.getROIs().toArray(new ROI[0]));
            }
            else
            {
                // inform user
                MessageDialog.showDialog("You need to open an image containing ROI.",
                        MessageDialog.INFORMATION_MESSAGE);
                return;
            }
        }

        final Sequence sequence = varSequence.getValue();

        if (sequence != null)
        {
            final ROI[] rois = varRois.getValue();
            final List<ROI> validRois = new ArrayList<ROI>();

            // we can only compute path intensity on path ROI or 2D ROI
            for (ROI roi : rois)
            {
                if (roi instanceof ROI2D)
                    validRois.add(roi);
                else if ((roi instanceof ROI3DPoint) || (roi instanceof ROI3DLine) || (roi instanceof ROI3DPolyLine))
                    validRois.add(roi);
            }

            if (validRois.size() == 0)
            {
                // inform user
                if (!isHeadLess())
                    MessageDialog.showDialog(
                            "The selected Sequence doesn't contain any ROI where we can compute intensity profil.",
                            MessageDialog.INFORMATION_MESSAGE);
                return;
            }

            // interactive mode
            if (!isHeadLess())
                getUI().setProgressBarMessage("Computing...");

            try
            {
                // set result
                varWorkbook.setValue(getPathIntensityProfil(sequence, validRois, getUI()));
            }
            finally
            {
                // interactive mode
                if (!isHeadLess())
                {
                    getUI().setProgressBarMessage("Done");
                    getUI().setProgressBarValue(0);
                }
            }
        }
    }

    /**
     * Computes intensity profil along path (contour for area ROI) for the given list of ROI.<br>
     * 
     * @return result in a XLSX workbook where we have one sheet per ROI.
     */
    private static XSSFWorkbook getPathIntensityProfil(Sequence sequence, List<ROI> rois, EzGUI ui)
    {
        final XSSFWorkbook result = (XSSFWorkbook) Workbooks.createEmptyWorkbook(WorkbookFormat.XLSX);

        // nothing to do
        if (rois.isEmpty())
            return result;

        final int sizeT = sequence.getSizeT();
        final int sizeZ = sequence.getSizeZ();
        final int sizeC = sequence.getSizeC();

        final double progressRatio = rois.size() * sizeT * sizeZ * sizeC;

        int roiIndex = 0;
        // can process now
        for (ROI roi : rois)
        {
            // create a sheet for each ROI
            final IcySpreadSheet sh = new IcySpreadSheet(
                    result.createSheet(roi.getName() + String.format(" - S%04d", Integer.valueOf(roiIndex))));

            // write columns headers
            int indCol = 0;
            sh.setValue(0, indCol++, "Point #");
            sh.setValue(0, indCol++, "X");
            sh.setValue(0, indCol++, "Y");
            sh.setValue(0, indCol++, "Z");
            sh.setValue(0, indCol++, "T");
            for (int c = 0; c < sizeC; c++)
                sh.setValue(0, indCol++, sequence.getChannelName(c));

            // ROI bounds
            final Rectangle5D roiBounds = roi.getBounds5D();
            // start
            int row = 1;

            for (int t = 0; t < sizeT; t++)
            {
                // ROI does not contain current T position ? --> next
                if ((roiBounds.getMinT() > t) || (roiBounds.getMaxT() < t))
                    continue;

                for (int z = 0; z < sizeZ; z++)
                {
                    // ROI does not contain current Z position ? --> next
                    if ((roiBounds.getMinZ() > z) || (roiBounds.getMaxZ() < z))
                        continue;

                    // interrupt processing if wanted
                    if (Thread.currentThread().isInterrupted())
                        return result;

                    // point index
                    int ptIndex = 0;

                    for (int c = 0, zStartRow = row; c < sizeC; c++)
                    {
                        // ROI does not contain current C position ? --> next
                        if ((roiBounds.getMinC() > c) || (roiBounds.getMaxC() < c))
                            continue;

                        // display progress
                        if (ui != null)
                        {
                            // compute progress bar avoid integer type loss
                            double progress = roiIndex;
                            progress *= sizeT * sizeZ * sizeC;
                            progress += t * sizeZ * sizeC;
                            progress += z * sizeC;
                            progress += c;
                            progress /= progressRatio;

                            ui.setProgressBarValue(progress);
                        }

                        // rewind to original Z start row when multi channel
                        if (c > 0)
                            row = zStartRow;

                        final List pts;

                        // 3D path ROI ?
                        if ((roi instanceof ROI3DPoint) || (roi instanceof ROI3DLine) || (roi instanceof ROI3DPolyLine))
                            pts = ((ROI3DShape) roi).getPoints();
                        // 2D path ROI ?
                        else if (roi instanceof ROI2DShape)
                            pts = ((ROI2DShape) roi).getPoints();
                        else
                        {
                            // initialize
                            pts = new ArrayList();

                            // only support 2D ROI so we can use the contour of the mask
                            if (roi instanceof ROI2D)
                            {
                                /// iterate over all component (separate object if any)
                                for (BooleanMask2D mask : ((ROI2D) roi).getBooleanMask(true).getComponents())
                                    // add points in contour order
                                    pts.addAll(mask.getConnectedContourPoints());
                            }
                        }

                        // get the first obj
                        final Object firstObj = pts.get(0);

                        // control points ?
                        if ((firstObj instanceof Point2D) || (firstObj instanceof Point3D))
                        {
                            Point3D startPt = getPoint3D(firstObj, z);

                            // specific case where we have only a single point
                            if (pts.size() == 1)
                                writeRow(sh, row++, ptIndex++, t, c, sequence, startPt);
                            else
                            {
                                // otherwise we use a special way to interpolate through 2 controls points
                                for (int i = 1; i < pts.size(); i++)
                                {
                                    final Point3D endPt = getPoint3D(pts.get(i), z);
                                    // get line iterator
                                    final Line3DIterator lineIt = new Line3DIterator(new Line3D(startPt, endPt), 1d,
                                            false);

                                    // iterate over line points
                                    while (lineIt.hasNext())
                                        writeRow(sh, row++, ptIndex++, t, c, sequence, lineIt.next());

                                    // done --> go next control point
                                    startPt = endPt;
                                }
                            }
                        }
                        // contour points ?
                        else if (firstObj instanceof Point)
                        {
                            for (int i = 0; i < pts.size(); i++)
                                writeRow(sh, row++, ptIndex++, t, c, sequence, getPoint3D(pts.get(i), z));
                        }
                    }
                }
            }

            roiIndex++;
        }

        return result;
    }

    /**
     * Computes intensity profil along path (contour for area ROI) for the given list of ROI.<br>
     * 
     * @return result in a XLSX workbook where we have one sheet per ROI.
     */
    public static XSSFWorkbook getPathIntensityProfil(Sequence sequence, List<ROI> rois)
    {
        return getPathIntensityProfil(sequence, rois, null);
    }

    private static Point3D getPoint3D(Object obj, int curZ)
    {
        final Point3D result;

        if (obj instanceof Point)
        {
            final Point pt2d = (Point) obj;
            result = new Point3D.Double(pt2d.getX(), pt2d.getY(), curZ);
        }
        else if (obj instanceof Point2D)
        {
            final Point2D pt2d = (Point2D) obj;
            result = new Point3D.Double(pt2d.getX(), pt2d.getY(), curZ);
        }
        else if (obj instanceof Point3D)
            result = (Point3D) obj;
        else
            result = null;

        return result;
    }

    private static void writeRow(IcySpreadSheet sh, int row, int ind, int t, int c, Sequence sequence, Point3D pt)
    {
        sh.setValue(row, 0, Integer.valueOf(ind));
        sh.setValue(row, 1, Double.valueOf(MathUtil.round(pt.getX(), 2)));
        sh.setValue(row, 2, Double.valueOf(MathUtil.round(pt.getY(), 2)));
        sh.setValue(row, 3, Double.valueOf(MathUtil.round(pt.getZ(), 2)));
        sh.setValue(row, 4, Integer.valueOf(t));
        sh.setValue(row, 5 + c, Double.valueOf(MathUtil
                .roundSignificant(sequence.getDataInterpolated(t, pt.getZ(), c, pt.getY(), pt.getX()), 5, true)));
    }
}
