package plugins.adufour.roi;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.Future;

import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComponent;
import javax.swing.JSeparator;
import javax.swing.SwingConstants;

import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.util.WorkbookUtil;

import icy.file.FileUtil;
import icy.gui.main.GlobalSequenceListener;
import icy.main.Icy;
import icy.preferences.XMLPreferences;
import icy.roi.ROI;
import icy.roi.ROI2D;
import icy.roi.ROI3D;
import icy.roi.ROI5D;
import icy.roi.ROIDescriptor;
import icy.roi.ROIEvent;
import icy.roi.ROIEvent.ROIEventType;
import icy.roi.ROIListener;
import icy.sequence.Sequence;
import icy.sequence.SequenceDataIterator;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceEvent.SequenceEventSourceType;
import icy.sequence.SequenceListener;
import icy.system.SystemUtil;
import icy.system.thread.Processor;
import icy.system.thread.ThreadUtil;
import icy.type.point.Point3D;
import icy.type.point.Point5D;
import icy.type.rectangle.Rectangle5D;
import icy.util.StringUtil;
import plugins.adufour.blocks.tools.roi.ROIBlock;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzDialog;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.roi.ROIConvexHullDescriptor.ROIConvexity;
import plugins.adufour.roi.ROIEllipsoidFittingDescriptor.ROIElongation;
import plugins.adufour.roi.ROIEllipsoidFittingDescriptor.ROIFlatness3D;
import plugins.adufour.roi.ROIEllipsoidFittingDescriptor.ROIPitchAngle;
import plugins.adufour.roi.ROIEllipsoidFittingDescriptor.ROIRollAngle;
import plugins.adufour.roi.ROIEllipsoidFittingDescriptor.ROIYawAngle;
import plugins.adufour.roi.ROIFeretDiameterDescriptor.ROIFeretDiameter;
import plugins.adufour.roi.ROIRoundnessDescriptor.ROIRoundness;
import plugins.adufour.roi.ROISphericityDescriptor.ROISphericity;
import plugins.adufour.vars.gui.VarEditor;
import plugins.adufour.vars.gui.swing.SwingVarEditor;
import plugins.adufour.vars.gui.swing.WorkbookEditor;
import plugins.adufour.vars.lang.Var;
import plugins.adufour.vars.lang.VarLong;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.lang.VarWorkbook;
import plugins.adufour.vars.util.VarListener;
import plugins.adufour.vars.util.VarReferencingPolicy;
import plugins.adufour.workbooks.IcySpreadSheet;
import plugins.adufour.workbooks.Workbooks;
import plugins.kernel.roi.descriptor.measure.ROIBasicMeasureDescriptorsPlugin;
import plugins.kernel.roi.descriptor.measure.ROIContourDescriptor;
import plugins.kernel.roi.descriptor.measure.ROIInteriorDescriptor;
import plugins.kernel.roi.descriptor.measure.ROIMassCenterDescriptorsPlugin;

/**
 * Basic measures on regions of interest
 * 
 * @author Alexandre Dufour
 */
public class ROIMeasures extends EzPlug implements ROIBlock, GlobalSequenceListener, SequenceListener, ROIListener
{
    public enum Measures
    {
        FULLPATH("Full path"), FOLDER("Parent folder"), DATASET("Dataset"), NAME("ROI"), COLOR("Color"), X("X"), Y("Y"),
        Z("Z"), T("T"), C("C"), X_GLOBAL("X (global)"), Y_GLOBAL("Y (global)"), Z_GLOBAL("Z (global)"),
        BOX_WIDTH("Box width"), BOX_HEIGHT("Box height"), BOX_DEPTH("Box depth"), CONTOUR("Contour"),
        INTERIOR("Interior"), SPHERICITY("Sphericity (%)"), ROUNDNESS("Roudness (%)"), CONVEXITY("Convexity (%)"),
        MAX_FERET("Max. Feret diam."), ELLIPSE_A("Ellipse (a)"), ELLIPSE_B("Ellipse (b)"), ELLIPSE_C("Ellipse (c)"),
        YAW("Yaw angle"), PITCH("Pitch angle"), ROLL("Roll angle"), ELONGATION("Elongation ratio"),
        FLATNESS3D("Flatness ratio (3D)"), INTENSITY_MIN("Min. "), INTENSITY_AVG("Avg. "), INTENSITY_MAX("Max. "),
        INTENSITY_SUM("Sum "), INTENSITY_STD("Std. dev. "), INTENSITY_TEXTURE_ASM("Angular second moment "),
        INTENSITY_TEXTURE_CONT("Contrast "), INTENSITY_TEXTURE_ENT("Entropy "), INTENSITY_TEXTURE_HOMO("Homogeneity "),
        PERIMETER("Perimeter (\u00B5m) "), AREA("Area (\u00B5m2) "), SURFACE_AREA("Surface area (\u00B5m2) "),
        VOLUME("Volume (\u00B5m3) "), BOX_X("Box X"), BOX_Y("Box Y"), BOX_Z("Box Z");

        Measures(String name)
        {
            this.name = name;
        }

        final String name;

        boolean selected = true;

        public void toggleSelection()
        {
            selected = !selected;
        }

        public void setSelected(boolean value)
        {
            selected = value;
        }

        @Override
        public String toString()
        {
            return name;
        }

        public boolean isSelected()
        {
            return selected;
        }

        public int getColumnIndex()
        {
            if (!isSelected())
                return -1;

            Measures[] allMeasures = values();
            int deselectedMeasures = 0;
            for (int i = 0; i < allMeasures.length; i++)
            {
                if (allMeasures[i] == this)
                    return i - deselectedMeasures;
                if (!allMeasures[i].isSelected())
                    deselectedMeasures++;
            }

            // should never happen
            return -1;
        }
    }

    Processor cpus = new Processor(65536, SystemUtil.getNumberOfCPUs());
    final VarMeasureSelector measureSelector;
    final VarROIArray rois = new VarROIArray("Regions of interest");
    final VarWorkbook book = new VarWorkbook("Workbook", (Workbook) null);
    final VarSequence sequence = new VarSequence("Sequence", null);

    boolean measureSelectionChanged = false;

    public ROIMeasures()
    {
        // here isHeadLess always return true as UI is not yet created
        // --> better to use Icy headless check
        if (!Icy.getMainInterface().isHeadLess())
            measureSelector = new VarMeasureSelector(new MeasureSelector());
        else
            measureSelector = new VarMeasureSelector(null);

        // use XML plugin preferences to initialize variable value
        final XMLPreferences prefs = getPreferences("measures");

        long value = 0;
        int i = 0;
        for (Measures measure : Measures.values())
        {
            measure.selected = prefs.node(measure.name()).getBoolean("selected", true);
            if ((i < 64) && measure.selected)
                value |= 1L << i;
            i++;
        }

        // so we correctly restore variable value from XML preferences
        // (we need to use the variable value to correctly load / save selected features with protocol)
        measureSelector.setValue(Long.valueOf(value));
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        measureSelector.setReferencingPolicy(VarReferencingPolicy.NONE);
        inputMap.add("measures", measureSelector);
        inputMap.add("Regions of interest", rois);
        inputMap.add("Sequence", sequence);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("Workbook", book);
    }

    @Override
    protected void initialize()
    {
        getUI().setActionPanelVisible(false);

        addComponent((JComponent) measureSelector.createVarEditor(true).getEditorComponent());

        WorkbookEditor viewer = new WorkbookEditor(book);
        viewer.setReadOnly(true);
        viewer.setEnabled(true);
        viewer.setFirstRowAsHeader(true);
        viewer.setOpenButtonVisible(false);
        JComponent jc = viewer.getEditorComponent();
        jc.setPreferredSize(new Dimension(400, 300));
        addComponent(jc);

        if (!isHeadLess())
            Icy.getMainInterface().addGlobalSequenceListener(this);

        getUI().clickRun();
    }

    @Override
    public void execute()
    {
        Workbook wb = book.getValue();

        // Create the workbook anew if:
        // - it did not exist
        // - we are in block mode (forget previous results)
        if (wb == null || isHeadLess())
        {
            book.setValue(wb = Workbooks.createEmptyWorkbook());
        }

        if (!isHeadLess())
            for (Sequence attachedSequence : getSequences())
            {
                attachedSequence.addListener(this);
                for (ROI roi : attachedSequence.getROIs())
                    roi.addListener(this);
            }

        updateStatistics();
    }

    private String getDataSetName(Sequence sequenceOfInterest)
    {
        String dataSetName = "ROI Statistics";

        if (sequenceOfInterest == null)
        {
            // try retrieving the sequence attached to the first ROI
            List<Sequence> sequences = rois.getValue()[0].getSequences();
            if (sequences.size() > 0)
                sequenceOfInterest = sequences.get(0);
        }

        if (sequenceOfInterest == null)
        {
            // no hope...
            dataSetName = "--";
        }
        else
        {
            // replace the sheet name by the file or sequence name
            dataSetName = FileUtil.getFileName(sequenceOfInterest.getFilename());
            if (StringUtil.isEmpty(dataSetName))
                dataSetName = sequenceOfInterest.getName();
        }

        // make the name "safe"
        return dataSetName;
    }

    String getSheetName(Sequence sequenceOfInterest)
    {
        String sheetName = "ROI Statistics";
        if (sequenceOfInterest != null && !isHeadLess())
            sheetName = getDataSetName(sequenceOfInterest);
        return WorkbookUtil.createSafeSheetName(sheetName);
    }

    private void updateStatistics()
    {
        if (isHeadLess())
        {
            updateStatistics(sequence.getValue());
        }
        else
            for (Sequence sequenceOfInterest : getSequences())
            {
                updateStatistics(sequenceOfInterest);
            }
    }

    private void updateStatistics(final Sequence sequenceOfInterest)
    {
        final Workbook wb = book.getValue();

        String sheetName = getSheetName(sequenceOfInterest);

        final Sheet sheet;
        Row header = null;

        if (wb.getSheet(sheetName) != null)
        {
            sheet = wb.getSheet(sheetName);
        }
        else
        {
            // create the sheet
            sheet = wb.createSheet(sheetName);
            measureSelectionChanged = true;
        }

        if (measureSelectionChanged)
        {
            measureSelectionChanged = false;
            header = sheet.createRow(0);

            // create the header row
            Measures[] measures = Measures.values();
            final int sizeC = (sequenceOfInterest != null) ? sequenceOfInterest.getSizeC() : 1;

            int nbIntensityStats = 0;
            for (Measures measure : measures)
                if (measure.name().startsWith("INTENSITY") && measure.isSelected())
                    nbIntensityStats++;

            int col = 0;
            for (Measures measure : measures)
            {
                if (!measure.isSelected())
                    continue;

                String name = measure.toString();

                if (measure.name().startsWith("INTENSITY") && sequenceOfInterest != null)
                {
                    for (int c = 0; c < sizeC; c++)
                    {
                        header.getCell(col + (nbIntensityStats * c))
                                .setCellValue(name + sequenceOfInterest.getChannelName(c));
                    }
                    col++;
                }
                // measure after intensity measures ? adjust column offset
                else if (measure.ordinal() >= Measures.INTENSITY_MIN.ordinal())
                {
                    header.getCell(col + (nbIntensityStats * (sizeC - 1))).setCellValue(name);
                    col++;
                }
                else
                {
                    header.getCell(col).setCellValue(name);
                    col++;
                }
            }
        }

        final List<ROI> rois2Update = isHeadLess() ? Arrays.asList(this.rois.getValue())
                : sequenceOfInterest.getROIs(true);

        Runnable fullUpdate = new Runnable()
        {
            @Override
            public void run()
            {
                try
                {
                    IcySpreadSheet icySheet = new IcySpreadSheet(sheet);

                    int rowID = 1;
                    icySheet.removeRows(rowID);

                    List<Future<List<Object>>> results = new ArrayList<Future<List<Object>>>(rois2Update.size());

                    for (ROI roi : rois2Update)
                    {
                        if (cpus.isShutdown() || cpus.isTerminating())
                            return;
                        results.add(cpus.submit(createUpdater(sequenceOfInterest, roi)));
                    }

                    for (Future<List<Object>> result : results)
                        updateWorkbook(wb, icySheet, rowID++, result.get(), false);
                }
                catch (InterruptedException e)
                {
                    Thread.currentThread().interrupt();
                    return;
                }
                catch (Exception e)
                {
                    return;
                }

                book.valueChanged(book, null, book.getValue());
            }
        };

        if (isHeadLess())
        {
            fullUpdate.run();
        }
        else
        {
            ThreadUtil.bgRun(fullUpdate);
        }
    }

    private void updateStatistics(ROI roi)
    {
        Workbook wb = book.getValue();

        for (Sequence sequenceOfInterest : roi.getSequences())
            if (Icy.getMainInterface().isOpened(sequenceOfInterest))
                try
                {
                    IcySpreadSheet sheet = Workbooks.getSheet(wb, getSheetName(sequenceOfInterest));

                    int rowID = sequenceOfInterest.getROIs(true).indexOf(roi) + 1;
                    {
                        List<Object> measures = createUpdater(sequenceOfInterest, roi).call();
                        updateWorkbook(wb, sheet, rowID, measures, true);
                    }
                }
                catch (Exception e)
                {
                    throw new RuntimeException(e);
                }
    }

    private void updateWorkbook(Workbook wb, IcySpreadSheet sheet, int rowID, List<Object> measures,
            boolean updateInterface)
    {
        for (int colID = 0; colID < measures.size(); colID++)
        {
            Object value = measures.get(colID);

            if (value instanceof Color)
            {
                sheet.setFillColor(rowID, colID, (Color) value);
            }
            else
                sheet.setValue(rowID, colID, value);
        }

        if (updateInterface)
            book.valueChanged(book, null, book.getValue());
    }

    private Callable<List<Object>> createUpdater(final Sequence sequenceOfInterest, final ROI roi2Update)
    {
        return new Callable<List<Object>>()
        {
            @Override
            public List<Object> call()
            {
                final List<Object> measures = new ArrayList<Object>();
                ROI roi = roi2Update;

                String path = "";
                if (sequenceOfInterest != null && sequenceOfInterest.getFilename() != null)
                    path = sequenceOfInterest.getFilename();

                if (Measures.FULLPATH.isSelected())
                {
                    if (StringUtil.isEmpty(path))
                        measures.add("---");
                    else
                        measures.add(FileUtil.getDirectory(path, false));
                }
                if (Measures.FOLDER.isSelected())
                {
                    if (StringUtil.isEmpty(path))
                        measures.add("---");
                    else
                        measures.add(FileUtil.getFileName(FileUtil.getDirectory(path, false)));
                }
                if (Measures.DATASET.isSelected())
                {
                    measures.add(getDataSetName(sequenceOfInterest));
                }
                if (Measures.NAME.isSelected())
                {
                    measures.add(roi.getName());
                }
                if (Measures.COLOR.isSelected())
                {
                    measures.add(roi.getColor());
                }

                // POSITION (MASS CENTER)

                Point5D center = ROIMassCenterDescriptorsPlugin.computeMassCenter(roi);
                Point3D globalCenter = center.toPoint3D();
                if (sequenceOfInterest != null)
                {
                    globalCenter.translate(sequenceOfInterest.getPositionX(), sequenceOfInterest.getPositionY(),
                            sequenceOfInterest.getPositionZ());
                }

                if (Measures.X.isSelected())
                {
                    measures.add(center.getX());
                }
                if (Measures.Y.isSelected())
                {
                    measures.add(center.getY());
                }
                if (Measures.Z.isSelected())
                {
                    measures.add(roi instanceof ROI2D && center.getZ() == -1 ? "ALL" : center.getZ());
                }
                if (Measures.T.isSelected())
                {
                    measures.add(center.getT() == -1 ? "ALL" : center.getT());
                }
                if (Measures.C.isSelected())
                {
                    measures.add(center.getC() == -1 ? "ALL" : center.getC());
                }
                if (Measures.X_GLOBAL.isSelected())
                {
                    measures.add(globalCenter.getX());
                }
                if (Measures.Y_GLOBAL.isSelected())
                {
                    measures.add(globalCenter.getY());
                }
                if (Measures.Z_GLOBAL.isSelected())
                {
                    measures.add(roi instanceof ROI2D && center.getZ() == -1 ? "ALL" : globalCenter.getZ());
                }
                // BOUNDS

                if (Measures.BOX_WIDTH.isSelected() || Measures.BOX_HEIGHT.isSelected()
                        || Measures.BOX_DEPTH.isSelected())
                {
                    Rectangle5D bounds5 = roi.getBounds5D();

                    if (Measures.BOX_WIDTH.isSelected())
                    {
                        measures.add(bounds5.getSizeX());
                    }
                    if (Measures.BOX_HEIGHT.isSelected())
                    {
                        measures.add(bounds5.getSizeY());
                    }
                    if (Measures.BOX_DEPTH.isSelected())
                    {
                        measures.add(roi instanceof ROI3D ? bounds5.getSizeZ() : "N/A");
                    }
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }

                // CONTOUR & INTERIOR

                if (Measures.CONTOUR.isSelected())
                {
                    measures.add(ROIContourDescriptor.computeContour(roi));
                }
                if (Measures.INTERIOR.isSelected())
                {
                    measures.add(ROIInteriorDescriptor.computeInterior(roi));
                }

                if (Thread.currentThread().isInterrupted())
                    return measures;

                // SHAPE MEASURES

                if (Measures.SPHERICITY.isSelected())
                {
                    measures.add(ROISphericity.computeSphericity(roi));
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }
                if (Measures.ROUNDNESS.isSelected())
                {
                    measures.add((roi.getNumberOfPoints() == 1) ? 100 : ROIRoundness.computeRoundness(roi));
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }
                if (Measures.CONVEXITY.isSelected())
                {
                    measures.add(ROIConvexity.computeConvexity(roi));
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }
                if (Measures.MAX_FERET.isSelected())
                {
                    measures.add(ROIFeretDiameter.computeFeretDiameter(roi, sequenceOfInterest));
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }

                // // for following measures we want a mask ROI
                // if (roi instanceof ROI2D && !(roi instanceof ROI2DArea))
                // {
                // ROI2D r2 = (ROI2D) roi;
                // ROI2DArea area = new ROI2DArea(r2.getBooleanMask(false));
                // area.setZ(r2.getZ());
                // area.setT(r2.getT());
                // area.setName(roi.getName());
                // area.setColor(roi.getColor());
                // roi = area;
                // }
                // else if (roi instanceof ROI3D && !(roi instanceof ROI3DArea))
                // {
                // ROI3D r3 = (ROI3D) roi;
                // ROI3DArea area = new ROI3DArea(r3.getBooleanMask(false));
                // area.setT(r3.getT());
                // area.setName(roi.getName());
                // area.setColor(roi.getColor());
                // roi = area;
                // }
                //
                // if (Thread.currentThread().isInterrupted())
                // return measures;

                // ELLIPSE FITTING

                if (Measures.ELLIPSE_A.isSelected() || Measures.ELLIPSE_B.isSelected()
                        || Measures.ELLIPSE_C.isSelected() || Measures.YAW.isSelected() || Measures.PITCH.isSelected()
                        || Measures.ROLL.isSelected() || Measures.ELONGATION.isSelected()
                        || Measures.FLATNESS3D.isSelected())
                {
                    double[] ellipse = ROIEllipsoidFittingDescriptor.computeOrientation(roi, sequenceOfInterest);

                    if (Measures.ELLIPSE_A.isSelected())
                    {
                        measures.add(ellipse != null ? ellipse[0] : "N/A");
                    }
                    if (Measures.ELLIPSE_B.isSelected())
                    {
                        measures.add(ellipse != null ? ellipse[1] : "N/A");
                    }
                    if (Measures.ELLIPSE_C.isSelected())
                    {
                        measures.add(roi instanceof ROI3D && ellipse != null ? ellipse[2] : "N/A");
                    }
                    if (Measures.YAW.isSelected())
                    {
                        measures.add(ellipse != null ? ROIYawAngle.computeYawAngle(ellipse) : "N/A");
                    }
                    if (Measures.PITCH.isSelected())
                    {
                        measures.add(ellipse != null ? ROIPitchAngle.computePitchAngle(ellipse) : "N/A");
                    }
                    if (Measures.ROLL.isSelected())
                    {
                        measures.add(ellipse != null ? ROIRollAngle.computeRollAngle(ellipse) : "N/A");
                    }
                    if (Measures.ELONGATION.isSelected())
                    {
                        measures.add(
                                ellipse != null && ellipse[1] != 0 ? ROIElongation.computeElongation(ellipse) : "N/A");
                    }
                    if (Measures.FLATNESS3D.isSelected())
                    {
                        measures.add(roi instanceof ROI3D && ellipse != null && ellipse[2] != 0
                                ? ROIFlatness3D.computeFlatness3D(ellipse)
                                : "N/A");
                    }
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }

                // INTENSITY STATISTICS

                // how many measures are selected?
                int nbIntensityStats = 0, nbTextureStats = 0;
                for (Measures measure : Measures.values())
                {
                    if (measure.name().startsWith("INTENSITY") && measure.isSelected())
                    {
                        nbIntensityStats++;

                        if (measure.name().contains("TEXTURE"))
                        {
                            nbTextureStats++;
                        }
                    }
                }

                if (nbIntensityStats > 0 && sequenceOfInterest != null && !(roi instanceof ROI5D))
                {
                    for (int c = 0; c < sequenceOfInterest.getSizeC(); c++)
                    {
                        if (Thread.currentThread().isInterrupted())
                            return measures;

                        // we want intensity measures for intersecting points as well
                        SequenceDataIterator iterator = new SequenceDataIterator(sequenceOfInterest, roi, true, -1, -1,
                                c);

                        // minimum / maximum / sum
                        double min = Double.MAX_VALUE, max = 0, sum = 0, cpt = 0;
                        while (!iterator.done())
                        {
                            double val = iterator.get();
                            if (val > max)
                                max = val;
                            if (val < min)
                                min = val;
                            sum += val;
                            cpt++;
                            iterator.next();
                        }
                        double avg = sum / cpt;

                        double std = 0;
                        if (Measures.INTENSITY_STD.isSelected())
                        {
                            iterator.reset();
                            while (!iterator.done())
                            {
                                double dev = iterator.get() - avg;
                                std += dev * dev;
                                iterator.next();
                            }
                            std = Math.sqrt(std / cpt);
                        }

                        if (Measures.INTENSITY_MIN.isSelected())
                        {
                            measures.add(min);
                        }
                        if (Measures.INTENSITY_AVG.isSelected())
                        {
                            measures.add(avg);
                        }
                        if (Measures.INTENSITY_MAX.isSelected())
                        {
                            measures.add(max);
                        }
                        if (Measures.INTENSITY_SUM.isSelected())
                        {
                            measures.add(sum);
                        }
                        if (Measures.INTENSITY_STD.isSelected())
                        {
                            measures.add(std);
                        }

                        // Texture

                        Map<ROIDescriptor, Object> map = null;

                        try
                        {
                            if (nbTextureStats > 0 && roi instanceof ROI2D)
                            {
                                ROI2D r2 = (ROI2D) roi.getCopy();
                                r2.setC(c);
                                map = new ROIHaralickTextureDescriptor().compute(r2, sequenceOfInterest);
                            }
                        }
                        catch (UnsupportedOperationException u)
                        {

                        }

                        if (Measures.INTENSITY_TEXTURE_ASM.isSelected())
                        {
                            measures.add(
                                    map == null ? "N.A." : map.get(ROIHaralickTextureDescriptor.angularSecondMoment));
                        }
                        if (Measures.INTENSITY_TEXTURE_CONT.isSelected())
                        {
                            measures.add(map == null ? "N.A." : map.get(ROIHaralickTextureDescriptor.contrast));
                        }
                        if (Measures.INTENSITY_TEXTURE_ENT.isSelected())
                        {
                            measures.add(map == null ? "N.A." : map.get(ROIHaralickTextureDescriptor.entropy));
                        }
                        if (Measures.INTENSITY_TEXTURE_HOMO.isSelected())
                        {
                            measures.add(map == null ? "N.A." : map.get(ROIHaralickTextureDescriptor.homogeneity));
                        }
                    }
                }

                if (Thread.currentThread().isInterrupted())
                    return measures;

                // PERIMETER, AREA, SURFACE AREA, VOLUME

                if (Measures.PERIMETER.isSelected())
                {
                    if (sequenceOfInterest != null)
                    {
                        try
                        {
                            // don't use ROI descriptor as they are dumb (unit is not fixed)
                            measures.add(roi.getLength(sequenceOfInterest));
                        }
                        catch (Exception u)
                        {
                            measures.add("N.A.");
                        }
                    }
                }
                if (Measures.AREA.isSelected())
                {
                    if (sequenceOfInterest != null)
                    {
                        try
                        {
                            // don't use ROI descriptor as they are dumb (unit is not fixed)
                            final double mul = ROIBasicMeasureDescriptorsPlugin.getMultiplierFactor(sequenceOfInterest,
                                    roi, 2);

                            // 0 means the operation is not supported for this ROI
                            if (mul == 0d)
                                throw new UnsupportedOperationException();

                            // convert number of point in area using internal sequence pixel size (approximation)
                            measures.add(sequenceOfInterest.calculateSize(roi.getNumberOfPoints() * mul, 2, 2));
                        }
                        catch (Exception u)
                        {
                            measures.add("N.A.");
                        }
                    }
                }

                if (Measures.SURFACE_AREA.isSelected())
                {
                    if (sequenceOfInterest != null)
                    {
                        try
                        {
                            // don't use ROI descriptor as they are dumb (unit is not fixed)
                            measures.add(((ROI3D) roi).getSurfaceArea(sequenceOfInterest));
                        }
                        catch (Exception u)
                        {
                            measures.add("N.A.");
                        }
                    }
                }
                if (Measures.VOLUME.isSelected())
                {
                    if (sequenceOfInterest != null)
                    {
                        try
                        {
                            // don't use ROI descriptor as they are dumb (unit is not fixed)
                            final double mul = ROIBasicMeasureDescriptorsPlugin.getMultiplierFactor(sequenceOfInterest,
                                    roi, 3);

                            // 0 means the operation is not supported for this ROI
                            if (mul == 0d)
                                throw new UnsupportedOperationException();

                            // convert number of point in volume using internal sequence pixel size (approximation)
                            measures.add(sequenceOfInterest.calculateSize(roi.getNumberOfPoints() * mul, 3, 3));
                        }
                        catch (Exception u)
                        {
                            measures.add("N.A.");
                        }
                    }
                }

                // BOUNDS - EXT

                if (Measures.BOX_X.isSelected() || Measures.BOX_Y.isSelected() || Measures.BOX_Z.isSelected())
                {
                    Rectangle5D bounds5 = roi.getBounds5D();

                    if (Measures.BOX_X.isSelected())
                    {
                        measures.add(bounds5.getX());
                    }
                    if (Measures.BOX_Y.isSelected())
                    {
                        measures.add(bounds5.getY());
                    }
                    if (Measures.BOX_Z.isSelected())
                    {
                        measures.add(bounds5.getZ());
                    }
                    if (Thread.currentThread().isInterrupted())
                        return measures;
                }

                return measures;
            }
        };
    }

    /**
     * @param roi
     * @return An array containing the [min, max] diameters of the best fitting
     *         ellipse for the specified ROI
     * @deprecated This method should not be used directly and will be removed in
     *             future releases in favor of a {@link ROIDescriptor}
     *             implementation
     */
    @Deprecated
    public static double[] computeEllipseDimensions(ROI roi)
    {
        double[] ellipse = ROIEllipsoidFittingDescriptor.computeOrientation(roi, null);

        return new double[] {ellipse[0], ellipse[1], ellipse[2]};
    }

    /**
     * The sphericity is the normalised ratio between the perimeter and area of a
     * ROI (in 2D), or its volume and surface area (in 3D).
     * 
     * @param roi
     *        the input component
     * @return 100% for a perfect circle (or sphere), and lower otherwise
     * @deprecated Use {@link ROISphericity#computeSphericity(ROI)} instead
     */
    @Deprecated
    public static double computeSphericity(ROI roi)
    {
        return ROISphericity.computeSphericity(roi);
    }

    /**
     * The roundness is approximated here by the ratio between the smallest and
     * largest distance from the mass center to any point on the surface, and
     * expressed as a percentage.
     * 
     * @return 100% for a perfect circle (or sphere in 3D), and lower otherwise
     * @deprecated Use {@link ROIRoundness#computeRoundness(ROI)} instead
     */
    @Deprecated
    public static double computeRoundness(ROI roi)
    {
        return ROIRoundness.computeRoundness(roi);
    }

    /**
     * @param roi
     * @return The maximum Feret diameter (i.e. the longest distance between any 2
     *         points of the ROI)
     * @deprecated Use {@link ROIFeretDiameter#computeFeretDiameter(ROI, Sequence)}
     *             instead
     */
    @Deprecated
    public static double computeMaxFeret(ROI roi)
    {
        return ROIFeretDiameter.computeFeretDiameter(roi, null);
    }

    /**
     * @param roi
     * @return The hull ratio, measured as the ratio between the object volume and
     *         its convex hull (envelope)
     * @deprecated Use {@link ROIConvexity#computeConvexity(ROI)} instead
     */
    @Deprecated
    public static double computeConvexity(ROI roi)
    {
        return ROIConvexity.computeConvexity(roi);
    }

    /**
     * @param roi
     * @return An array containing [contour, area] of the smallest convex envelope
     *         surrounding the object. The 2 values are returned together because
     *         their computation is simultaneous
     * @deprecated use {@link Convexify#createConvexROI(ROI)} instead
     */
    @Deprecated
    public static double[] computeHullDimensions(ROI roi)
    {
        ROI convexHull = Convexify.createConvexROI(roi);

        return new double[] {convexHull.getNumberOfContourPoints(), convexHull.getNumberOfPoints()};
    }

    // PUBLIC STATIC METHODS (useful for script access) //

    /**
     * @param roi
     * @return
     * @deprecated use {@link ROIMassCenterDescriptorsPlugin#computeMassCenter(ROI)}
     *             instead.
     */
    @Deprecated
    public static Point5D getMassCenter(ROI roi)
    {
        return ROIMassCenterDescriptorsPlugin.computeMassCenter(roi);
    }

    /**
     * @param roi
     * @return
     * @deprecated use {@link ROIMassCenterDescriptorsPlugin#computeMassCenter(ROI)}
     *             instead.
     */
    @Deprecated
    public static Point2D getMassCenter(ROI2D roi)
    {
        return getMassCenter((ROI) roi).toPoint2D();
    }

    /**
     * @param roi
     * @return
     * @deprecated use {@link ROIMassCenterDescriptorsPlugin#computeMassCenter(ROI)}
     *             instead.
     */
    @Deprecated
    public static Point3D getMassCenter(ROI3D roi)
    {
        return getMassCenter((ROI) roi).toPoint3D();
    }

    // MAIN LISTENER //

    @Override
    public void sequenceOpened(Sequence openedSequence)
    {
        openedSequence.addListener(this);
        for (ROI roi : openedSequence.getROIs())
            roi.addListener(this);

        // update(null);
        updateStatistics(openedSequence);

        // this is mandatory since sheet creation cannot be detected
        book.valueChanged(book, null, book.getValue());
    }

    @Override
    public void sequenceClosed(Sequence closedSequence)
    {
        closedSequence.removeListener(this);
        for (ROI roi : closedSequence.getROIs())
            roi.removeListener(this);

        String sheetName = getSheetName(closedSequence);

        int index = book.getValue().getSheetIndex(sheetName);

        if (index >= 0)
        {
            book.getValue().removeSheetAt(index);
            // this is mandatory since sheet creation cannot be detected
            book.valueChanged(book, null, book.getValue());
        }
    }

    // SEQUENCE LISTENER //

    @Override
    public void sequenceChanged(SequenceEvent sequenceEvent)
    {
        if (sequenceEvent.getSourceType() == SequenceEventSourceType.SEQUENCE_DATA)
        {
            updateStatistics(sequenceEvent.getSequence());
        }
        else if (sequenceEvent.getSourceType() == SequenceEventSourceType.SEQUENCE_ROI)
        {
            ROI roi = (ROI) sequenceEvent.getSource();

            switch (sequenceEvent.getType())
            {
                case ADDED:

                    if (roi == null)
                    {
                        // multiple ROI were added
                        updateStatistics();
                    }
                    else
                    {
                        for (Sequence sequenceOfInterest : roi.getSequences())
                            updateStatistics(sequenceOfInterest);
                        // remove it first to avoid duplicates
                        roi.removeListener(this);
                        roi.addListener(this);
                    }
                    break;

                case REMOVED:

                    if (roi == null)
                    {
                        // multiple ROIs have been removed
                        System.err.println("[ROI Statistics] Warning: potential memory leak");
                    }
                    else
                    {
                        roi.removeListener(this);
                    }
                    updateStatistics();
                    break;

                case CHANGED: // don't do anything, roiChanged() will do this for us
            }
        }
    }

    @Override
    public void clean()
    {
        final MeasureSelector selectorGui = measureSelector.gui;
        // release EzDialog (otherwise it remains referenced by IcyFrame.frames)
        if (selectorGui != null)
            selectorGui.selector.close();

        // remove listeners
        for (final Sequence openedSequence : Icy.getMainInterface().getSequences())
            openedSequence.removeListener(this);

        Icy.getMainInterface().removeGlobalSequenceListener(this);

        cpus.shutdownNow();
    }

    @Override
    public void roiChanged(final ROIEvent event)
    {
        if (event.getType() == ROIEventType.ROI_CHANGED
                || event.getType() == ROIEventType.PROPERTY_CHANGED && (event.getPropertyName().equalsIgnoreCase("name")
                        || event.getPropertyName().equalsIgnoreCase("color")))
        {
            updateStatistics(event.getSource());
            // this is mandatory since sheet modification cannot be detected
            book.valueChanged(book, null, book.getValue());
        }
    }

    private static interface VarMeasureSelectorListener extends VarListener<Long>
    {
        public abstract void triggered(VarMeasureSelector source);
    }

    /**
     * Custom Button SwingVarEditor
     * 
     * @author Stephane
     */
    private static class ButtonVarEditor extends SwingVarEditor<Long>
    {
        private ActionListener listener;

        public ButtonVarEditor(VarMeasureSelector variable)
        {
            super(variable);

            setNameVisible(false);
        }

        @Override
        protected JButton createEditorComponent()
        {
            if (getEditorComponent() != null)
                deactivateListeners();

            listener = new ActionListener()
            {
                @Override
                public void actionPerformed(ActionEvent e)
                {
                    ((VarMeasureSelector) variable).trigger();
                }
            };

            return new JButton(variable.getName());
        }

        @Override
        public JButton getEditorComponent()
        {
            return (JButton) super.getEditorComponent();
        }

        @Override
        public double getComponentVerticalResizeFactor()
        {
            return 0.0;
        }

        @Override
        protected void activateListeners()
        {
            getEditorComponent().addActionListener(listener);
        }

        @Override
        protected void deactivateListeners()
        {
            getEditorComponent().removeActionListener(listener);
        }

        @Override
        protected void updateInterfaceValue()
        {
            // nothing to do (it's just a button with a name)
        }
    }

    /**
     * Custom VarTrigger for measure selector so we store selected parameters (up to 64) instead of number of trigger inside value
     * 
     * @author Stephane Dallongeville
     */
    private static class VarMeasureSelector extends VarLong
    {
        // GUI (only if no headless)
        final MeasureSelector gui;

        /**
         * Creates a new trigger with the given name
         */
        public VarMeasureSelector(MeasureSelector gui)
        {
            // by default we want to select all
            super("Select features...", -1L);

            this.gui = gui;

            if (gui != null)
                addListener(gui);
        }

        @Override
        public VarEditor<Long> createVarEditor()
        {
            return new ButtonVarEditor(this);
        }

        /**
         * Triggers the variable (equivalent to incrementing the value of this variable)
         */
        public void trigger()
        {
            // fire trigger listeners
            for (VarListener<Long> listener : listeners)
                if (listener instanceof VarMeasureSelectorListener)
                    ((VarMeasureSelectorListener) listener).triggered(this);
        }
    }

    private class MeasureSelector implements VarMeasureSelectorListener
    {
        final EzDialog selector;
        final JCheckBox checkBoxes[];

        public MeasureSelector()
        {
            final Measures[] measures = Measures.values();

            selector = new EzDialog("Select measures...");
            selector.setLayout(new BoxLayout(selector.getContentPane(), BoxLayout.Y_AXIS));

            checkBoxes = new JCheckBox[measures.length];

            for (int i = 0; i < measures.length; i++)
            {
                final Measures measure = measures[i];
                String name = measure.toString();

                if (measure.name().startsWith("INTENSITY"))
                {
                    if (measure.name().contains("TEXTURE"))
                    {
                        name += "(texture)";
                    }
                    else
                    {
                        name += "intensity";
                    }
                }

                final JCheckBox check = new JCheckBox(name, measure.isSelected());
                check.addActionListener(new ActionListener()
                {
                    @Override
                    public void actionPerformed(ActionEvent arg0)
                    {
                        measure.setSelected(check.isSelected());
                        measureSelectionChanged = true;
                    }
                });
                checkBoxes[i] = check;

                selector.addComponent(check);

                if (measure == Measures.BOX_DEPTH || measure == Measures.FLATNESS3D)
                {
                    selector.addComponent(new JSeparator(SwingConstants.VERTICAL));
                }
            }
        }

        public void loadParameters(VarMeasureSelector source)
        {
            // load parameter values from source
            if (source != null)
            {
                final Measures[] measures = Measures.values();
                final long value = source.getValue().longValue();

                // can't save more than 64 :-(
                for (int i = 0; i < Math.min(checkBoxes.length, 64); i++)
                {
                    final boolean selected = (value & (1L << i)) != 0L;

                    measures[i].setSelected(selected);
                    if (checkBoxes[i].isSelected() != selected)
                        checkBoxes[i].setSelected(selected);
                }

                // update checkbox state
                ThreadUtil.invokeLater(new Runnable()
                {
                    @Override
                    public void run()
                    {
                        for (int i = 0; i < measures.length; i++)
                        {
                            final boolean selected = measures[i].isSelected();

                            if (checkBoxes[i].isSelected() != selected)
                                checkBoxes[i].setSelected(selected);
                        }
                    }
                });
            }
        }

        public void saveParameters(VarMeasureSelector source)
        {
            // update parameter values to source
            if (source != null)
            {
                final Measures[] measures = Measures.values();
                long value = 0;

                // can't save more than 64 :-(
                for (int i = 0; i < Math.min(checkBoxes.length, 64); i++)
                    if (measures[i].isSelected())
                        value |= 1L << i;

                // update value in trigger
                source.setValue(Long.valueOf(value));
            }
        }

        @Override
        public void triggered(VarMeasureSelector source)
        {
            loadParameters(source);

            selector.pack();
            selector.showDialog(true);

            saveParameters(source);

            if (measureSelectionChanged)
            {
                // save settings
                final XMLPreferences prefs = getPreferences("measures");

                for (Measures measure : Measures.values())
                    prefs.node(measure.name()).putBoolean("selected", measure.isSelected());

                book.setValue(null);

                if (!isHeadLess())
                    execute();
            }
        }

        @Override
        public void valueChanged(Var<Long> source, Long oldValue, Long newValue)
        {
            // set parameters from value (in case we modify value from external
            loadParameters((VarMeasureSelector) source);
        }

        @Override
        public void referenceChanged(Var<Long> source, Var<? extends Long> oldReference,
                Var<? extends Long> newReference)
        {
            //
        }
    }
}
