/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui.pair;

import plugins.adufour.vars.gui.model.VarEditorModel;

/**
 * VarEditorModel of Pair objects. The two values of the pair are simply bounds.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class PairRangeModel implements VarEditorModel<Pair> {
	private final Pair defaultValue_;
	private double firstMinimum_, firstMaximum_;
	private double secondMinimum_, secondMaximum_;
	private final Pair steps_;

	/** Default constructor */
	public PairRangeModel(Pair defaultValue, double min, double max, Pair steps) {
		if (max < min) {
			throw new IllegalArgumentException("max must be higher than min");
		}

		firstMinimum_ = min;
		firstMaximum_ = max;
		secondMinimum_ = min;
		secondMaximum_ = max;
		defaultValue_ = defaultValue;
		steps_ = steps;

		if (!isValid(defaultValue_)) {
			throw new IllegalArgumentException("Default value not valid");
		}
		updateBounds(defaultValue_);
	}

	@Override
	public Pair getDefaultValue() {
		return defaultValue_;
	}

	/** Returns the minimum for the first value of the pair. */
	public double getFirstMinimum() {
		return firstMinimum_;
	}

	/** Returns the maximum for the first value of the pair. */
	public double getFirstMaximum() {
		return firstMaximum_;
	}

	/** Returns the minimum for the second value of the pair. */
	public double getSecondMinimum() {
		return secondMinimum_;
	}

	/** Returns the maximum for the second value of the pair. */
	public double getSecondMaximum() {
		return secondMaximum_;
	}

	/** Set the minimum for the first value of the pair. */
	public void setFirstMinimum(double min) {
		firstMinimum_ = Math.min(min, firstMaximum_);
	}

	/** Set the maximum for the first value of the pair. */
	public void setFirstMaximum(double max) {
		firstMaximum_ = Math.max(max, firstMinimum_);
	}

	/** Set the bounds for the first value of the pair. */
	public void setFirstBounds(double min, double max) {
		setFirstMinimum(min);
		setFirstMaximum(max);
	}

	/** Set the minimum for the second value of the pair. */
	public void setSecondMinimum(double min) {
		secondMinimum_ = Math.min(min, secondMaximum_);
	}

	/** Set the maximum for the second value of the pair. */
	public void setSecondMaximum(double max) {
		secondMaximum_ = Math.max(max, secondMinimum_);
	}

	/** Set the bounds for the second value of the pair. */
	public void setSecondBounds(double min, double max) {
		setSecondMinimum(min);
		setSecondMaximum(max);
	}

	/** Set the minimum for both values of the pair. */
	public void setMinimum(double min) {
		setFirstMinimum(min);
		setSecondMinimum(min);
	}

	/** Set the maximum for both values of the pair. */
	public void setMaximum(double max) {
		setFirstMaximum(max);
		setSecondMaximum(max);
	}

	/** Set the bounds for both values of the pair. */
	public void setBounds(double min, double max) {
		setFirstBounds(min, max);
		setSecondBounds(min, max);
	}

	/** Set the step size for the first value of the pair. */
	public void setFirstStepSize(double step) {
		steps_.setFirst(step);
	}

	/** Returns the step size for the first value of the pair. */
	public double getFirstStepSize() {
		return steps_.getFirst();
	}

	/** Set the step size for the second value of the pair. */
	public void setSecondStepSize(double step) {
		steps_.setSecond(step);
	}

	/** Returns the step size for the second value of the pair. */
	public double getSecondStepSize() {
		return steps_.getSecond();
	}

	/** Set the step size for the both values of the pair. */
	public void setStepSize(double step) {
		setFirstStepSize(step);
		setSecondStepSize(step);
	}

	/**
	 * Update the bounds accordingly to the given value if needed. By default
	 * this function does nothing.
	 */
	public void updateBounds(Pair value) {
	}

	@Override
	public boolean isValid(Pair value) {
		return firstMinimum_ <= value.getFirst()
				&& firstMaximum_ >= value.getFirst()
				&& secondMinimum_ <= value.getSecond()
				&& secondMaximum_ >= value.getSecond();
	}
}
