package plugins.vannary.morphomaths.blocks;

import icy.plugin.abstract_.PluginActionable;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.system.IcyHandledException;
import icy.type.DataType;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzVar;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarListener;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.vannary.morphomaths.MorphOp;

public class DistanceMap extends PluginActionable implements Block
{

    private EzVarSequence inputSequence;
    private EzVarBoolean inputOverwriteResult;
    private EzVarBoolean input3dModality;
    private EzVarInteger inputProcessedZ;

    @Override
    public void declareInput(VarList inputMap)
    {
        inputSequence = new EzVarSequence("Sequence");
        inputOverwriteResult = new EzVarBoolean("Results on input", false);
        input3dModality = new EzVarBoolean("Compute in 3D", false);
        inputProcessedZ = new EzVarInteger("Z value");

        inputProcessedZ.getVariable().setEnabled(false);
        input3dModality.addVarChangeListener(new EzVarListener<Boolean>()
        {

            @Override
            public void variableChanged(EzVar<Boolean> source, Boolean newValue)
            {
                if (newValue)
                {
                    inputProcessedZ.getVariable().setEnabled(false);
                }
                else
                {
                    inputProcessedZ.getVariable().setEnabled(true);
                }
            }
        });

        inputMap.add(inputSequence.name, inputSequence.getVariable());
        inputMap.add(inputOverwriteResult.name, inputOverwriteResult.getVariable());
        inputMap.add(input3dModality.name, input3dModality.getVariable());
        inputMap.add(inputProcessedZ.name, inputProcessedZ.getVariable());
    }

    private EzVarSequence outputSequence;

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputSequence = new EzVarSequence("Dilation Sequence");
        outputMap.add(outputSequence.name, outputSequence.getVariable());
    }

    private Sequence transformedSequence;

    @Override
    public void run()
    {
        Sequence inSequence;
        if (inputSequence == null)
        {
            inSequence = getActiveSequence();
        }
        else
        {
            inSequence = inputSequence.getValue(true);
        }
        if (inSequence == null)
            throw new IcyHandledException("No sequence selected");

        boolean isReusingSequence;
        boolean is3D;
        int processedZ;
        if (inputSequence == null)
        {
            isReusingSequence = false;
            is3D = inSequence.getSizeZ() > 1;
            processedZ = 0;
        }
        else
        {
            isReusingSequence = inputOverwriteResult.getValue();
            is3D = input3dModality.getValue();
            processedZ = inputProcessedZ.getValue();
        }

        if (isReusingSequence)
        {
            transformedSequence = inSequence;
        }
        else
        {
            transformedSequence = SequenceUtil.convertToType(SequenceUtil.toGray(inSequence), DataType.DOUBLE, false);
            transformedSequence.setName(inSequence.getName() + "_Dilation");
        }

        if (is3D)
        {
            MorphOp.distanceMap3D(transformedSequence, 0, true);
        }
        else
        {
            if (processedZ >= 0 && processedZ < transformedSequence.getSizeZ())
            {
                MorphOp.distanceMap2D(transformedSequence, processedZ, 0, true);
            }
            else
            {
                throw new RuntimeException("Invalid z coordinate: " + processedZ + ". Expected [0 - "
                        + transformedSequence.getSizeZ() + ")");
            }
        }

        transformedSequence.updateChannelsBounds();
        if (outputSequence != null)
        {
            outputSequence.setValue(transformedSequence);
        }
        else
        {
            addSequence(transformedSequence);
        }
    }

}
