package plugins.vannary.morphomaths;

import java.awt.Point;
import java.awt.image.DataBufferDouble;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.type.DataType;
import icy.type.collection.array.Array1DUtil;
import icy.type.collection.array.Array2DUtil;
import icy.type.point.Point3D;
import plugins.adufour.connectedcomponents.ConnectedComponent;
import plugins.adufour.connectedcomponents.ConnectedComponents;
import plugins.adufour.connectedcomponents.ConnectedComponents.ExtractionType;

/**
 * Tools for basic morphology operations on 2D and 3D sequences (+t) These
 * functions write directly in the input Sequence. Modifications are done on the
 * first band.
 * 
 * @author Paul Compere, VMY
 * 
 */

public class MorphOp {

	final static int TYPESKEL_MAX = 0;
	final static int TYPESKEL_SMOOTH = 1;
	final static int TYPEWSHED_SEP_BASINS = 0;
	final static int TYPEWSHED_BASINS_MAP = 1;
	final static int TYPEWSHED_WSHEDONLY = 2;

	// ----------------------------------------------------------------------
	// Erosion and dilatation operations
	// ----------------------------------------------------------------------

	/**
	 * Erosion on a grey scale image Affects to the pixel the min on the structuring
	 * element's domain
	 * 
	 * @param seq        Sequence to be eroded
	 * @param z          Z value of the image to erode
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void erodeGreyScale(Sequence seq, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		// Structuring element dimensions
		double eWidth = eltS.length;
		double eHeight = eltS[0].length;

		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();

		// Temporary sequence containing the original data

		for (int t = 0; t < seq.getSizeT(); t++) {
			int offset = 0;
			IcyBufferedImage img = seq.getImage(t, z);

			double[] tabInDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());
			double[] tabOutDouble = Arrays.copyOf(tabInDouble, tabInDouble.length);

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++, offset++) {
					// Max value for a double
					double val = Double.MAX_VALUE;
					for (int yElt = 0; yElt < eHeight; yElt++) {
						for (int xElt = 0; xElt < eWidth; xElt++) {
							if (eltS[xElt][yElt] > 0) {
								int xRel = xElt - xCenterElt;
								int yRel = yElt - yCenterElt;
								if ((xRel + x) >= 0 && (yRel + y) >= 0 && (xRel + x) < width && (yRel + y) < height) {
									int offRel = xRel + x + width * (y + yRel);
									double temp = tabInDouble[offRel];
									if (temp < val) {
										val = temp;
									}
								}
							}
						}
					}
					tabOutDouble[offset] = val;
				}
			}
			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabOutDouble, img.getDataXY(0)));

		}
	}

	/**
	 * Dilatation on a grey scale image Affects to the pixel the max on the
	 * structuring element's domain
	 * 
	 * @param seq        Sequence to be dilated
	 * @param z          Z value of the image to dilate
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void dilateGreyScale(Sequence seq, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		// Structuring element dimensions
		double eWidth = eltS.length;
		double eHeight = eltS[0].length;

		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();

		for (int t = 0; t < seq.getSizeT(); t++) {
			int offset = 0;
			IcyBufferedImage img = seq.getImage(t, z);

			double[] tabInDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());
			double[] tabOutDouble = Arrays.copyOf(tabInDouble, tabInDouble.length);

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++, offset++) {
					// Min value for a double
					double val = 0;
					for (int yElt = 0; yElt < eHeight; yElt++) {
						for (int xElt = 0; xElt < eWidth; xElt++) {
							if (eltS[xElt][yElt] > 0) {
								int xRel = xElt - xCenterElt;
								int yRel = yElt - yCenterElt;
								if ((xRel + x) >= 0 && (yRel + y) >= 0 && (xRel + x) < width && (yRel + y) < height) {
									int offRel = xRel + x + width * (y + yRel);
									// If a pixel is outside the object
									double temp = tabInDouble[offRel];
									if (temp > val) {
										val = temp;
									}
								}
							}
						}
					}
					tabOutDouble[offset] = val;
				}
			}

			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabOutDouble, img.getDataXY(0)));
		}

	}

	/**
	 * Erosion on a grey scale 3D image Affects to the pixel the min on the
	 * structuring element's domain
	 * 
	 * @param seq        Sequence to be eroded
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * @param zCenterElt z of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void erodeGreyScale3D(Sequence seq, double[][][] eltS, int xCenterElt, int yCenterElt,
			int zCenterElt) {
		// Structuring element dimensions
		int eWidth = eltS.length;
		int eHeight = eltS[0].length;
		int eDepth = eltS[0][0].length;

		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();

		int sizeT = seq.getSizeT();

		for (int t = 0; t < sizeT; t++) {
			int depth = seq.getSizeZ(t);

			double[][] dataInDouble = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, 0), seq.isSignedDataType());
			double[][] dataOutDouble = Array2DUtil.doubleArrayToDoubleArray(dataInDouble, 0, null, 0,
					dataInDouble.length);

			for (int z = 0; z < depth; z++) {

				int offset = 0;
				for (int y = 0; y < height; y++) {
					for (int x = 0; x < width; x++, offset++) {

						// max and min values
						double val = Double.MAX_VALUE;
						// For each
						for (int zElt = 0; zElt < eDepth; zElt++) {
							for (int yElt = 0; yElt < eHeight; yElt++) {
								for (int xElt = 0; xElt < eWidth; xElt++) {
									if (eltS[xElt][yElt][zElt] > 0) {
										int xRel = xElt - xCenterElt;
										int yRel = yElt - yCenterElt;
										int zRel = zElt - zCenterElt;
										int xTemp = xRel + x;
										int yTemp = yRel + y;
										int zTemp = zRel + z;

										if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height
												&& (zTemp) >= 0 && (zTemp) < seq.getSizeZ(t)) {
											int offRel = xTemp + width * (yTemp);
											// Make the min
											double temp = dataInDouble[zTemp][offRel];
											if (temp < val) {
												val = temp;
											}
										}
									}
								}
							}
						}
						// Put the min in the image
						dataOutDouble[z][offset] = val;
					}
				}
			}
			seq.beginUpdate();
			for (int z = 0; z < depth; z++) {
				IcyBufferedImage img = seq.getImage(t, z);
				img.setDataXY(0, Array1DUtil.doubleArrayToArray(dataOutDouble[z], img.getDataXY(0)));
			}
			seq.endUpdate();
		}
	}

	/**
	 * Dilatation on a grey scale 3D image Affects to the pixel the max on the
	 * structuring element's domain
	 * 
	 * Performs an erosion of the background
	 * 
	 * @param seq        Sequence to be closed
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * @param zCenterElt z of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void dilateGreyScale3D(Sequence seq, double[][][] eltS, int xCenterElt, int yCenterElt,
			int zCenterElt) {
		// Structuring element dimensions
		double eWidth = eltS.length;
		double eHeight = eltS[0].length;
		double eDepth = eltS[0][0].length;

		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();

		int sizeT = seq.getSizeT();

		for (int t = 0; t < sizeT; t++) {
			int depth = seq.getSizeZ(t);

			double[][] dataInDouble = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, 0), seq.isSignedDataType());
			double[][] dataOutDouble = Array2DUtil.doubleArrayToDoubleArray(dataInDouble, 0, null, 0,
					dataInDouble.length);

			for (int z = 0; z < depth; z++) {

				int offset = 0;
				for (int y = 0; y < height; y++) {
					for (int x = 0; x < width; x++, offset++) {

						// min value
						double val = 0;
						// For each
						for (int zElt = 0; zElt < eDepth; zElt++) {
							for (int yElt = 0; yElt < eHeight; yElt++) {
								for (int xElt = 0; xElt < eWidth; xElt++) {
									if (eltS[xElt][yElt][zElt] > 0) {
										int xRel = xElt - xCenterElt;
										int yRel = yElt - yCenterElt;
										int zRel = zElt - zCenterElt;
										int xTemp = xRel + x;
										int yTemp = yRel + y;
										int zTemp = zRel + z;

										if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height
												&& (zTemp) >= 0 && (zTemp) < seq.getSizeZ(t)) {
											int offRel = xTemp + width * (yTemp);
											// Make the max
											double temp = dataInDouble[zTemp][offRel];
											if (temp > val) {
												val = temp;
											}
										}
									}
								}
							}
						}
						// Put the max in the image
						dataOutDouble[z][offset] = val;
					}
				}
			}

			seq.beginUpdate();
			for (int z = 0; z < depth; z++) {
				IcyBufferedImage img = seq.getImage(t, z);
				img.setDataXY(0, Array1DUtil.doubleArrayToArray(dataOutDouble[z], img.getDataXY(0)));
			}
			seq.endUpdate();
		}
	}

	// ----------------------------------------------------------------------
	// Opening and closing operations
	// ----------------------------------------------------------------------

	/**
	 * Opening on a grey scale 2D image Erodes then dilates by the transposed
	 * structuring element
	 * 
	 * @param seq        Sequence to be opened
	 * @param z          Z value of the image to open
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void openGreyScale(Sequence seq, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		erodeGreyScale(seq, z, eltS, xCenterElt, yCenterElt);
		// Dilate by the transposed
		int width = eltS.length;
		int height = eltS[0].length;
		double[][] eltST = transpose(eltS);
		dilateGreyScale(seq, z, eltST, width - xCenterElt - 1, height - yCenterElt - 1);
	}

	/**
	 * Closing on a grey scale 2D image Dilates then erodes by the transposed
	 * structuring element
	 * 
	 * @param seq        Sequence to be closed
	 * @param z          Z value of the image to close
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void closeGreyScale(Sequence seq, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		dilateGreyScale(seq, z, eltS, xCenterElt, yCenterElt);
		// Erode by the transposed
		int width = eltS.length;
		int height = eltS[0].length;
		double[][] eltST = transpose(eltS);
		erodeGreyScale(seq, z, eltST, width - xCenterElt - 1, height - yCenterElt - 1);
	}

	/*
	 * Returns the transposed structuring element /!\ Central symetry /!\
	 */
	private static double[][] transpose(double[][] elt) {
		int width = elt.length;
		int height = elt[0].length;
		double[][] transp = new double[width][height];
		for (int j = 0; j < height; j++) {
			for (int i = 0; i < width; i++) {
				transp[i][j] = elt[width - i - 1][height - j - 1];
			}
		}
		return transp;
	}

	/**
	 * Opening on a grey scale 3D image Erodes then dilates
	 * 
	 * @param seq        Sequence to be closed
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * @param zCenterElt z of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void openGreyScale3D(Sequence seq, double[][][] eltS, int xCenterElt, int yCenterElt,
			int zCenterElt) {
		erodeGreyScale3D(seq, eltS, xCenterElt, yCenterElt, zCenterElt);
		// Dilate by transposed
		int depth = eltS[0][0].length;
		int height = eltS[0].length;
		int width = eltS.length;
		double[][][] eltST = transpose(eltS);
		dilateGreyScale3D(seq, eltST, width - 1 - xCenterElt, height - 1 - yCenterElt, depth - 1 - zCenterElt);
	}

	/**
	 * Closing on a grey scale 3D image Dilates then erodes
	 * 
	 * @param seq        Sequence to be closed
	 * @param eltS       structuring element
	 * @param xCenterElt x of the center of the element
	 * @param yCenterElt y of the center of the element
	 * @param zCenterElt z of the center of the element
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void closeGreyScale3D(Sequence seq, double[][][] eltS, int xCenterElt, int yCenterElt, int zCenterElt) {
		dilateGreyScale3D(seq, eltS, xCenterElt, yCenterElt, zCenterElt);
		// Erode by transposed
		int depth = eltS[0][0].length;
		int height = eltS[0].length;
		int width = eltS.length;
		double[][][] eltST = transpose(eltS);
		erodeGreyScale3D(seq, eltST, width - 1 - xCenterElt, height - 1 - yCenterElt, depth - 1 - zCenterElt);
	}

	/*
	 * Returns the transposed structuring element /!\ Central symetry /!\
	 */
	private static double[][][] transpose(double[][][] elt) {
		int depth = elt[0][0].length;
		int height = elt[0].length;
		int width = elt.length;
		double[][][] transp = new double[width][height][depth];
		for (int z = 0; z < depth; z++) {
			for (int j = 0; j < height; j++) {
				for (int i = 0; i < width; i++) {
					transp[i][j][z] = elt[width - i - 1][height - j - 1][depth - z - 1];
				}
			}
		}
		return transp;
	}

	// ------------------------------------------
	// Top hat transforms
	// -------------------------------------------

	/**
	 * White top hat, shows light areas, difference between the input opening of the
	 * input image and its identity
	 * 
	 * @param seqIn      input Sequence
	 * @param z          Z index of the image
	 * @param eltS       Structuring element
	 * @param xCenterElt center of the structuring element on x axis
	 * @param yCenterElt center of the structuring element on y axis
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */
	public static void whiteTopHat(Sequence seqIn, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		// Structuring element dimensions
		int eWidth = eltS.length;
		int eHeight = eltS[0].length;

		Sequence seqTemp = SequenceUtil.getCopy(seqIn);
		erodeGreyScale(seqTemp, z, eltS, xCenterElt, yCenterElt);
		// openGreyScale(seqIn,z, eltS, xCenterElt, yCenterElt);

		// Image dimensions
		int width = seqIn.getFirstImage().getWidth();
		int height = seqIn.getFirstImage().getHeight();

		// Transpose elt
		double[][] eltST = transpose(eltS);
		xCenterElt = eWidth - 1 - xCenterElt;
		yCenterElt = eHeight - 1 - yCenterElt;

		for (int t = 0; t < seqIn.getSizeT(); t++) {

			IcyBufferedImage img = seqIn.getImage(t, z);
			IcyBufferedImage imgTemp = seqTemp.getImage(t, z);
			int offset = 0;

			double[] tabInDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());
			double[] tabOutDouble = Array1DUtil.arrayToDoubleArray(imgTemp.getDataXY(0), imgTemp.isSignedDataType());

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++, offset++) {
					// min values for a byte
					double val = 0;
					// Check for all the pixels in the structuring element
					for (int yElt = 0; yElt < eHeight; yElt++) {
						for (int xElt = 0; xElt < eWidth; xElt++) {
							if (eltST[xElt][yElt] > 0) {
								// Offset relative to the structuring element
								int xRel = xElt - xCenterElt;
								int yRel = yElt - yCenterElt;
								// temporary values created on optimization
								// purpose
								int xTemp = xRel + x;
								int yTemp = yRel + y;
								// Check if the pixel is contained in the bound
								// of the image
								if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height) {
									int offRel = xTemp + width * (yTemp);
									// Pour l'érosion prendre le min sur l'elts
									double temp = tabOutDouble[offRel];
									if (temp > val)
										val = temp;
								}
							}
						}
					}
					tabInDouble[offset] = tabInDouble[offset] - val;
				}
			}
			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabInDouble, img.getDataXY(0)));

			// seqIn.setImage(t, z, imgTemp);
		}

	}

	/**
	 * Black top hat, shows dark areas, difference between the input image and its
	 * closing
	 * 
	 * @param seqIn      input Sequence
	 * @param z          Z index of the image
	 * @param eltS       Structuring element
	 * @param xCenterElt center of the structuring element on x axis
	 * @param yCenterElt center of the structuring element on y axis
	 * 
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void blackTopHat(Sequence seqIn, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		// Structuring element dimensions
		int eWidth = eltS.length;
		int eHeight = eltS[0].length;

		Sequence seqTemp = SequenceUtil.getCopy(seqIn);

		dilateGreyScale(seqTemp, z, eltS, xCenterElt, yCenterElt);
		// Transpose elt
		double[][] eltST = transpose(eltS);
		xCenterElt = eWidth - 1 - xCenterElt;
		yCenterElt = eHeight - 1 - yCenterElt;

		// Image dimensions
		int width = seqIn.getFirstImage().getWidth();
		int height = seqIn.getFirstImage().getHeight();

		for (int t = 0; t < seqIn.getSizeT(); t++) {

			IcyBufferedImage img = seqIn.getImage(t, z);
			IcyBufferedImage imgTemp = seqTemp.getImage(t, z);
			int offset = 0;

			double[] tabInDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());
			double[] tabOutDouble = Array1DUtil.arrayToDoubleArray(imgTemp.getDataXY(0), imgTemp.isSignedDataType());

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++, offset++) {
					// max value for a byte
					double val = Double.MAX_VALUE;
					// Check for all the pixels in the structuring element
					for (int yElt = 0; yElt < eHeight; yElt++) {
						for (int xElt = 0; xElt < eWidth; xElt++) {
							if (eltST[xElt][yElt] > 0) {
								// Offset relative to the structuring element
								int xRel = xElt - xCenterElt;
								int yRel = yElt - yCenterElt;
								// temporary values created on optimization
								// purpose
								int xTemp = xRel + x;
								int yTemp = yRel + y;
								// Check if the pixel is contained in the bound
								// of the image
								if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height) {
									int offRel = xTemp + width * (yTemp);
									// Pour l'érosion prendre le min sur l'elts
									double temp = tabOutDouble[offRel];
									if (temp < val)
										val = temp;
								}
							}
						}
					}
					tabInDouble[offset] = val - tabInDouble[offset];
				}
			}
			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabInDouble, img.getDataXY(0)));

			// seqIn.setImage(t, z, imgTemp);
		}
	}

	/**
	 * White top hat, shows light areas, difference between the input opening of the
	 * input image and its identity
	 * 
	 * @param seqIn      input Sequence
	 * @param z          Z index of the image
	 * @param eltS       Structuring element
	 * @param xCenterElt center of the structuring element on x axis
	 * @param yCenterElt center of the structuring element on y axis
	 * @param threshold  threshold
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void whiteTopHat3D(Sequence seqIn, double[][][] eltS, int xCenterElt, int yCenterElt,
			int zCenterElt) {

		// Structuring element dimensions
		int eWidth = eltS.length;
		int eHeight = eltS[0].length;
		int eDepth = eltS[0][0].length;

		Sequence seqTemp = SequenceUtil.getCopy(seqIn);

		erodeGreyScale3D(seqTemp, eltS, xCenterElt, yCenterElt, zCenterElt);

		// Transpose elt
		double[][][] eltST = transpose(eltS);
		xCenterElt = eWidth - 1 - xCenterElt;
		yCenterElt = eHeight - 1 - yCenterElt;
		zCenterElt = eDepth - 1 - zCenterElt;

		// Image dimensions
		int width = seqIn.getFirstImage().getWidth();
		int height = seqIn.getFirstImage().getHeight();

		for (int t = 0; t < seqIn.getSizeT(); t++) {

			int depth = seqIn.getSizeZ(t);

			double[][] dataInDouble = Array2DUtil.arrayToDoubleArray(seqIn.getDataXYZ(t, 0), seqIn.isSignedDataType());
			double[][] dataTempDouble = Array2DUtil.arrayToDoubleArray(seqTemp.getDataXYZ(t, 0),
					seqTemp.isSignedDataType());

			for (int z = 0; z < depth; z++) {
				int offset = 0;
				for (int y = 0; y < height; y++) {
					for (int x = 0; x < width; x++, offset++) {
						// min value for a byte
						double val = 0;

						for (int zElt = 0; zElt < eDepth; zElt++) {
							for (int yElt = 0; yElt < eHeight; yElt++) {
								for (int xElt = 0; xElt < eWidth; xElt++) {
									if (eltST[xElt][yElt][zElt] > 0) {
										// Values representing to position
										// of the pixel of the element
										// compared to the pixel (x,y)
										// xRel,yRel and zRel reprend values
										// in terms of rows, lines or stacks
										// For instance xRel = -1 means the
										// pixel is on the column (x-1)
										int xRel = xElt - xCenterElt;
										int yRel = yElt - yCenterElt;
										int zRel = zElt - zCenterElt;
										// Temp values for optimisation
										// purpose
										int xTemp = xRel + x;
										int yTemp = yRel + y;
										int zTemp = zRel + z;
										// Make sure the pixel is in the
										// bounds of the image
										if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height
												&& (zRel + z) >= 0 && (zRel + z) < seqIn.getSizeZ(t)) {
											// Take the min on the
											// structuring element for the
											// erosion
											double temp = dataTempDouble[zTemp][xTemp + width * (yTemp)];
											if (temp > val) {
												val = temp;
											}
										}
									}
								}
							}
						}

						dataInDouble[z][offset] = dataInDouble[z][offset] - val;
					}
				}
			}

			seqIn.beginUpdate();
			for (int z = 0; z < depth; z++) {
				IcyBufferedImage img = seqIn.getImage(t, z);
				img.setDataXY(0, Array1DUtil.doubleArrayToArray(dataInDouble[z], img.getDataXY(0)));
			}
			seqIn.endUpdate();
		}
	}

	/**
	 * Black top hat 3D, shows dark areas, difference between the input image and
	 * its closing
	 * 
	 * @param seqIn      input Sequence
	 * @param eltS       Structuring element
	 * @param xCenterElt center of the structuring element on x axis
	 * @param yCenterElt center of the structuring element on y axis
	 * @param zCenterElt center of the structuring element on z axis
	 * 
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void blackTopHat3D(Sequence seqIn, double[][][] eltS, int xCenterElt, int yCenterElt,
			int zCenterElt) {
		// Structuring element dimensions
		int eWidth = eltS.length;
		int eHeight = eltS[0].length;
		int eDepth = eltS[0][0].length;

		Sequence seqTemp = SequenceUtil.getCopy(seqIn);
		dilateGreyScale3D(seqTemp, eltS, xCenterElt, yCenterElt, zCenterElt);

		// Transpose elt
		double[][][] eltST = transpose(eltS);
		xCenterElt = eWidth - 1 - xCenterElt;
		yCenterElt = eHeight - 1 - yCenterElt;
		zCenterElt = eDepth - 1 - zCenterElt;

		// Image dimensions
		int width = seqIn.getFirstImage().getWidth();
		int height = seqIn.getFirstImage().getHeight();

		for (int t = 0; t < seqIn.getSizeT(); t++) {

			int depth = seqIn.getSizeZ(t);
			double[][] dataInDouble = Array2DUtil.arrayToDoubleArray(seqIn.getDataXYZ(t, 0), seqIn.isSignedDataType());
			double[][] dataOutDouble = Array2DUtil.arrayToDoubleArray(seqTemp.getDataXYZ(t, 0),
					seqTemp.isSignedDataType());

			for (int z = 0; z < depth; z++) {

				int offset = 0;

				for (int y = 0; y < height; y++) {
					for (int x = 0; x < width; x++, offset++) {

						// max values for a byte
						double val = Double.MAX_VALUE;

						for (int zElt = 0; zElt < eDepth; zElt++) {
							for (int yElt = 0; yElt < eHeight; yElt++) {
								for (int xElt = 0; xElt < eWidth; xElt++) {
									if (eltST[xElt][yElt][zElt] > 0) {
										// Values representing to position
										// of the pixel of the element
										// compared to the pixel (x,y)
										// xRel,yRel and zRel reprend values
										// in terms of rows, lines or stacks
										// For instance xRel = -1 means the
										// pixel is on the column (x-1)
										int xRel = xElt - xCenterElt;
										int yRel = yElt - yCenterElt;
										int zRel = zElt - zCenterElt;
										// Temp values for optimisation
										// purpose
										int xTemp = xRel + x;
										int yTemp = yRel + y;
										int zTemp = zRel + z;
										// Make sure the pixel is in the
										// bounds of the image
										if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height
												&& (zRel + z) >= 0 && (zRel + z) < depth) {
											// Take the min on the
											// structuring element for the
											// erosion
											val = Math.min(dataOutDouble[zTemp][xTemp + width * (yTemp)], val);
										}
									}
								}
							}
						}
						// Put the min
						dataInDouble[z][offset] = val - dataInDouble[z][offset];
					}
				}
			}

			seqIn.beginUpdate();
			for (int z = 0; z < depth; z++) {
				IcyBufferedImage img = seqIn.getImage(t, z);
				img.setDataXY(0, Array1DUtil.doubleArrayToArray(dataInDouble[z], img.getDataXY(0)));
			}
			seqIn.endUpdate();
		}
	}

	// --------------------------------------------
	// Gradient
	// --------------------------------------------

	/**
	 * Morphologic gradient Difference between the dilatation and erosion
	 * 
	 * @param seq        Sequence to transform
	 * @param z          Z stack to transform
	 * @param eltS       Structuring element ([x][y])
	 * @param xCenterElt center of the structuring element on x axis
	 * @param yCenterElt center of the structuring element on y axis
	 */
	public static void gradient(Sequence seq, int z, double[][] eltS, int xCenterElt, int yCenterElt) {
		// Structuring element dimensions
		double eWidth = eltS.length;
		double eHeight = eltS[0].length;

		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();

		Sequence seqTemp = SequenceUtil.getCopy(seq);

		for (int t = 0; t < seq.getSizeT(); t++) {

			int offset = 0;

			IcyBufferedImage img = seq.getImage(t, z);
			IcyBufferedImage imgTemp = seqTemp.getImage(t, z);
			double[] tabInDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());
			double[] tabOutDouble = Array1DUtil.arrayToDoubleArray(imgTemp.getDataXY(0), imgTemp.isSignedDataType());

			for (int y = 0; y < height; y++) {
				for (int x = 0; x < width; x++, offset++) {
					// Value containing the erosion
					double min = Double.MAX_VALUE;
					// Value containing the dilatation
					double max = 0;
					// Check for all the pixels in the structuring element
					for (int yElt = 0; yElt < eHeight; yElt++) {
						for (int xElt = 0; xElt < eWidth; xElt++) {
							if (eltS[xElt][yElt] > 0) {
								// Offset relative to the structuring element
								int xRel = xElt - xCenterElt;
								int yRel = yElt - yCenterElt;
								// temporary values created on optimization
								// purpose
								int xTemp = xRel + x;
								int yTemp = yRel + y;
								// Check if the pixel is contained in the bound
								// of the image
								if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height) {
									int offRel = xTemp + width * (yTemp);
									// Pour l'érosion prendre le min sur l'elts
									double temp = tabOutDouble[offRel];
									if (temp < min)
										min = temp;
									if (temp > max)
										max = temp;
								}
							}
						}
					}
					tabInDouble[offset] = (max - min);
				}
			}

			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabInDouble, img.getDataXY(0)));

		}
	}

	/**
	 * Morphologic gradient 3D Difference between the dilatation and erosion
	 * 
	 * @param seq        Sequence to transform
	 * @param eltS       Structuring element [x][y][z]
	 * @param xCenterElt center of the structuring element on x axis
	 * @param yCenterElt center of the structuring element on y axis
	 * @param zCenterElt center of the structuring element on z axis
	 */
	public static void gradient3D(Sequence seq, double[][][] eltS, int xCenterElt, int yCenterElt, int zCenterElt) {
		// Structuring element dimensions
		int eWidth = eltS.length;
		int eHeight = eltS[0].length;
		int eDepth = eltS[0][0].length;

		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();

		int sizeT = seq.getSizeT();

		for (int t = 0; t < sizeT; t++) {
			int depth = seq.getSizeZ(t);

			double[][] dataInDouble = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, 0), seq.isSignedDataType());
			double[][] dataOutDouble = Array2DUtil.doubleArrayToDoubleArray(dataInDouble, 0, null, 0,
					dataInDouble.length);

			for (int z = 0; z < depth; z++) {

				int offset = 0;
				for (int y = 0; y < height; y++) {
					for (int x = 0; x < width; x++, offset++) {

						// max and min values
						double min = Double.MAX_VALUE;
						double max = 0;
						// For each
						for (int zElt = 0; zElt < eDepth; zElt++) {
							for (int yElt = 0; yElt < eHeight; yElt++) {
								for (int xElt = 0; xElt < eWidth; xElt++) {
									if (eltS[xElt][yElt][zElt] > 0) {
										int xRel = xElt - xCenterElt;
										int yRel = yElt - yCenterElt;
										int zRel = zElt - zCenterElt;
										int xTemp = xRel + x;
										int yTemp = yRel + y;
										int zTemp = zRel + z;

										if ((xTemp) >= 0 && (yTemp) >= 0 && (xTemp) < width && (yTemp) < height
												&& (zTemp) >= 0 && (zTemp) < seq.getSizeZ(t)) {
											double temp = dataInDouble[zTemp][xTemp + width * (yTemp)];
											// min for erosion, max for
											// dilatation
											if (temp < min)
												min = temp;
											if (temp > max)
												max = temp;
										}
									}
								}
							}
						}
						// Put the min in the image
						dataOutDouble[z][offset] = max - min;
					}
				}
			}

			seq.beginUpdate();
			for (int z = 0; z < depth; z++) {
				IcyBufferedImage img = seq.getImage(t, z);
				img.setDataXY(0, Array1DUtil.doubleArrayToArray(dataOutDouble[z], img.getDataXY(0)));
			}
			seq.endUpdate();

		}
	}

	// --------------------------------------------------
	// Distance map
	// --------------------------------------------------
	/**
	 * Calculates the distance map of a sequence 2D + t The data type of the
	 * sequence is changed to double
	 * 
	 * @param seqIn     Input Sequence
	 * @param threshold Threshold to determine if a pixel belongs to the object
	 * @param Indicates whether the object is above the threshold or not (strictly)
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */

	public static void distanceMap2D(Sequence seqIn, int z, double threshold, boolean above) {
		DistanceTransforms distTrans = new DistanceTransforms();

		// Image dimensions
		int width = seqIn.getFirstImage().getWidth();
		int height = seqIn.getFirstImage().getHeight();
		// offset max
		int sizeImage = width * height;

		Sequence seqOut = new Sequence();
		for (int t = 0; t < seqIn.getSizeT(); t++) {

			IcyBufferedImage img = seqIn.getImage(t, z);

			IcyBufferedImage imgOut = new IcyBufferedImage(width, height, 1, DataType.DOUBLE);
			DataBufferDouble bufOut = (DataBufferDouble) imgOut.getRaster().getDataBuffer();
			double[] map = bufOut.getData();

			double[] tabInDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

			// double[] map= Arrays.copyOf(tabInDouble, tabInDouble.length);

			for (int i = 0; i < sizeImage; i++) {
				if (above ? (tabInDouble[i]) > threshold : (tabInDouble[i]) < threshold) {
					// Pixel belongs to the object, marked with infinity
					map[i] = sizeImage;
				} else {
					// Pixel doesn't belong to the object, marked with 0
					map[i] = 0;
				}
			}

			distTrans.updateUnsignedChamferDistance2D(width, height, map);
			seqOut.setImage(t, z, imgOut);

		}

		// Remove the images from the input sequence and put the new one : may
		// be a different datatype
		seqIn.removeAllImages();
		for (int t = 0; t < seqOut.getSizeT(); t++) {
			seqIn.setImage(t, z, seqOut.getImage(t, 0));
		}

	}

	/**
	 * Calculates the distance map of a sequence 3D + t The data type of the
	 * sequence is changed to double
	 * 
	 * @param seqIn     Sequence to transform
	 * @param threshold Threshold to determine if a pixel belongs to the object
	 * @param above     Indicates whether the object is above the threshold or not
	 *                  (strictly)
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */
	public static void distanceMap3D(Sequence seqIn, double threshold, boolean above) {
		Chamfer3 distTrans = new Chamfer3();

		// Taille de l'image
		int width = seqIn.getFirstImage().getWidth();
		int height = seqIn.getFirstImage().getHeight();

		// offset max
		int maxPix = width * height;

		Sequence seqOut = new Sequence();
		for (int t = 0; t < seqIn.getSizeT(); t++) {
			int depth = seqIn.getSizeZ(t);
			double[][] map = new double[depth][maxPix];

			byte[][] tabInDouble = seqIn.getDataXYZAsByte(t, 0); // Array1DUtil.arrayToDoubleArray(img.getDataXY(0),
																	// img.isSignedDataType());

			for (int z = 0; z < depth; z++) {
				for (int offset = 0; offset < maxPix; offset++) {
					if (above ? (tabInDouble[z][offset] & 0xff) > threshold
							: (tabInDouble[z][offset] & 0xff) < threshold) {
						// Pixel appartenant a l'objet, distance initialisee
						// a 1;
						map[z][offset] = maxPix;
					} else {
						// Pixel n'appartenant pas à l'objet, distance 0
						map[z][offset] = 0;
					}
				}
			}

			distTrans.updateUnsignedChamferDistance3D(width, height, depth, map);
			for (int z = 0; z < depth; z++) {
				IcyBufferedImage imgOut = new IcyBufferedImage(width, height, 1, DataType.DOUBLE);
				DataBufferDouble bufOut = (DataBufferDouble) imgOut.getRaster().getDataBuffer();
				double[] dataOut = bufOut.getData();
				for (int offset = 0; offset < maxPix; offset++) {
					dataOut[offset] = map[z][offset];
				}
				seqOut.setImage(t, z, imgOut);
			}

		}

		// Remove the images from the input sequence and put the new one : may
		// be a different datatype
		seqIn.removeAllImages();
		for (int t = 0; t < seqOut.getSizeT(); t++) {
			for (int z = 0; z < seqOut.getSizeZ(t); z++) {
				seqIn.setImage(t, z, seqOut.getImage(t, z));
			}
		}
	}

	// ----------------------------------------------------------------------------
	// Skeleton, author Vannary Meas-Yedid
	// ----------------------------------------------------------------------------

	// Representfs a fictive pixel, used as check point in some methods
	final private static Point FICTIF = new Point(-1, -1);
	final private static Point3D FICTIF3D = new Point3D.Float(-1, -1, -1);
	// Value of pixels out of the skeleton
	final private static byte CH0 = 0;
	// Values for pixels in the skeleton
	final private static byte CH1 = 1;
	final private static byte CH2 = 2;
	final private static byte CH3 = 3;

	/*
	 * Getters for an 1D array : returns the value at the 1D offset for the 2D
	 * coords x,y
	 */

	private static double getTabVal(double[] tab, int x, int y, int nc, int nr) {

		if (x < 0 || x >= nc || y < 0 || y >= nr)
			return (0);
		else {
			double d = (tab[x + nc * y]);
			return d;
		}
	}

	/*
	 * Setters for an 1D array : sets the value at the 1D offset for the 2D coords
	 * x,y
	 */

	@SuppressWarnings("unused")
	private static void setTabVal(double[] tab, int x, int y, double d, int nc, int nr) {
		if (x >= 0 && x < nc && y >= 0 && y < nr)
			tab[x + nc * y] = d;
	}

	private static double getTabVal3D(double[][] tab, int x, int y, int z, int nc, int nr, int nz) {
		if (x < 0 || x >= nc || y < 0 || y >= nr || z < 0 || z >= nz)
			return (0);
		else
			return (tab[z][x + nc * y]);
	}

	@SuppressWarnings("unused")
	private static void setTabVal3D(double[][] tab, int x, int y, int z, double d, int nc, int nr, int nz) {
		if (x >= 0 && x < nc && y >= 0 && y < nr && z >= 0 && z < nz)
			tab[z][x + nc * y] = d;
	}

	/*
	 * Tests if the value at the specified coordinates is higher than CH0
	 */
	private static boolean isPt(double[] tab, int i, int j, int ncols, int nrows) {
		return (getTabVal(tab, i, j, ncols, nrows) > CH0);
	}

	@SuppressWarnings("unused")
	private static boolean isPt3D(double[][] tab, int i, int j, int z, int ncols, int nrows, int nz) {
		return (getTabVal3D(tab, i, j, z, ncols, nrows, nz) > CH0);
	}

	@SuppressWarnings("unused")
	private static boolean isPtSkel(double[] tab, int i, int j, int ncols, int nrows) {
		// return (getTabVal(tab, i, j, ncols, nrows) == CH2);
		return true;
	}

	@SuppressWarnings("unused")
	private static boolean isPtSkel3D(double[][] tab, int i, int j, int z, int ncols, int nrows, int nz) {

		for (int m = z - 1; m < z + 1; m++) {
			for (int l = j - 1; l < j + 1; l++) {
				for (int k = i - 1; k < i + 1; k++) {
					if (getTabVal3D(tab, m, l, k, ncols, nrows, nz) == CH2)
						return true;
				}
			}
		}

		return false;

	}

	// Inverts a fifo (LinkedList)
	private static void invertFifo(LinkedList<Point> fifo) {
		int size = fifo.size();
		ArrayList<Point> list = new ArrayList<Point>();
		for (int i = 0; i < size; i++) {
			list.add(fifo.pollLast());
		}

		for (int i = 0; i < size; i++) {
			fifo.add(list.get(0));
			list.remove(0);
		}
	}

	private static void invertFifo3D(LinkedList<Point3D> fifo) {
		int size = fifo.size();
		ArrayList<Point3D> list = new ArrayList<Point3D>();
		for (int i = 0; i < size; i++) {
			list.add(fifo.pollLast());
		}

		for (int i = 0; i < size; i++) {
			fifo.add(list.get(0));
			list.remove(0);
		}
	}

	/*
	 * Initialisation de la squelettisation maximum : Les points d'ancrage sont les
	 * maxima locaux de la fonction distance - Retourne un tableau de l'image avec 0
	 * = pas de points 1 = point de l'ensemble 2 = point d'ancrage - L'image binaire
	 * source est supprime
	 * ======================================================================
	 */

	private static void initSquelet1(Sequence seq, int z, double threshold, boolean above) {

		double nmin, nmax;
		int i, j, offset;

		nmin = 1;
		nmax = Double.MAX_VALUE;

		int nc, nr;
		nc = seq.getFirstImage().getWidth();
		nr = seq.getFirstImage().getHeight();
		Sequence seqOut = SequenceUtil.getCopy(seq);
		boolean signed = seq.getDataType_().isSigned();
		// distanceMap2D(seqOut, z, threshold, above);
		double[] tab_Dist = new double[nc * nr];
		seqOut = getDistance(seqOut, 4); // VMY modif
		// Chamfer3 distanceMap = new Chamfer3();
		// distanceMap.unsignedChamferDistanceMap(seqOut, threshold, above);
		for (int t = 0; t < seq.getSizeT(); t++) {
			for (int c = 0; c < seq.getSizeC(); c++) {
				double n = 0;
				// DataBufferDouble bufOut = (DataBufferDouble)
				// seqOut.getImage(t, z).getRaster().getDataBuffer();
				// tab_Dist = bufOut.getData();

				Array1DUtil.arrayToDoubleArray(seqOut.getDataXY(t, z, c), tab_Dist, signed);
				offset = 0;

				// DataBufferDouble bufInDouble = (DataBufferDouble)
				// seq.getFirstImage().getRaster().getDataBuffer();
				// double[] tab_Img = bufInDouble.getData();

				IcyBufferedImage img = seq.getImage(t, z);
				double[] tab_Img = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());
				// double[] tabOutDouble = Arrays.copyOf(tabInDouble,
				// tabInDouble.length);
				// 2 maxima locaux ne doivent pas etre voisins + point d'ancrage
				// ?
				for (j = 0; j < nr; j++)
					for (i = 0; i < nc; i++) {
						if (tab_Img[i + j * nc] != 0)
							tab_Img[i + j * nc] = 3;
					}

				for (j = 0; j < nr; j++)
					for (i = 0; i < nc; i++) {
						// System.out.println(i + " " + j);
						offset = i + nc * j;
						n = getTabVal(tab_Dist, i, j, nc, nr);

						if (n >= nmin && n <= nmax) {

							if (getTabVal(tab_Dist, i, j - 1, nc, nr) <= n && getTabVal(tab_Dist, i - 1, j, nc, nr) <= n
									&& getTabVal(tab_Dist, i + 1, j, nc, nr) <= n
									&& getTabVal(tab_Dist, i, j + 1, nc, nr) <= n
									&& getTabVal(tab_Img, i - 1, j - 1, nc, nr) != CH2
									&& getTabVal(tab_Img, i, j - 1, nc, nr) != CH2
									&& getTabVal(tab_Img, i + 1, j - 1, nc, nr) != CH2
									&& getTabVal(tab_Img, i - 1, j, nc, nr) != CH2
									&& (i > 1 && i < (nc - 2) && j > 1 && j < (nr - 2)))
								tab_Img[offset] = CH2;
							else
								tab_Img[offset] = CH1;
						} else { // * Est-ce utile ?
							tab_Img[offset] = CH0;
							// if (n > nmax)
							// fifo.add(new Point(i, j));
						}
					}
				img.setDataXY(0, Array1DUtil.doubleArrayToArray(tab_Img, img.getDataXY(0)));
			}
		}
	}

	/*
	 * Initialisation du squelette lisse d'ordre n Remarque : Le squelette lisse
	 * contient le marquage homotopique
	 */
	private static void initSquelet2(Sequence seq, int z, double threshold, boolean above, int nmin) {

		double[] tab_Dist;

		int i, j, offset;

		int nc, nr;
		nc = seq.getFirstImage().getWidth();
		nr = seq.getFirstImage().getHeight();

		Sequence seqOut = SequenceUtil.getCopy(seq);
		// distanceMap2D(seqOut, z, threshold, above);
		seqOut = getDistance(seqOut, 8);

		for (int t = 0; t < seq.getSizeT(); t++) {
			DataBufferDouble bufOut = (DataBufferDouble) seqOut.getImage(0, z).getRaster().getDataBuffer();
			tab_Dist = bufOut.getData();
			offset = 0;
			// DataBufferDouble bufInDouble = (DataBufferDouble)
			// seq.getFirstImage().getRaster().getDataBuffer();
			// double[] tab_Img_Double = bufInDouble.getData();

			IcyBufferedImage img = seq.getImage(t, z);
			double[] tab_Img_Double = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

			// puts("Determination des maxima locaux.");
			for (j = 0; j < nr; j++)
				for (i = 0; i < nc; i++, offset++) {
					double n = getTabVal(tab_Dist, i, j, nc, nr);
					if (n >= nmin) {
						if (getTabVal(tab_Dist, i - 1, j - 1, nc, nr) <= n && getTabVal(tab_Dist, i, j - 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i + 1, j - 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 1, j, nc, nr) <= n
								&& getTabVal(tab_Dist, i + 1, j, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 1, j + 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i, j + 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i + 1, j + 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i + 2, j - 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i + 2, j - 2, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 1, j - 2, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 2, j - 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 2, j + 1, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 2, j + 2, nc, nr) <= n
								&& getTabVal(tab_Dist, i - 1, j + 2, nc, nr) <= n
								&& getTabVal(tab_Dist, i + 2, j + 1, nc, nr) <= n
								&& getTabVal(tab_Img_Double, i + 2, j - 1, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i + 2, j - 2, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i - 1, j - 2, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i - 2, j - 1, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i - 1, j - 1, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i, j - 1, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i + 1, j - 1, nc, nr) != CH2
								&& getTabVal(tab_Img_Double, i - 1, j, nc, nr) != CH2)
							tab_Img_Double[offset] = CH2;
						else
							tab_Img_Double[offset] = CH1;
					} else
						tab_Img_Double[offset] = CH0;
				}
			seq.beginUpdate();

			// IcyBufferedImage img = seq.getImage(t, 0);
			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tab_Img_Double, img.getDataXY(0)));

			seq.endUpdate();
		}
	}

	private static void initSquelet3D(Sequence seq, double threshold, boolean above) {

		double nmin, nmax;
		int i, j, offset, z;

		nmin = 1;
		nmax = Double.MAX_VALUE;

		int nc, nr, nz;
		nc = seq.getFirstImage().getWidth();
		nr = seq.getFirstImage().getHeight();
		nz = seq.getSizeZ();
		Sequence seqOut = SequenceUtil.getCopy(seq);
		testDistance3D(seqOut);
		// distanceMap3D(seqOut, threshold, above);
		// Chamfer3 distanceMap = new Chamfer3();
		// distanceMap.unsignedChamferDistanceMap(seqOut, threshold, above);
		int depth = seq.getSizeZ();
		// tab de la distance map 3D
		double[][] tab_Dist = new double[depth][nc * nr];
		// tab de la sequence
		double[][] tab_Img = new double[depth][nc * nr];
		for (int t = 0; t < seq.getSizeT(); t++) {
			for (int c = 0; c < seq.getSizeC(); c++) {
				double n = 0;
				Array2DUtil.arrayToDoubleArray(seqOut.getDataXYZ(t, c), tab_Dist, seqOut.isSignedDataType());
				Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, c), tab_Img, seq.isSignedDataType());
				offset = 0;

				// On cherche les maximas locaux
				for (z = 0; z < nz; z++) {
					for (j = 0; j < nr; j++) {
						for (i = 0; i < nc; i++, offset++) {
							n = getTabVal3D(tab_Dist, i, j, z, nc, nr, nz);
							if (n >= nmin && n <= nmax) {
								if (getTabVal3D(tab_Dist, i - 1, j - 1, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j - 1, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j - 1, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i - 1, j, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i - 1, j + 1, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j + 1, z - 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j + 1, z - 1, nc, nr, nz) <= n

										&& getTabVal3D(tab_Dist, i - 1, j - 1, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j - 1, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j - 1, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i - 1, j, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i - 1, j + 1, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j + 1, z, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j + 1, z, nc, nr, nz) <= n

										&& getTabVal3D(tab_Dist, i - 1, j - 1, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j - 1, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j - 1, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i - 1, j, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i - 1, j + 1, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i, j + 1, z + 1, nc, nr, nz) <= n
										&& getTabVal3D(tab_Dist, i + 1, j + 1, z + 1, nc, nr, nz) <= n

										&& getTabVal3D(tab_Img, i - 1, j - 1, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i, j - 1, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i + 1, j - 1, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i - 1, j, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i, j, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i + 1, j, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i - 1, j + 1, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i, j + 1, z - 1, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i + 1, j + 1, z - 1, nc, nr, nz) != CH2

										&& getTabVal3D(tab_Img, i - 1, j - 1, z, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i, j - 1, z, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i + 1, j - 1, z, nc, nr, nz) != CH2
										&& getTabVal3D(tab_Img, i - 1, j, z, nc, nr, nz) != CH2)

									tab_Img[z][offset] = CH2;
								else
									tab_Img[z][offset] = CH1;
							} else {
								tab_Img[z][offset] = CH0;
								// if (n > nmax)
								// fifo.add(new Point3D.Float(i, j, z));
							}
						}
					}
					offset = 0;
				}

				/*
				 * for (z =0; z < nz; z ++) { IcyBufferedImage img = seq.getImage(t, z);
				 * double[] tab = new double[tab_Img.length]; for (i=0;i<tab_Img.length;i++) {
				 * tab[i]=tab_Img[i][z]; } img.setDataXY(0, Array1DUtil.doubleArrayToArray(tab,
				 * img.getDataXY(0)));
				 * 
				 * }
				 */
				Array2DUtil.doubleArrayToArray(tab_Img, seq.getDataXYZ(t, c));
			}
		}
	}

	private static boolean confBarb(double[] tab, int i, int j, int ncols, int nrows) {
		boolean CurrentP;
		int n = 0;
		int v = 0;
		CurrentP = isPt(tab, i + 1, j, ncols, nrows);
		if (!CurrentP)
			v++;
		if (isPt(tab, i + 1, j + 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i + 1, j + 1, ncols, nrows);
		}
		if (!CurrentP)
			v++;
		if (isPt(tab, i, j + 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i, j + 1, ncols, nrows);
		}
		if (!CurrentP)
			v++;
		if (isPt(tab, i - 1, j + 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i - 1, j + 1, ncols, nrows);
		}
		if (!CurrentP)
			v++;
		if (isPt(tab, i - 1, j, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i - 1, j, ncols, nrows);
		}
		if (!CurrentP)
			v++;
		if (isPt(tab, i - 1, j - 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i - 1, j - 1, ncols, nrows);
		}
		if (!CurrentP)
			v++;
		if (isPt(tab, i, j - 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i, j - 1, ncols, nrows);
		}
		if (!CurrentP)
			v++;
		if (isPt(tab, i + 1, j - 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i + 1, j - 1, ncols, nrows);
		}

		return ((n <= 2 || n == 0) && v >= 5);

	}

	// pas disponible
	private static boolean confBarb3D(double[][] tab, int i, int j, int z, int ncols, int nrows, int nz) {

		return false;
	}

	private static boolean confHomo(double[] tab, int i, int j, int ncols, int nrows) {
		boolean CurrentP;
		int n = 0;

		CurrentP = isPt(tab, i + 1, j, ncols, nrows);

		if (isPt(tab, i + 1, j - 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i + 1, j - 1, ncols, nrows);
		}
		if (isPt(tab, i, j - 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i, j - 1, ncols, nrows);
		}
		if (isPt(tab, i - 1, j - 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i - 1, j - 1, ncols, nrows);
		}
		if (isPt(tab, i - 1, j, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i - 1, j, ncols, nrows);
		}
		if (isPt(tab, i - 1, j + 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i - 1, j + 1, ncols, nrows);
		}
		if (isPt(tab, i, j + 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i, j + 1, ncols, nrows);
		}
		if (isPt(tab, i + 1, j + 1, ncols, nrows) != CurrentP) {
			n++;
			CurrentP = isPt(tab, i + 1, j - 1, ncols, nrows);
		}
		/*
		 * if (isPt(tab,i+1,j)!=CurrentP) { n++; CurrentP=isPt(tab,i+1,j); }
		 */
		// plus de voisins vides non consécutif
		return (n > 2 || n == 0);
	}

	// on verifie qu'on peut supprimer le pixel en conservant la connexite
	private static boolean confHomo3D(double[][] tab, int i, int j, int z, int ncols, int nrows, int nz) {
		Sequence input = new Sequence();

		int offset = 0;
		Map<Integer, List<ConnectedComponent>> componentsMap = new TreeMap<Integer, List<ConnectedComponent>>();

		for (int m = z - 1; m <= z + 1; m++) {
			double[] tabVoxel = new double[3 * 3];
			for (int l = j - 1; l <= j + 1; l++) {
				for (int k = i - 1; k <= i + 1; k++, offset++) {
					if (getTabVal3D(tab, k, l, m, ncols, nrows, nz) != 0) {
						tabVoxel[offset] = 255;
					} else {
						tabVoxel[offset] = 0;
					}
				}
			}
			// voxel central m = z du cube
			if (m == z)
				tabVoxel[4] = 0;
			offset = 0;
			IcyBufferedImage image = new IcyBufferedImage(3, 3, tabVoxel);

			input.addImage(image);
		}
		int nbObjects = 0;
		componentsMap = ConnectedComponents.extractConnectedComponents(input, 255, ExtractionType.VALUE, false, false,
				false, 0, Integer.MAX_VALUE, null);
		for (List<ConnectedComponent> ccs : componentsMap.values()) {
			nbObjects += ccs.size();
		}
		if (nbObjects > 1 || nbObjects == 0) {
			System.out.println("x : " + i + " | y : " + j + " | z : " + z);
		}
		// si nbOjects == 1, on peut supprimer le pixel central
		return (nbObjects > 1);
	}

	/*
	 * CH0 vide CH1 forme CH2 pt ancrage CH3 squelette
	 */
	private static void progage3D(Sequence seq) {
		int nc, nr, nz;

		nc = seq.getFirstImage().getWidth();
		nr = seq.getFirstImage().getHeight();
		nz = seq.getSizeZ();

		int connex = 4;
		int[] d = new int[4];

		d[0] = 1;
		d[1] = nc;
		d[2] = -1;
		d[3] = -nc;

		// d[4] = 1 + nc; d[5] = -1 + nc; d[6] = -1 - nc; d[7] = 1 - nc;

		int k, i, j, l, z, m;
		int offset;
		int sizeImage = nc * nr;
		Point3D p;
		LinkedList<Point3D> fifo = new LinkedList<Point3D>();
		LinkedList<Point3D> tabPix = new LinkedList<Point3D>();
		double[][] tabDouble = new double[nz][nr * nc];

		for (int t = 0; t < seq.getSizeT(); t++) {
			for (int c = 0; c < seq.getSizeC(); c++) {
				Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, c), tabDouble, seq.isSignedDataType());

				// pixels de contour
				for (z = 0; z < nz; z++) {
					for (j = 0; j < nr; j++) {
						for (i = 0; i < nc; i++) {
							if (tabDouble[z][i + nc * j] == CH1) {
								k = 0;
								offset = i + nc * j;
								while (k < connex) {
									if ((offset + d[k]) >= 0 && (offset + d[k]) < sizeImage
											&& tabDouble[z][offset + d[k]] == CH0) {

										fifo.add(new Point3D.Float(i, j, z));
										tabDouble[z][i + nc * j] = CH3;

										break;
									}
									k++;

								}

							}
						}
					}
				}

				fifo.add(FICTIF3D);

				while (true) {

					p = fifo.poll();
					i = (int) p.getX();
					j = (int) p.getY();
					z = (int) p.getZ();
					if (p.equals(FICTIF3D)) {

						if (fifo.isEmpty()) {
							for (int q = 0; q < tabPix.size(); q++) {
								p = tabPix.get(q);
								i = (int) p.getX();
								j = (int) p.getY();
								z = (int) p.getZ();
								if (confBarb3D(tabDouble, i, j, z, nc, nr, nz)) {
									tabDouble[z][i + nc * j] = CH1; // CH2 LV
									fifo.add(new Point3D.Float(i, j, z));

								}
							}

							while (!fifo.isEmpty()) {
								p = fifo.poll();

								i = (int) p.getX();
								j = (int) p.getY();
								z = (int) p.getZ();
								if (confBarb3D(tabDouble, i, j, z, nc, nr, nz)) {
									tabDouble[z][i + nc * j] = CH0;

									for (m = z - 1; m <= z + 1; m++)
										for (l = j - 1; l <= j + 1; l++)
											for (k = i - 1; k <= i + 1; k++)
												if (getTabVal3D(tabDouble, k, l, m, nc, nr, nz) == CH3
														&& (k != i || l != j)) {
													fifo.add(new Point3D.Float(k, l, m));
													tabDouble[m][k + nc * l] = CH1; // CH2
												}
								} else
									tabDouble[i + nc * j][z] = CH3;
							}

							for (int q = 0; q < tabPix.size(); q++) {
								p = tabPix.get(q);
								k = (int) p.getX();
								l = (int) p.getY();
								m = (int) p.getZ();
								if (tabDouble[m][k * nr + l] == CH3)
									tabDouble[m][k * nr + l] = CH2;
							}
							break;
						} else {
							invertFifo3D(fifo);
							fifo.add(FICTIF3D);
						}
					} else {
						if (getTabVal3D(tabDouble, i - 1, j, z, nc, nr, nz) == CH1) {
							fifo.add(new Point3D.Float(i - 1, j, z));
							tabDouble[z][(i - 1) + nc * j] = CH3;
						}
						if (getTabVal3D(tabDouble, i, j - 1, z, nc, nr, nz) == CH1) {
							fifo.add(new Point3D.Float(i, j - 1, z));
							tabDouble[z][i + nc * (j - 1)] = CH3;
						}
						if (getTabVal3D(tabDouble, i + 1, j, z, nc, nr, nz) == CH1) {
							fifo.add(new Point3D.Float(i + 1, j, z));
							tabDouble[z][(i + 1) + nc * j] = CH3;
						}
						if (getTabVal3D(tabDouble, i, j + 1, z, nc, nr, nz) == CH1) {
							fifo.add(new Point3D.Float(i, j + 1, z));
							tabDouble[z][i + nc * (j + 1)] = CH3;
						}
						/*
						 * if (getTabVal3D(tabDouble, i, j, z-1, nc, nr, nz) == CH1) { fifo.add(new
						 * Point3D.Float(i, j, z-1)); tabDouble[z-1][i + nc * j] = CH3; } if
						 * (getTabVal3D(tabDouble, i, j, z+1, nc, nr, nz) == CH1) { fifo.add(new
						 * Point3D.Float(i, j, z+1)); tabDouble[z+1][i + nc * j] = CH3; }
						 */
						Array2DUtil.doubleArrayToArray(tabDouble, seq.getDataXYZ(t, c));
						seq.dataChanged();
						if (confHomo3D(tabDouble, i, j, z, nc, nr, nz)) {
							tabPix.add(new Point3D.Float(i, j, z));
						} else {
							tabDouble[z][i + nc * j] = CH0;
						}

					}

				}
				Array2DUtil.doubleArrayToArray(tabDouble, seq.getDataXYZ(t, c));
			}
		}
	}

	private static void propage(Sequence seq, int z) {
		int nc, nr;

		nc = seq.getFirstImage().getWidth();
		nr = seq.getFirstImage().getHeight();
		int connex = 4;
		int[] d = new int[8];
		d[0] = 1;
		d[1] = nc;
		d[2] = -1;
		d[3] = -nc;
		d[4] = 1 + nc;
		d[5] = -1 + nc;
		d[6] = -1 - nc;
		d[7] = 1 - nc;
		int t = 0;
		int k, i, j, l;
		int offset;
		int sizeImage = nc * nr;
		LinkedList<Point> fifo = new LinkedList<Point>();
		LinkedList<Point> tabPix = new LinkedList<Point>();
		Point p;
		IcyBufferedImage img = seq.getImage(t, z);
		double[] tabDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

		for (j = 0; j < nr; j++)
			for (i = 0; i < nc; i++) { // Initialisation de la file d'attente
				if (tabDouble[i + nc * j] != CH0) {
					boolean flag = false;
					k = 0;
					offset = i + nc * j;
					// voisinage selon son choix
					while ((flag == false) && (k < connex)) {
						if ((offset + d[k]) >= 0 && (offset + d[k]) < sizeImage && tabDouble[offset + d[k]] == CH0) {

							fifo.add(new Point(i, j));
							tabDouble[i + nc * j] = CH3;

							flag = true;
						}
						k++;
					}
				}

			}
		img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabDouble, img.getDataXY(0)));
		fifo.add(FICTIF);

		// puts("Propagation.");
		while (true) {
			p = fifo.poll();
			i = p.x;
			j = p.y;
			if (p.equals(FICTIF)) {

				if (fifo.isEmpty()) {
					for (int q = 0; q < tabPix.size(); q++) {
						p = tabPix.get(q);
						i = p.x;
						j = p.y;
						if (confBarb(tabDouble, i, j, nc, nr)) {
							tabDouble[i + nc * j] = CH1; // CH2 LV
							fifo.add(new Point(i, j));

						}
					}

					while (!fifo.isEmpty()) {
						p = fifo.poll();
						i = p.x;
						j = p.y;
						if (confBarb(tabDouble, i, j, nc, nr)) {
							tabDouble[i + nc * j] = CH0;

							for (l = j - 1; l <= j + 1; l++)
								for (k = i - 1; k <= i + 1; k++)
									if (getTabVal(tabDouble, k, l, nc, nr) == CH3 && (k != i || l != j)) {

										fifo.add(new Point(k, l));
										tabDouble[k + nc * l] = CH1; // CH2
									}
						} else
							tabDouble[i + nc * j] = CH3;
					}

					for (int q = 0; q < tabPix.size(); q++) {
						p = tabPix.get(q);
						k = p.x;
						l = p.y;
						if (tabDouble[k * nr + l] == CH3)
							tabDouble[k * nr + l] = CH2;
					}
					break;
				} else {

					invertFifo(fifo);
					fifo.add(FICTIF);
				}
			} else {
				for (l = j - 1; l <= j + 1; l++)
					for (k = i - 1; k <= i + 1; k++)
						if (getTabVal(tabDouble, k, l, nc, nr) == CH1 && (k == i || l == j)) {
							fifo.add(new Point(k, l));
							tabDouble[k + nc * l] = CH3;
							break;
						}
				if (confHomo(tabDouble, i, j, nc, nr)) {
					tabPix.add(new Point(i, j));
				} else {
					tabDouble[i + nc * j] = CH0;
				}
			}
		}

		img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabDouble, img.getDataXY(0)));
	}

	/* A revoir */
	private static void initVoisins(Point[] voisin) {
		/* Codage Freeman */
		voisin[0] = new Point(1, 0);
		voisin[1] = new Point(1, -1);
		voisin[2] = new Point(0, -1);
		voisin[3] = new Point(-1, -1);
		voisin[4] = new Point(-1, 0);
		voisin[5] = new Point(-1, 1);
		voisin[6] = new Point(0, 1);
		voisin[7] = new Point(1, 1);
	}

	private static boolean ebarbulage(double[] tab, int i, int j, int nc, int nr) {
		int tabVois;
		int k;
		Point[] voisin = new Point[8];

		initVoisins(voisin);
		tabVois = 0;
		for (k = 0; k < 8; k++) {
			if (isPt(tab, i + voisin[k].x, j + voisin[k].y, nc, nr))
				tabVois |= (1 << k);
		}

		for (k = 0; k < 4; k++) {
			/* struture en T */
			if (((tabVois & (1 << 2 * k)) == 1) && (((tabVois & (1 << ((2 + 2 * k) % 8)))) == 1)
					&& ((tabVois & (1 << ((4 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((1 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((3 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((5 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((6 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((7 + 2 * k) % 8)))) == 1) {
				/* printf (" cas T :eb : %i %j \n", *j,*i); */
				return (true);
			}
			/* Structure en T2 oct 2002 */
			if (((tabVois & (1 << 1 + 2 * k)) == 1) && ((tabVois & (1 << ((2 + 2 * k) % 8))) == 1)
					&& ((tabVois & (1 << ((3 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((4 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((5 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((6 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((7 + 2 * k) % 8)))) == 1) {
				return (true);
			}

			if (((tabVois & (1 << (2 * k))) == 1) && ((tabVois & (1 << ((2 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((1 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((3 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((4 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((5 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((6 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((7 + 2 * k) % 8)))) == 1) {
				/* printf (" Cas 1 eb : %i %i \n", *j,*i); */
				return (true);
			}

			if (((tabVois & (1 << 2 * k)) == 1) && ((tabVois & (1 << ((1 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((2 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((3 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((4 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((5 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((6 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((7 + 2 * k) % 8)))) == 1) {
				/* printf ("Cas 2 eb : %i %i \n", *j,*i); */
				return (true);
			}

			if (((tabVois & (1 << ((2 + 2 * k) % 8))) == 1) && ((tabVois & (1 << ((1 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((2 * k) % 8))) == 1) && ((~tabVois & (1 << ((3 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((4 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((5 + 2 * k) % 8))) == 1)
					&& ((~tabVois & (1 << ((6 + 2 * k) % 8))) == 1) && ((~tabVois & (1 << ((7 + 2 * k) % 8)))) == 1) {
				{
					/* printf ("Cas 4 eb : %i %j \n", *j,*i); */
					return (true);
				}
			}
		}

		return (false);
	}

	private static void ebarbulFinal(Sequence seq, int z) {

		for (int t = 0; t < seq.getSizeT(); t++) {
			int i, j;
			boolean val;
			int nc = seq.getFirstImage().getWidth(), nr = seq.getFirstImage().getHeight();

			IcyBufferedImage img = seq.getImage(t, z);
			double[] tabDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

			for (j = 1; j < nr; j++)
				for (i = 1; i < nc; i++) {
					if ((tabDouble[i + nc * j] == CH2)) {
						val = ebarbulage(tabDouble, i, j, nc, nr);
						if (val)
							tabDouble[i + nc * j] = CH0;
					}
				}

			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabDouble, img.getDataXY(0)));
		}
	}

	public static int getNb4Neighbors(double[] tab, int i, int j, int width, int height) {
		int nb = 0;
		if (getTabVal(tab, i - 1, j, width, height) > 0)
			nb++;
		if (getTabVal(tab, i, j - 1, width, height) > 0)
			nb++;
		if (getTabVal(tab, i + 1, j, width, height) > 0)
			nb++;
		if (getTabVal(tab, i, j + 1, width, height) > 0)
			nb++;
		return nb;
	}

	private static void four2Eight(Sequence seqIn, int z) {

		int x, y;
		int N1r, N1c, N2r, N2c;
		int width = seqIn.getFirstImage().getWidth(), height = seqIn.getFirstImage().getHeight();
		for (int t = 0; t < seqIn.getSizeT(); t++) {
			int offset = 0;

			IcyBufferedImage img = seqIn.getImage(t, z);
			double[] tabDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

			for (y = 0; y < height; y++)
				for (x = 0; x < width; x++, offset++) {
					if (x == 551 && (y == 150))
						System.out.print(" point ");
					N1r = x - 1;
					N1c = y;
					N2r = x;
					N2c = y + 1;
					if (isPt(tabDouble, x, y, width, height) && isPt(tabDouble, N1r, N1c, width, height)
							&& isPt(tabDouble, N2r, N2c, width, height)) {
						if (!isPt(tabDouble, x + 1, y - 1, width, height))
							tabDouble[offset] = CH0;
					} else {
						N1r = x - 1;
						N1c = y;
						N2r = x;
						N2c = y - 1;
						if (isPt(tabDouble, x, y, width, height) && isPt(tabDouble, N1r, N1c, width, height)
								&& (isPt(tabDouble, N2r, N2c, width, height))) {
							if (!isPt(tabDouble, x + 1, y + 1, width, height))
								tabDouble[offset] = CH0;
						} else {
							N1r = x + 1;
							N1c = y;
							N2r = x;
							N2c = y - 1;
							if (isPt(tabDouble, x, y, width, height) && isPt(tabDouble, N1r, N1c, width, height)
									&& (isPt(tabDouble, N2r, N2c, width, height))) {
								if (!isPt(tabDouble, x - 1, y + 1, width, height))
									tabDouble[offset] = CH0;
							}

							else {
								N1r = x + 1;
								N1c = y;
								N2r = x;
								N2c = y + 1;
								if (isPt(tabDouble, x, y, width, height) && isPt(tabDouble, N1r, N1c, width, height)
										&& (isPt(tabDouble, N2r, N2c, width, height))) {
									if (!isPt(tabDouble, x - 1, y - 1, width, height))
										tabDouble[offset] = CH0;
								}

							}

						}
					}

					// T case

					if (getNb4Neighbors(tabDouble, x, y, width, height) == 3) {
						tabDouble[offset] = CH0;
					}
					// case L
					N1r = x - 1;
					N1c = y;
					N2r = x;
					N2c = y + 1;
					if (isPt(tabDouble, x, y, width, height) && isPt(tabDouble, N1r, N1c, width, height)
							&& isPt(tabDouble, N2r, N2c, width, height)) {
						if (!isPt(tabDouble, x + 1, y - 1, width, height)
								&& !isPt(tabDouble, x - 1, y - 1, width, height)
								&& !isPt(tabDouble, x + 1, y, width, height))
							tabDouble[offset] = CH0;
					}
				}

			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabDouble, img.getDataXY(0)));
		}

		// TODO VERIFICATION: tabDouble ou seqIn
		/*
		 * Correction
		 */
		for (int t = 0; t < seqIn.getSizeT(); t++) {
			int offset = 0;

			IcyBufferedImage img = seqIn.getImage(t, z);
			double[] tabDouble = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

			for (y = 0; y < height; y++)
				for (x = 0; x < width; x++, offset++) {

					if (tabDouble[offset] > 0) {
						ArrayList<Integer> lNeighbors = getOffsetNeighbors(x, y, width, height);
						int nbNeighbors = 0;
						int[] tabVoisins = new int[8];
						for (int i = 0; i < lNeighbors.size(); i++) {
							if (tabDouble[lNeighbors.get(i)] > 0) {
								tabVoisins[nbNeighbors] = lNeighbors.get(i);
								nbNeighbors++;
							}
						}

						if (nbNeighbors == 1) {
							ArrayList<Integer> lNeighbor2 = getOffsetNeighbors(tabVoisins[0], width, height);
							int nbNb = 0;
							int[] tabVoisins2 = new int[8];
							for (int i = 0; i < lNeighbor2.size(); i++) {
								if (tabDouble[lNeighbor2.get(i)] > 0) {
									tabVoisins2[nbNb] = lNeighbor2.get(i);
									nbNb++;
								}
							}
							// Si son voisin a plus d'un autre voisin c'est
							// que ce n'est pas une extremité
							if (nbNb > 2) {
								int offsetVoisin = tabVoisins[0];
								int xp = offsetVoisin % width;
								int yp = (int) Math.floor(offsetVoisin / width);
								if (nbNb == 3) {
									// Verifier que ce n'est pas une pointe
									// de Y allonge avant de supprimer
									int x1 = 0, y1 = 0, x2 = 0, y2 = 0;
									boolean trouve = false;
									if (tabVoisins2[0] != offset) {
										x1 = tabVoisins2[0] % width;
										y1 = (int) Math.floor(tabVoisins2[0] / width);
									} else {
										trouve = true;
									}
									if (tabVoisins2[1] != offset) {
										if (trouve) {
											x1 = tabVoisins2[1] % width;
											y1 = (int) Math.floor(tabVoisins2[1] / width);
										} else {
											x2 = tabVoisins2[1] % width;
											y2 = (int) Math.floor(tabVoisins2[1] / width);
										}
									} else {
										trouve = true;
									}
									if (trouve) {
										x2 = tabVoisins2[2] % width;
										y2 = (int) Math.floor(tabVoisins2[2] / width);
									}
									// Y
									if (y1 != yp && y1 == y2 && x1 + x2 == xp + xp) {
										continue;
									}
									if (x1 != xp && x1 == x2 && y1 + y2 == yp + yp) {
										continue;
									}
								}

								// Verifier que ce n'est pas le bout d'un
								// grand L
								if (xp == x) {
									if (yp == y - 1 && isPt(tabDouble, x, yp - 1, width, height)) {
										continue;
									}
									if (yp == y + 1 && isPt(tabDouble, x, yp + 1, width, height)) {
										continue;
									}
								}
								if (yp == y) {
									if (xp == x - 1 && isPt(tabDouble, xp - 1, y, width, height)) {
										continue;
									}
									if (xp == x + 1 && isPt(tabDouble, xp + 1, y, width, height)) {
										continue;
									}
								}
								tabDouble[offset] = CH0;
							}
						}

						else if (nbNeighbors == 2) {
							int off1 = tabVoisins[0];
							int x1 = off1 % width;
							int y1 = (int) Math.floor(off1 / width);
							int off2 = tabVoisins[1];
							int x2 = off2 % width;
							int y2 = (int) Math.floor(off2 / width);

							/*
							 * Extremite de L
							 */
							if (x1 == x - 1 && x1 == x2) {
								if ((y1 == y && (y2 == y + 1 || y2 == y - 1))
										|| y2 == y && (y1 == y + 1 || y1 == y - 1)) {
									tabDouble[offset] = CH0;// 1 et 5
									continue;
								}
							}
							if (y1 == y + 1 && y1 == y2) {
								if ((x1 == x && (x2 == x - 1 || x2 == x + 1))
										|| (x2 == x && (x1 == x - 1 || x1 == x + 1))) {
									tabDouble[offset] = CH0;// 2 et 6
									continue;
								}
							}
							if (x1 == x + 1 && x1 == x2) {
								if ((y1 == y && (y2 == y - 1 || y2 == y + 1))
										|| (y2 == y && (y1 == y - 1 || y2 == y + 1))) {
									tabDouble[offset] = CH0; // 3 et 7
									continue;
								}
							}
							if (y1 == y - 1 && y1 == y2) {
								if ((x1 == x && (x2 == x + 1 || x2 == x - 1))
										|| (x2 == x && (x1 == x + 1 || x1 == x - 1))) {
									tabDouble[offset] = CH0; // 4 et 8
									continue;
								}
							}
							/*
							 * Ajustement de l'alignement
							 */

							if (y1 != y && y1 == y2 && x1 + x2 == x + x) {
								tabDouble[y1 * width + x] = tabDouble[offset];
								tabDouble[offset] = CH0; // alignement
															// horizontal
								continue;
							}
							if (x1 != x && x1 == x2 && y1 + y2 == y + y) {
								tabDouble[x1 + y * width] = tabDouble[offset];
								tabDouble[offset] = CH0;
								continue;
							}

						}

					}
				}

			img.setDataXY(0, Array1DUtil.doubleArrayToArray(tabDouble, img.getDataXY(0)));
		}
	}

	/**
	 * Calculates the 2D skeleton
	 * 
	 * @param seq       Sequence to be analyzed
	 * @param z         : Z index of the stack to analyze
	 * @param threshold : Threshold to recognize the object
	 * @param above     : Indicates whether the object is above the threshold or not
	 *                  (strictly)
	 * @param typeSkel  : the type of skeleton
	 * @param n         : smoothing order, this value only matters for a smooth
	 *                  skeleton
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */
	public static void skeleton(Sequence seq, int z, double threshold, boolean above, int typeSkel, int n) {

		switch (typeSkel) {
		case TYPESKEL_MAX:
			initSquelet1(seq, z, threshold, above);
			break;
		case TYPESKEL_SMOOTH:
			initSquelet2(seq, z, threshold, above, n);
			// initSqueletTest(seq, z, threshold, above);
			break;
		}

		propage(seq, z);
		ebarbulFinal(seq, z);
		four2Eight(seq, z);

	}

	public static void skeleton3D(Sequence seq, double threshold, boolean above) {
		initSquelet3D(seq, threshold, above);
		progage3D(seq);
	}

	public static int valPixel(double[][] tab, int i, int j, int k, int width, int height, int depth) {
		if (getTabVal3D(tab, i, j, k - 1, width, height, depth) == 0)
			return 3;
		if (getTabVal3D(tab, i, j, k + 1, width, height, depth) == 0)
			return 3;
		if (getTabVal3D(tab, i - 1, j, k, width, height, depth) == 0)
			return 3;
		if (getTabVal3D(tab, i, j - 1, k, width, height, depth) == 0)
			return 3;
		if (getTabVal3D(tab, i + 1, j, k, width, height, depth) == 0)
			return 3;
		if (getTabVal3D(tab, i, j + 1, k, width, height, depth) == 0)
			return 3;

		if (getTabVal3D(tab, i - 1, j - 1, k, width, height, depth) == 0)
			return 4;
		if (getTabVal3D(tab, i - 1, j + 1, k, width, height, depth) == 0)
			return 4;
		if (getTabVal3D(tab, i + 1, j - 1, k, width, height, depth) == 0)
			return 4;
		if (getTabVal3D(tab, i + 1, j + 1, k, width, height, depth) == 0)
			return 4;

		if (getTabVal3D(tab, i - 1, j - 1, k - 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i - 1, j + 1, k - 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i + 1, j - 1, k - 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i + 1, j + 1, k - 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i - 1, j - 1, k + 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i - 1, j + 1, k + 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i + 1, j - 1, k + 1, width, height, depth) == 0)
			return 5;
		if (getTabVal3D(tab, i + 1, j + 1, k + 1, width, height, depth) == 0)
			return 5;
		return 0;
	}

	public static boolean samePoint3D(Point3D p1, Point3D p2) {
		if (p1.getX() == p2.getX() && p1.getY() == p2.getY() && p1.getZ() == p2.getZ())
			return true;
		return false;
	}

	public static double min(double d, double d2) {
		if (d <= d2)
			return d;
		else
			return d2;
	}

	public static boolean containsPixel(ArrayList<Point3D> list, Point3D p) {
		for (Point3D po : list) {
			if (samePoint3D(po, p)) {
				return true;
			}
		}
		return false;
	}

	public static void testDistance3D(Sequence seq) {
		int width = seq.getWidth();
		int height = seq.getHeight();
		int depth = seq.getSizeZ();
		int times = seq.getSizeT();
		int channels = seq.getSizeC();
		double[][] tab = new double[depth][width * height];
		double[][] tab_Dist = new double[depth][width * height];
		LinkedList<Point3D> pixelContour = new LinkedList<Point3D>();
		ArrayList<Point3D> pixelForme = new ArrayList<Point3D>();

		for (int t = 0; t < times; t++) {
			for (int c = 0; c < channels; c++) {
				tab = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, c), seq.isSignedDataType());
				for (int z = 0; z < depth; z++) {
					for (int j = 0; j < height; j++) {
						for (int i = 0; i < width; i++) {
							if (tab[z][i + j * width] != 0) {
								tab[z][i + j * width] = 2;
								tab_Dist[z][i + j * width] = Double.MAX_VALUE;
								int val = valPixel(tab, i, j, z, width, height, depth);
								if (val != 0) {
									tab_Dist[z][i + j * width] = val;
									pixelContour.add(new Point3D.Integer(i, j, z));
									tab[z][i + j * width] = 1;
								} else {
									pixelForme.add(new Point3D.Integer(i, j, z));
								}
							}

						}
					}
				}

				while (!pixelContour.isEmpty()) {
					Point3D pEnCours = pixelContour.poll();
					int i = (int) pEnCours.getX();
					int j = (int) pEnCours.getY();
					int k = (int) pEnCours.getZ();
					double d = tab_Dist[k][i + j * width];
					tab[k][i + j * width] = 1;
					// System.out.println("pixel en cours : " +pixelContour.get(index).toString());
					/*
					 * if (getTabVal3D(tab, i, j, k-1, width, height, depth)==2) { Point3D p = new
					 * Point3D.Integer(i, j, k-1); pixelContour.add(p); tab_Dist[k-1][i + j *
					 * width]= min (tab_Dist[k-1][i + j * width],d+3); } if (getTabVal3D(tab, i, j,
					 * k+1, width, height, depth)==2){ Point3D p = new Point3D.Integer(i, j, k+1);
					 * pixelContour.add(p); tab_Dist[k+1][i + j * width]= min (tab_Dist[k+1][i + j *
					 * width],d+3); }
					 */

					if (getTabVal3D(tab, i - 1, j, k, width, height, depth) == 2) {
						Point3D p = new Point3D.Integer(i - 1, j, k);
						pixelContour.add(p);
						tab_Dist[k][(i - 1) + j * width] = min(tab_Dist[k][(i - 1) + j * width], d + 3);
					}
					if (getTabVal3D(tab, i, j - 1, k, width, height, depth) == 2) {
						Point3D p = new Point3D.Integer(i, j - 1, k);
						pixelContour.add(p);
						tab_Dist[k][i + (j - 1) * width] = min(tab_Dist[k][i + (j - 1) * width], d + 3);

					}
					if (getTabVal3D(tab, i + 1, j, k, width, height, depth) == 2) {
						Point3D p = new Point3D.Integer(i + 1, j, k);
						pixelContour.add(p);
						tab_Dist[k][i + 1 + j * width] = min(tab_Dist[k][i + 1 + j * width], d + 3);

					}
					if (getTabVal3D(tab, i, j + 1, k, width, height, depth) == 2) {
						Point3D p = new Point3D.Integer(i, j + 1, k);
						pixelContour.add(p);
						tab_Dist[k][i + (j + 1) * width] = min(tab_Dist[k][i + (j + 1) * width], d + 3);
					}
					/*
					 * if (getTabVal3D(tab_Dist, i-1, j-1, k, width, height, depth)==2) {
					 * pixelContour.add(new Point3D.Integer(i-1, j-1, k)); tab_Dist[k][i-1 + (j-1) *
					 * width]= min (tab_Dist[k][i-1 + (j-1) * width],d+4); } if
					 * (getTabVal3D(tab_Dist, i-1, j+1, k, width, height, depth)==Double.MAX_VALUE)
					 * { pixelContour.add(new Point3D.Integer(i-1, j + 1, k)); tab_Dist[k][i-1 +
					 * (j+1) * width]= min (tab_Dist[k][i-1 + (j+1) * width],d+4); } if
					 * (getTabVal3D(tab_Dist, i+1, j-1, k, width, height, depth)==Double.MAX_VALUE)
					 * { pixelContour.add(new Point3D.Integer(i+1, j - 1, k)); tab_Dist[k][i+1 +
					 * (j-1) * width]= min (tab_Dist[k][i+1 + (j-1) * width],d+4); } if
					 * (getTabVal3D(tab_Dist, i+1, j+1, k, width, height, depth)==Double.MAX_VALUE)
					 * { pixelContour.add(new Point3D.Integer(i+1, j + 1, k)); tab_Dist[k][i+1 +
					 * (j+1) * width]= min (tab_Dist[k][i+1 + (j+1) * width],d+4); }
					 * 
					 * if (getTabVal3D(tab_Dist, i-1, j-1, k-1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i-1, j-1,
					 * k-1)); tab_Dist[k-1][i-1 + (j-1) * width]= min (tab_Dist[k-1][i-1 + (j-1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i-1, j+1, k-1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i-1, j + 1,
					 * k-1)); tab_Dist[k-1][i-1 + (j+1) * width]= min (tab_Dist[k-1][i-1 + (j+1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i+1, j-1, k-1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i+1, j - 1,
					 * k-1)); tab_Dist[k-1][i+1 + (j-1) * width]= min (tab_Dist[k-1][i+1 + (j-1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i+1, j+1, k-1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i+1, j + 1,
					 * k-1)); tab_Dist[k-1][i+1 + (j+1) * width]= min (tab_Dist[k-1][i+1 + (j+1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i-1, j-1, k+1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i-1, j-1,
					 * k+1)); tab_Dist[k+1][i-1 + (j-1) * width]= min (tab_Dist[k+1][i-1 + (j-1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i-1, j+1, k+1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i-1, j + 1,
					 * k+1)); tab_Dist[k+1][i-1 + (j+1) * width]= min (tab_Dist[k+1][i-1 + (j+1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i+1, j-1, k+1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i+1, j - 1,
					 * k+1)); tab_Dist[k+1][i+1 + (j-1) * width]= min (tab_Dist[k+1][i+1 + (j-1) *
					 * width],d+5); } if (getTabVal3D(tab_Dist, i+1, j+1, k+1, width, height,
					 * depth)==Double.MAX_VALUE) { pixelContour.add(new Point3D.Integer(i+1, j + 1,
					 * k+1)); tab_Dist[k+1][i+1 + (j+1) * width]= min (tab_Dist[k+1][i+1 + (j+1) *
					 * width],d+5); }
					 */

				}
				Array2DUtil.doubleArrayToArray(tab_Dist, seq.getDataXYZ(t, c));

			}
		}

	}

	// ---------------------------------------------------------------------------
	// Amincissement
	// ---------------------------------------------------------------------------
	public static IcyBufferedImage thinning(Sequence seq, int nmin) {
		int i, j;
		double n = 0;
		int nr = seq.getHeight(), nc = seq.getWidth();

		IcyBufferedImage img = seq.getFirstImage();
		double[] result = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

		seq.setName(seq.getName() + "_thin" + nmin);

		/*
		 * MArray resultDistance = getDistance(myArray, 8); VMY distance ?? nmin = 2;
		 * dmin = 2 29/07/04
		 */

		Sequence seqD = getDistance(seq, 8);
		IcyBufferedImage imgDist = seqD.getFirstImage();

		/*
		 * distanceMap2D(seqDist, z, threshold, above);
		 * 
		 * IcyBufferedImage imgDist = seqDist.getFirstImage(); double[] resultDistance =
		 * Array1DUtil.arrayToDoubleArray(imgDist.getDataXY(0),
		 * imgDist.isSignedDataType());;
		 */

		double[] resultDistance = Array1DUtil.arrayToDoubleArray(imgDist.getDataXY(0), imgDist.isSignedDataType());

		for (j = 0; j < nr; j++)
			for (i = 0; i < nc; i++) {
				n = getDist(resultDistance, i, j, nc, nr);
				if (n > 0) {
					if (n < nmin) {
						if (swip_voisins(nmin, resultDistance, nc, nr, i, j)) {

							result[i + nc * j] = CH3;
						} else {
							result[i + nc * j] = CH0;
							// System.out.println("CH0 i:"+i+" j:" +j+" n:"+ n);
						}
					} else {
						result[i + nc * j] = CH1;
						// System.out.println("CH3 i:"+i+" j:" +j+" n:"+ n);
					}
				}
			}

		for (j = 0; j < nr; j++)
			for (i = 0; i < nc; i++) {
				if (result[i + j * nc] >= CH1)
					result[i + nc * j] = 255;
				else
					result[i + nc * j] = 0;
			}
		img.setDataXY(0, Array1DUtil.doubleArrayToArray(result, img.getDataXY(0)));

		return img;
	}

	final static int MAXVAL = 37265;

	// TODO Distance 4 ou 8 connexites
	/**
	 * Return the distance map MArray of the input MArray. 2 methods, 4 or 8
	 * connexity method. connex = 4 for 4 connexity, 8 for 8 connexity.
	 * 
	 * @param MArray myArray
	 * @param int    method
	 * @return
	 */
	public static Sequence getDistance(Sequence seq, int connex) {
		IcyBufferedImage imgIn = seq.getFirstImage();
		int nr = seq.getHeight(), nc = seq.getWidth(), k = nc * nr;
		IcyBufferedImage imgOut = new IcyBufferedImage(nc, nr, seq.getSizeC(), DataType.DOUBLE);
		double[] tabIn = Array1DUtil.arrayToDoubleArray(imgIn.getDataXY(0), imgIn.isSignedDataType());

		int black = 0;
		int[] d = new int[8];

		double[] tab = new double[k];

		switch (connex) {
		case 4:

			d[0] = 1;
			d[1] = nc;
			d[2] = -1;
			d[3] = -nc;
			break;
		case 8: // 8 or 4 connexity method

			d[0] = 1;
			d[1] = nc;
			d[2] = -1;
			d[3] = -nc;
			d[4] = 1 + nc;
			d[5] = -1 + nc;
			d[6] = -1 - nc;
			d[7] = 1 - nc;

			// neighborhood pixels
			/*
			 * d[0] = 1; d[1] = 1-nc; d[2] = -nc; d[3] = -1-nc; d[4] = -1; d[5] = -1+nc;
			 * d[6] = nc ; d[7] = 1 + nc;
			 */
			break;
		}

		// Initializing result image
		for (int i = 0; i < k; i++) {
			if (tabIn[i] != black) {
				// result.set(i, 255);
				tab[i] = MAXVAL;
			} else {
				// result.set(i, 0);
				tab[i] = 0;
			}
		}

		/*
		 * // Set borders to 0 for (int i = 0; i < nc; i++) { // result.set(i, 0); //
		 * result.set(i + k - nc, 0); tab[i] = 0; tab[i + k - nc] = 0; } for (int i = 0;
		 * i < k; i += nc) { // result.set(i, 0); // result.set(i + nc - 1, 0); tab[i] =
		 * 0; tab[i + nc - 1] = 0; }
		 */

		LinkedList<PixWatershed> fifo = new LinkedList<PixWatershed>();

		// find border pixels
		for (int i = 0; i < k; i++) {
			int j = 0;
			int flag = 0;
			if (tab[i] == MAXVAL) {
				while ((flag == 0) && (j < connex)) {
					if ((i) % nc == 0 || (i) % nc == (nc - 1) || (i + d[j] < 0) || i + d[j] > (nc * nr))
						tab[i] = 1;
					if ((i + d[j]) >= 0 && (i + d[j]) < k && tab[i + d[j]] == black) {
						PixWatershed p_num = new PixWatershed(i, nc); // A
																		// VERIFIER
																		// !!!
						fifo.add(p_num);
						// result.set(i, 1);
						tab[i] = 1;
						flag = 1;
					}
					j++;
				}
			}
		}

		int maxd = -1;
		while (!fifo.isEmpty()) { // inner layers
			PixWatershed p_num = fifo.poll();
			int i = p_num.getOffset();
			for (int j = 0; j < connex; j++) {
				int l = i + d[j];
				if (l >= 0 && l < k) {
					if (tab[l] == MAXVAL) {
						tab[l] = tab[i] + 1;
						// ATTENTION creer un nouveau pixel, ne pas travailler
						// sur le meme
						PixWatershed p = new PixWatershed(l, nc);
						maxd = Math.max((int) tab[l], maxd);
						fifo.add(p);
					}
				} else {
					// System.out.println(" l: " + l);
				}
			}
		}

		// TODO Copie vers IcyBuffer
		/*
		 * for (int i = 0; i < k; i++) { result.set(i, tab[i]); }
		 */

		// imgOut.setDataXY(0, Array1DUtil.doubleArrayToArray(tab,
		// imgOut.getDataXY(0), false));
		imgOut.setDataXY(0, Array1DUtil.doubleArrayToArray(tab, imgOut.getDataXY(0)));

		/*
		 * int depth = seq.getSizeZ(); seq.beginUpdate(); for (int z = 0; z < depth;
		 * z++) { IcyBufferedImage img = seq.getImage(0, z); img.setDataXY(0,
		 * Array1DUtil.doubleArrayToArray(dataOutDouble[z], img.getDataXY(0))); }
		 * seq.endUpdate();
		 */
		// return imgOut;
		Sequence seqOut = new Sequence();
		seqOut.addImage(imgOut);
		return (seqOut);
	}

	// ATTENTION: white et bords image
	// A comparer avec getDistance
	public static Sequence computeDistance(Sequence seq, int connex) {
		int nr = seq.getHeight(), nc = seq.getWidth(), k = nc * nr;
		IcyBufferedImage imgOut = new IcyBufferedImage(nc, nr, seq.getSizeC(), DataType.DOUBLE);
		// double[] tabIn = Array1DUtil.arrayToDoubleArray(imgIn.getDataXY(0),
		// imgIn.isSignedDataType());

		int black = 0;
		int white = 255;

		int[] d = new int[8];
		IcyBufferedImage img = seq.getFirstImage(); // TODO t,z
		double[] tab = Array1DUtil.arrayToDoubleArray(img.getDataXY(0), img.isSignedDataType());

		switch (connex) {
		case 4:
			d = new int[4];
			d[0] = -nc;
			d[1] = 1;
			d[2] = nc;
			d[3] = -1;
			break;
		case 8: // 8 or 4 connexity method
			// neighborhood pixels
			d[0] = -1 - nc;
			d[1] = nc;
			d[2] = nc + 1;
			d[3] = -1;
			d[4] = 1;
			d[5] = -1 + nc;
			d[6] = nc + 1;
			d[7] = 1 + nc;
			break;

		}
		LinkedList<PixWatershed> fifo = new LinkedList<PixWatershed>();

		// find border pixels
		for (int i = 0; i < k; i++) {
			int j = 0;
			int flag = 0;
			if (tab[i] != black) {
				while ((flag == 0) && (j < connex)) {
					if ((i + d[j]) < k && ((i + d[j]) >= 0) && tab[i + d[j]] == black) {
						PixWatershed p_num = new PixWatershed(i, nc); // A
																		// VERIFIER
																		// !!!
						fifo.add(p_num);
						tab[i] = CH2;
						flag = 1;

					}
					j++;
				}
			}
		}

		int maxd = -1;
		while (!fifo.isEmpty()) { // inner layers
			PixWatershed p_num = fifo.poll();
			int i = p_num.getOffset();
			for (int j = 0; j < connex; j++) {
				int l = i + d[j];
				if (l >= 0 && l < k) {
					if (tab[l] == white) {
						tab[l] = tab[i] + 1;
						// ATTENTION creer un nouveau pixel, ne pas travailler
						// sur le meme
						PixWatershed p = new PixWatershed(l, nc);
						maxd = Math.max((int) tab[l], maxd);
						fifo.add(p);
					}
				} else {
					System.out.println(" l: " + l);
				}
			}

		}
		imgOut.setDataXY(0, Array1DUtil.doubleArrayToArray(tab, imgOut.getDataXY(0)));

		Sequence seqOut = new Sequence();
		seqOut.addImage(imgOut);
		return (seqOut);
	}

	// control la valeur de coordoonnees
	static double getDist(double[] Tab, int x, int y, int nc, int nr) {
		if (x < 0 || x >= nc || y < 0 || y >= nr)
			return (0);
		else
			return (Tab[x + nc * y]);
	}

	static boolean swip_voisins(int val, double[] resultDist, int nc, int nr, int i, int j) {
		boolean find;
		int k;
		int d[] = new int[12];

		// neighborhood pixels
		d[0] = -nc;
		d[1] = 1;
		d[2] = nc;
		d[3] = -1;
		d[4] = 1 - nc;
		d[5] = 1 + nc;
		d[6] = nc - 1;
		d[7] = -1 - nc;
		d[8] = -2 * nc;
		d[9] = 2;
		d[10] = 2 * nc;
		d[11] = -2;

		k = 0;
		find = false;

		do {
			// System.out.println("i:"+i+" j:"+j+" j * nc + i + d[k]:"+(j * nc +
			// i + d[k++])+ " k:"+k+" d[k]:"+ d[k]);
			// TODO: o� est le probleme ???
			if ((j * nc + i + d[k]) < nc * nr && (j * nc + i + d[k]) > 0)
				if (resultDist[j * nc + i + d[k]] >= val)
					find = true;
			k++;
		} while ((!find) && (k < 12));
		return (find);
	}

	// ---------------------------------------------------------------------------
	// Watershed
	// ---------------------------------------------------------------------------

	final static private int MASK = -2; /*
										 * valeur initialise affectee a chaque niveau
										 */
	final static private int WSHED = 0; /* valeur finale des pixels de la LPE */
	final static private int INIT = -1; /* valeur initiale des pixels de J */
	final static private int INQUEUE = -3;
	final static private int BORDER = -4;

	// The sequence is a distance map
	private static void calculerLPE(Sequence seq, int z, int cv) {
		// Image dimensions
		int nc = seq.getFirstImage().getWidth();
		int nr = seq.getFirstImage().getHeight();
		// offset max
		int nbPixels = nc * nr;

		int[] d = new int[8];
		d[0] = -nc;
		d[1] = 1;
		d[2] = nc;
		d[3] = -1;
		d[4] = 1 - nc;
		d[5] = 1 + nc;
		d[6] = nc - 1;
		d[7] = -1 - nc;

		Sequence seqOut = new Sequence();

		for (int t = 0; t < seq.getSizeT(); t++) {
			IcyBufferedImage imgIn = seq.getImage(t, z);
			int[] tabIn = Array1DUtil.arrayToIntArray(imgIn.getDataXY(0), imgIn.isSignedDataType());

			IcyBufferedImage imgOut = new IcyBufferedImage(seq.getSizeX(), seq.getSizeY(), 1, seq.getDataType_());
			double[] tabOut = Array1DUtil.arrayToDoubleArray(imgOut.getDataXY(0), imgOut.isSignedDataType());

			// bord de l'image
			for (int i = 0; i < nc; i++) {
				tabOut[i] = BORDER;
				tabOut[i + nbPixels - nc] = BORDER;
			}
			for (int i = 0; i < nbPixels; i += +nc) {
				tabOut[i] = BORDER;
				tabOut[i + nc - 1] = BORDER;
			}

			// init pixels 'queue'
			LinkedList<PixWatershed> fifo = new LinkedList<PixWatershed>();

			int flag = 0;
			int cr_label = 0;

			// begin the tri-distributive function
			int hmin = Short.MAX_VALUE;
			int hmax = 0;
			for (int i = 0; i < nbPixels; i++) {
				int val = ((int) tabIn[i]) & 0xFFFFFF;
				hmin = Math.min(val, hmin);
				hmax = Math.max(val, hmax) & 0xFF;
			}
			// Histogram
			int[] Hist = new int[hmax + 1];
			// Cumulated histogram
			int[] HistCum = new int[hmax + 1];
			;
			// init histogram
			for (int i = 0; i < hmax; i++) {
				Hist[i] = 0;
				HistCum[i] = 0;
			}
			// Fill histogram
			for (int i = 0; i < nbPixels; i++) {
				Hist[((int) tabIn[i]) & 0xFFFFFF]++;
			}
			for (int i = hmin + 1; i <= hmax; i++) {
				HistCum[i] = HistCum[i - 1] + Hist[i - 1];
			}

			int[] ims = new int[nc * nr];
			for (int i = 1; i < nc - 1; i++)
				for (int j = 1; j < nr - 1; j++) {
					int l = i + j * nc;
					ims[HistCum[tabIn[l]]] = l;
					HistCum[tabIn[l]] += 1;
				}
			// end of the tri-distributive function
			int lower_b;
			for (int i = hmin; i <= hmax; i++) {
				if (i != 0)
					lower_b = HistCum[i - 1];
				else
					lower_b = 0;

				int l;
				for (int j = lower_b; j < HistCum[i]; j++) {
					l = ims[j];
					tabOut[l] = MASK;

					for (int r = 0; r < cv; r++) {
						int pos = l + d[r];
						PixWatershed pixel = new PixWatershed(pos, nc);
						if (pos < nbPixels && pos >= 0)
							if ((tabOut[pos]) > 0 || (tabOut[pos] == WSHED)) {
								fifo.add(pixel);
								tabOut[l] = INQUEUE;
							}

					}

				}

				while (!fifo.isEmpty()) {
					PixWatershed pixel = fifo.pollFirst();
					l = pixel.getOffset();
					int pos = 0;
					for (int r = 0; r < cv; r++) {
						pos = l + d[r];
						if (pos >= 0 && pos < nbPixels) {
							if (tabOut[pos] > 0) {
								if ((tabOut[l] == INQUEUE) || (tabOut[l] == WSHED && flag != 0))
									tabOut[l] = tabOut[pos];
								else if ((tabOut[l] > 0) && (tabOut[l] != tabOut[pos])) {
									tabOut[l] = WSHED;
									flag = 0;
								}
							} else if (tabOut[pos] == WSHED) {
								if (tabOut[l] == INQUEUE) {
									tabOut[l] = WSHED;
									flag = 1;

								}
							} else if (tabOut[pos] == MASK) {
								tabOut[pos] = INQUEUE;
								PixWatershed pixel2 = new PixWatershed(pos, nc);
								fifo.add(pixel2);
							}
						}
					}
				}
				for (int j = lower_b; j < HistCum[i]; j++) {
					l = ims[j];
					if (tabOut[l] == MASK) {
						tabOut[l] = ++cr_label;
						PixWatershed pixel = new PixWatershed(l, nc);
						fifo.add(pixel);
					}
					while (!fifo.isEmpty()) {
						PixWatershed pixel = fifo.pollFirst();
						int pos = pixel.getOffset();

						for (int r = 0; r < cv; r++) {
							int lpos = pos + d[r];
							if (lpos >= 0 && lpos < nbPixels)
								if (tabOut[lpos] == MASK) {
									PixWatershed pixel2 = new PixWatershed(lpos, nc);
									fifo.add(pixel2);
									tabOut[lpos] = cr_label;
								}
						}
					}

				}
			}
			imgOut.setDataXY(0, Array1DUtil.doubleArrayToArray(tabOut, imgOut.getDataXY(0)));
			seqOut.setImage(t, z, imgOut);
		}

		seq.removeAllImages();
		for (int t = 0; t < seqOut.getSizeT(); t++) {
			seq.setImage(t, 0, seqOut.getImage(t, 0));
		}

	}

	// Calcule la carte des bassins et versants
	private static void calculerVersantsEtiquetes(Sequence seq, int z) {
		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();
		// offset max
		int maxPix = width * height;

		Sequence seqOut = new Sequence();

		for (int t = 0; t < seq.getSizeT(); t++) {
			/*
			 * Initialisation, sorting pixels
			 */
			IcyBufferedImage imgIn = seq.getImage(t, z);
			// Temporary image which will contain the data to be put in the
			// original image
			IcyBufferedImage imgOut;
			imgOut = new IcyBufferedImage(imgIn.getWidth(), imgIn.getHeight(), imgIn.getSizeC(), DataType.DOUBLE);
			/*
			 * DataBufferDouble bufOut = (DataBufferDouble) imgOut.getRaster()
			 * .getDataBuffer(); double[] tJ = bufOut.getData();
			 */

			double[] tJ = Array1DUtil.arrayToDoubleArray(imgOut.getDataXY(0), imgOut.isSignedDataType());
			// image de travail des distances
			int[] Id = new int[maxPix];

			// Pixel fictif
			PixWatershed fictif = new PixWatershed(-1, -1, -1);

			// Sorted pixels
			PixWatershed[] pixels = new PixWatershed[maxPix];

			// Max and min pixel values
			int hmin = 65535, hmax = 0;
			// Histogram
			int[] Hi;
			// Cumulated histogram
			int[] HCi;

			LinkedList<PixWatershed> fifo = new LinkedList<PixWatershed>();
			int dist_courante;
			int label_courant = 0;

			double[] tabIn = Array1DUtil.arrayToDoubleArray(imgIn.getDataXY(0), imgIn.isSignedDataType());

			for (int i = 0; i < maxPix; i++) {
				int val = ((int) tabIn[i]);// & 0xFFFFFF;
				hmin = Math.min(val, hmin);
				hmax = Math.max(val, hmax);
			}

			// Init histograms
			Hi = new int[hmax + 1];
			for (int i = hmin; i <= hmax; i++) {
				Hi[i] = 0;
			}
			HCi = new int[hmax + 1];
			HCi[hmin] = 0;

			// Fill histogram
			for (int i = 0; i < maxPix; i++) {
				Hi[((int) tabIn[i])]++;// & 0xFFFFFF]++;
			}

			// Cumulated histogram
			for (int h = hmin + 1; h <= hmax; h++) {
				HCi[h] = (HCi[h - 1] + Hi[h - 1]);
				System.out.println(" h:" + h + " Hi[h-1]:" + Hi[h - 1] + " HCi[h - 1]:" + HCi[h - 1]);
			}
			// Sort
			for (int offset = 0; offset < maxPix; offset++) { // TODO
				int i = HCi[(int) tabIn[offset]];
				pixels[i] = new PixWatershed(offset, width);// & 0xFFFFFF]
				HCi[(int) tabIn[offset]]++;// & 0xFFFFFF]++;
			}

			/*
			 * PART 2
			 */

			// init J
			for (int i = 0; i < tJ.length; i++) {
				tJ[i] = INIT;
			}

			for (int h = hmin; h <= hmax; h++) {
				int start;
				int end;
				if (h == hmin)
					start = 0;
				else
					start = HCi[h - 1];
				end = HCi[h];

				// Pour tout p tq I(p) = h
				for (int i = start; i < end; i++) {

					int offset = pixels[i].getOffset();
					if (offset == 5074)
						System.out.println(" i:" + i + " offset:" + offset + " x:" + pixels[offset].getX() + " y:"
								+ pixels[offset].getY());
					tJ[offset] = MASK;
					int x = pixels[i].getX();
					int y = pixels[i].getY();
					/*
					 * S'il existe p' appartenant a NG(p) tqJ(p')>0 ou J(p')=WSHED
					 */

					ArrayList<Integer> lNeighbors = getOffsetNeighbors(x, y, width, height);
					for (int ind = 0; ind < lNeighbors.size(); ind++) {
						int offneighbor = lNeighbors.get(ind);

						if ((tJ[offneighbor]) > 0 || (tJ[offneighbor]) == WSHED) {
							Id[offset] = 1;
							fifo.add(pixels[i]);
							// break;
						}
					}
				}

				dist_courante = 1;
				fifo.add(fictif);
				// Boucle infinie
				while (true) {
					PixWatershed p = fifo.poll();
					if (p.equals(fictif)) {
						if (fifo.isEmpty())
							break;
						else {
							fifo.add(fictif);
							dist_courante++;
							p = fifo.poll();
						}
					}
					/*
					 * Pour tout pixel p' appartenant a NG(p)
					 */
					int x = p.getX();
					int y = p.getY();
					int offset = p.getOffset();
					ArrayList<Integer> lNeighBors = getOffsetNeighbors(x, y, width, height);
					for (int i = 0; i < lNeighBors.size(); i++) {
						int offneighbor = lNeighBors.get(i);

						if (Id[offneighbor] < dist_courante && (tJ[offneighbor] > 0 || tJ[offneighbor] == WSHED)) {
							/*
							 * p' fait partie d'un bassin deja etiquete ou de la LPE
							 */
							if (tJ[offneighbor] > 0) {
								if (tJ[offset] == MASK || tJ[offset] == WSHED)
									tJ[offset] = tJ[offneighbor];
								else if (tJ[offset] != tJ[offneighbor])
									tJ[offset] = WSHED;
							} else if (tJ[offset] == MASK)
								tJ[offset] = WSHED;
						} else if (tJ[offneighbor] == MASK && Id[offneighbor] == 0) {
							Id[offneighbor] = dist_courante + 1;
							fifo.add(new PixWatershed(offneighbor, width));
						}
					}
				} // fin boucle infinie
				/* De nouveaux minima sont-ils apparus? */
				for (int i = start; i < end; i++) {
					int offset = pixels[i].getOffset();
					if (tJ[offset] == MASK) {
						label_courant++;
						if (label_courant == 45) {
							System.out.println(" STOP");
						}
						fifo.add(pixels[i]);
						tJ[offset] = label_courant;
						while (!fifo.isEmpty()) {
							PixWatershed p2 = fifo.poll();
							// Pour tout pixel p" appartenant a NG(p')
							ArrayList<Integer> lNeighbors = getOffsetNeighbors(p2.getX(), p2.getY(), width, height);
							for (int ind = 0; ind < lNeighbors.size(); ind++) {
								int offneighbor = lNeighbors.get(ind);
								if (tJ[offneighbor] == MASK) {
									fifo.add(new PixWatershed(offneighbor, width));
									tJ[offneighbor] = label_courant;
								}
							}
						}
					}
				}

			}
			imgOut.setDataXY(0, Array1DUtil.doubleArrayToArray(tJ, imgOut.getDataXY(0)));
			// imgOut.setDataXY(0, Array1DUtil.intArrayToArray(Id, imgOut.getDataXY(0), true
			// ));
			seqOut.setImage(t, z, imgOut);

		}

		// Remove the images from the input sequence and put the new one : may
		// be a different datatype
		seq.removeAllImages();
		for (int t = 0; t < seqOut.getSizeT(); t++) {
			seq.setImage(t, 0, seqOut.getImage(t, 0));
		}
	}

	private static void calculerVersantsEtiquetes3D(Sequence seq) {
		// Image dimensions
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();
		// offset max
		int maxPix = width * height;

		Sequence seqOut = new Sequence();

		for (int t = 0; t < seq.getSizeT(); t++) {
			int depth = seq.getSizeZ(t);
			/*
			 * Initialisation, sorting pixels
			 */

			double[][] tJ = new double[depth][maxPix];
			// image de travail des distances
			int[][] Id = new int[depth][maxPix];

			// Pixel fictif
			PixWatershed3D fictif = new PixWatershed3D(-1, -1, -1);

			// Sorted pixels
			PixWatershed3D[] pixels = new PixWatershed3D[maxPix * depth];

			// Max and min pixel values
			int hmin = 65535, hmax = 0;
			// Histogram
			int[] Hi;
			// Cumulated histogram
			int[] HCi;

			LinkedList<PixWatershed3D> fifo = new LinkedList<PixWatershed3D>();
			int dist_courante;
			int label_courant = 0;

			double[][] tabIn = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, 0), seq.isSignedDataType());

			for (int z = 0; z < depth; z++) {
				for (int i = 0; i < maxPix; i++) {
					int val = (int) tabIn[z][i] & 0xFFFFFF;
					hmin = Math.min(val, hmin);
					hmax = Math.max(val, hmax) & 0xFFFFFF;
				}
			}

			// Init histograms
			Hi = new int[hmax + 1];
			for (int i = hmin; i <= hmax; i++) {
				Hi[i] = 0;
			}
			HCi = new int[hmax + 1];
			for (int z = 0; z < depth; z++)
				HCi[hmin] = 0;

			// Fill histogram
			for (int z = 0; z < depth; z++)
				for (int i = 0; i < maxPix; i++) {
					Hi[(int) tabIn[z][i] & 0xFFFFFF]++;
				}

			// Cumulated histogram

			for (int h = hmin + 1; h <= hmax; h++) {
				HCi[h] = (HCi[h - 1] + Hi[h - 1]);
			}
			// Sort
			for (int z = 0; z < depth; z++)
				for (int offset = 0; offset < maxPix; offset++) {
					pixels[HCi[(int) tabIn[z][offset] & 0xFFFFFF]] = new PixWatershed3D(offset, z, width);
					HCi[(int) tabIn[z][offset] & 0xFFFFFF]++;
				}

			/*
			 * PART 2
			 */

			// init J
			for (int z = 0; z < depth; z++)
				for (int i = 0; i < tJ[0].length; i++) {
					tJ[z][i] = INIT;
				}

			for (int h = hmin; h <= hmax; h++) {
				int start;
				int end;

				if (h == hmin)
					start = 0;
				else
					start = HCi[h - 1];
				end = HCi[h];

				// Pour tout p tq I(p) = h
				for (int i = start; i < end; i++) {

					int offset = pixels[i].getOffset();
					int x = pixels[i].getX();
					int y = pixels[i].getY();
					int z = pixels[i].getZ();
					tJ[z][offset] = MASK;

					/*
					 * S'il existe p' appartenant a NG(p) tqJ(p')>0 ou J(p')=WSHED
					 */
					ArrayList<int[]> lNeighbors = getOffsetNeighbors3D(x, y, z, width, height, depth);
					for (int ind = 0; ind < lNeighbors.size(); ind++) {
						int zNeighbor = lNeighbors.get(ind)[0];
						int offneighbor = lNeighbors.get(ind)[1];
						if ((tJ[zNeighbor][offneighbor]) > 0 || (tJ[zNeighbor][offneighbor]) == WSHED) {
							Id[z][offset] = 1;
							fifo.add(pixels[i]);
							break;
						}
					}
				}

				dist_courante = 1;
				fifo.add(fictif);
				// Boucle infinie
				while (true) {
					PixWatershed3D p = fifo.poll();
					if (p.equals(fictif)) {
						if (fifo.isEmpty())
							break;
						else {
							fifo.add(fictif);
							dist_courante++;
							p = fifo.poll();
						}
					}
					/*
					 * Pour tout pixel p' appartenant a NG(p)
					 */
					int x = p.getX();
					int y = p.getY();
					int z = p.getZ();
					int offset = p.getOffset();
					ArrayList<int[]> lNeighBors = getOffsetNeighbors3D(x, y, z, width, height, depth);
					for (int i = 0; i < lNeighBors.size(); i++) {
						int zNeighbor = lNeighBors.get(i)[0];
						int offneighbor = lNeighBors.get(i)[1];
						if (Id[zNeighbor][offneighbor] < dist_courante
								&& (tJ[zNeighbor][offneighbor] > 0 || tJ[zNeighbor][offneighbor] == WSHED)) {
							/*
							 * p' fait partie d'un bassin deja etiquete ou de la LPE
							 */
							if (tJ[zNeighbor][offneighbor] > 0) {
								if (tJ[z][offset] == MASK || tJ[z][offset] == WSHED)
									tJ[z][offset] = tJ[zNeighbor][offneighbor];
								else if (tJ[z][offset] != tJ[zNeighbor][offneighbor])
									tJ[z][offset] = WSHED;
							} else if (tJ[z][offset] == MASK)
								tJ[z][offset] = WSHED;
						} else if (tJ[zNeighbor][offneighbor] == MASK && Id[zNeighbor][offneighbor] == 0) {
							Id[zNeighbor][offneighbor] = dist_courante + 1;
							fifo.add(new PixWatershed3D(offneighbor, zNeighbor, width));
						}
					}
				} // fin boucle infinie
				/* De nouveaux minima sont-ils apparus? */
				for (int i = start; i < end; i++) {
					int offset = pixels[i].getOffset();
					int z = pixels[i].getZ();
					if (tJ[z][offset] == MASK) {
						label_courant++;
						fifo.add(pixels[i]);
						tJ[z][offset] = label_courant;
						while (!fifo.isEmpty()) {
							PixWatershed3D p2 = fifo.poll();
							// Pour tout pixel p" appartenant à NG(p')
							ArrayList<int[]> lNeighbors = getOffsetNeighbors3D(p2.getX(), p2.getY(), p2.getZ(), width,
									height, depth);

							for (int ind = 0; ind < lNeighbors.size(); ind++) {
								int zNeighbor = lNeighbors.get(ind)[0];
								int offneighbor = lNeighbors.get(ind)[1];

								if (tJ[zNeighbor][offneighbor] == MASK) {
									fifo.add(new PixWatershed3D(offneighbor, z, width));
									tJ[zNeighbor][offneighbor] = label_courant;
								}
							}
						}
					}
				}
			}

			for (int z = 0; z < depth; z++) {
				IcyBufferedImage imgOut = new IcyBufferedImage(width, height, 1, DataType.DOUBLE);
				DataBufferDouble bufOut = (DataBufferDouble) imgOut.getRaster().getDataBuffer();
				double[] dataOut = bufOut.getData();
				for (int offset = 0; offset < maxPix; offset++) {
					dataOut[offset] = tJ[z][offset];
				}
				seqOut.setImage(t, z, imgOut);
			}
		}

		// Remove the images from the input sequence and put the new one : may
		// be a different datatype
		seq.removeAllImages();
		for (int t = 0; t < seqOut.getSizeT(); t++) {
			for (int z = 0; z < seqOut.getSizeZ(t); z++) {
				seq.setImage(t, z, seqOut.getImage(t, z));
			}
		}
	}

	// Separe les versants
	private static void separerVersants(Sequence seq, int z) {
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();
		int maxPix = width * height;

		for (int t = 0; t < seq.getSizeT(); t++) {
			IcyBufferedImage img = seq.getImage(t, z);
			DataBufferDouble buf = (DataBufferDouble) img.getRaster().getDataBuffer();
			double[] tabVersants = buf.getData();

			for (int offset = 0; offset < maxPix; offset++) {

				ArrayList<Integer> lNeighbors = getOffsetNeighbors(offset, width, height);
				for (int i = 0; i < lNeighbors.size(); i++) {
					int offNeighbor = lNeighbors.get(i);
					if (tabVersants[offNeighbor] < tabVersants[offset] && tabVersants[offNeighbor] > 0) {
						tabVersants[offset] = WSHED;
						break;
					}
				}
			}
		}

	}

	private static void separerVersants3D(Sequence seq) {
		int width = seq.getFirstImage().getWidth();
		int height = seq.getFirstImage().getHeight();
		int maxPix = width * height;

		for (int t = 0; t < seq.getSizeT(); t++) {
			int depth = seq.getSizeZ(t);

			double[][] tabVersants = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, 0), seq.isSignedDataType());

			for (int z = 0; z < depth; z++)
				for (int offset = 0; offset < maxPix; offset++) {

					ArrayList<int[]> lNeighbors = getOffsetNeighbors3D(offset, z, width, height, depth);
					;
					for (int i = 0; i < lNeighbors.size(); i++) {
						int zNeighbor = lNeighbors.get(i)[0];
						int offNeighbor = lNeighbors.get(i)[1];
						if (tabVersants[zNeighbor][offNeighbor] < tabVersants[z][offset]
								&& tabVersants[zNeighbor][offNeighbor] > 0) {
							tabVersants[z][offset] = WSHED;
							break;
						}
					}
				}
		}
	}

	// Returns an arraylist containing the offsets of the neighbors of a pixel
	// according
	// to it's own offset and the image dimensions
	private static ArrayList<Integer> getOffsetNeighbors(int offset, int width, int height) {
		int x = offset % width;
		int y = offset / width;
		return getOffsetNeighbors(x, y, width, height);
	}

	private static ArrayList<int[]> getOffsetNeighbors3D(int offset, int z, int width, int height, int depth) {
		int x = offset % width;
		int y = offset / width;
		return getOffsetNeighbors3D(x, y, z, width, height, depth);
	}

	// Returns an arraylist containing the offsets of the neighbors of a pixel
	// according
	// to it's coordinates (x,y) and the image dimensions
	// Returns an array containing the offsets of the neighbors of a pixel in
	// x,y
	private static ArrayList<Integer> getOffsetNeighbors(int x, int y, int width, int height) {
		ArrayList<Integer> listNeighbors = new ArrayList<Integer>();

		// topleft neighbor
		if (x - 1 >= 0 && y - 1 >= 0) {
			int offneighbor = x - 1 + (y - 1) * width;
			listNeighbors.add(offneighbor);
		}
		// top neighbor
		if (y - 1 >= 0) {
			int offneighbor = x + (y - 1) * width;
			listNeighbors.add(offneighbor);

		}
		// topright neighbor
		if (y - 1 >= 0 && x + 1 < width) {
			int offneighbor = x + 1 + (y - 1) * width;
			listNeighbors.add(offneighbor);
		}
		// left neighbor
		if (x - 1 >= 0) {
			int offneighbor = x - 1 + y * width;
			listNeighbors.add(offneighbor);
		}
		// right neighbor
		if (x + 1 < width) {
			int offneighbor = x + 1 + y * width;
			listNeighbors.add(offneighbor);
		}
		// botleft neighbor
		if (x - 1 >= 0 && y + 1 < height) {
			int offneighbor = x - 1 + (y + 1) * width;
			listNeighbors.add(offneighbor);

		}
		// bot neighbor
		if (y + 1 < height) {
			int offneighbor = x + (y + 1) * width;
			listNeighbors.add(offneighbor);

		}
		// botright neighbor
		if (x + 1 < width && y + 1 < height) {
			int offneighbor = x + 1 + (y + 1) * width;
			listNeighbors.add(offneighbor);
		}
		return listNeighbors;
	}

	private static ArrayList<int[]> getOffsetNeighbors3D(int x, int y, int z, int width, int height, int depth) {
		ArrayList<int[]> listNeighbors = new ArrayList<int[]>();
		// int maxPix = width*height;

		// topleft neighbor
		if (x - 1 >= 0 && y - 1 >= 0) {
			int offneighbor = x - 1 + (y - 1) * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });
		}
		// top neighbor
		if (y - 1 >= 0) {
			int offneighbor = x + (y - 1) * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });

		}
		// topright neighbor
		if (y - 1 >= 0 && x + 1 < width) {
			int offneighbor = x + 1 + (y - 1) * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });
		}
		// left neighbor
		if (x - 1 >= 0) {
			int offneighbor = x - 1 + y * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });
		}
		// right neighbor
		if (x + 1 < width) {
			int offneighbor = x + 1 + y * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });
		}
		// botleft neighbor
		if (x - 1 >= 0 && y + 1 < height) {
			int offneighbor = x - 1 + (y + 1) * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });

		}
		// bot neighbor
		if (y + 1 < height) {
			int offneighbor = x + (y + 1) * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });

		}
		// botright neighbor
		if (x + 1 < width && y + 1 < height) {
			int offneighbor = x + 1 + (y + 1) * width;
			// if (offneighbor<maxPix)
			listNeighbors.add(new int[] { z, offneighbor });
			if (z - 1 >= 0)
				listNeighbors.add(new int[] { z - 1, offneighbor });
			if (z + 1 < depth)
				listNeighbors.add(new int[] { z + 1, offneighbor });
		}
		// /*
		// * Z-1 stack neighbors
		// */
		// //topleft neighbor
		// if (x-1>=0 && y-1>=0 && z-1>=0){
		// int offneighbor = x-1+ (y-1)*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});
		// }
		// //top neighbor
		// if (y-1 >=0 && z-1>=0){
		// int offneighbor = x+ (y-1)*width - maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});

		// }
		// //topright neighbor
		// if (y-1>=0 && x+1<width && z-1>=0){
		// int offneighbor= x+1 + (y-1)*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});
		// }
		// //left neighbor
		// if (x-1>=0 && z-1>=0){
		// int offneighbor = x-1+y*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});
		// }
		// //mid neighbor
		// if (z-1>=0 && z-1>=0){
		// int offneighbor = x+y*width-maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});

		// }
		// //right neighbor
		// if (x+1<width && z-1>=0){
		// int offneighbor= x+1+y*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});
		// }
		// //botleft neighbor
		// if (x-1>=0 && y+1<height && z-1>=0){
		// int offneighbor = x-1 + (y+1)*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});

		// }
		// //bot neighbor
		// if (y+1<height && z-1>=0){
		// int offneighbor = x + (y+1)*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});

		// }
		// //botright neighbor
		// if (x+1<width && y+1<height && z-1>=0){
		// int offneighbor = x+1 + (y+1)*width- maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z-1, offneighbor});
		// }
		// /*
		// * Z+1 stack neighbors
		// */
		// //topleft neighbor
		// if (x-1>=0 && y-1>=0 && z+1<depth){
		// int offneighbor = x-1+ (y-1)*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});
		// }
		// //top neighbor
		// if (y-1 >=0 && z+1<depth){
		// int offneighbor = x+ (y-1)*width + maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});

		// }
		// //topright neighbor
		// if (y-1>=0 && x+1<width && z+1<depth){
		// int offneighbor= x+1 + (y-1)*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});
		// }
		// //left neighbor
		// if (x-1>=0 && z+1<depth){
		// int offneighbor = x-1+y*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});
		// }
		// //mid neighbor
		// if (z+1>=0 && z+1<depth){
		// int offneighbor = x+y*width+maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});

		// }
		// //right neighbor
		// if (x+1<width && z+1<depth){
		// int offneighbor= x+1+y*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});
		// }
		// //botleft neighbor
		// if (x-1>=0 && y+1<height && z+1<depth){
		// int offneighbor = x-1 + (y+1)*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});

		// }
		// //bot neighbor
		// if (y+1<height && z+1<depth){
		// int offneighbor = x + (y+1)*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});

		// }
		// //botright neighbor
		// if (x+1<width && y+1<height && z+1<depth){
		// int offneighbor = x+1 + (y+1)*width+ maxPix;
		// if (offneighbor<maxPix)
		// listNeighbors.add( new int[] {z+1, offneighbor});
		// }
		return listNeighbors;
	}

	/*
	 * Test function : shows water in white and the rest in black
	 */
	private static void isolerWshed(Sequence seq, int z) {
		for (int t = 0; t < seq.getSizeT(); t++) {
			IcyBufferedImage img = seq.getImage(t, z);
			DataBufferDouble buf = (DataBufferDouble) img.getRaster().getDataBuffer();
			double[] data = buf.getData();
			for (int i = 0; i < data.length; i++) {
				if (data[i] != WSHED) {
					data[i] = 0;
				} else {
					data[i] = 1;
				}
			}
		}
	}

	private static void isolerWshed3D(Sequence seq) {
		for (int t = 0; t < seq.getSizeT(); t++) {
			double[][] data = Array2DUtil.arrayToDoubleArray(seq.getDataXYZ(t, 0), seq.isSignedDataType());
			for (int z = 0; z < seq.getSizeZ(t); z++) {
				for (int i = 0; i < data[z].length; i++) {
					if (data[z][i] != WSHED) {
						data[z][i] = 1;
					}
				}
			}
		}
	}

	/**
	 * Calculates the "carte des bassins et des versants" of an image and the
	 * watershed. Pixels belonging to the watershed have the value 0 in the result
	 * sequence.
	 * 
	 * @param seq       Sequence to be transformed
	 * @param z         Z value of the image to be transformed
	 * @param typeWshed Type of the display
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */
	public static void watershed2D(Sequence seq, int z, int typeWshed) {

		switch (typeWshed) {
		case TYPEWSHED_SEP_BASINS:
			// passer en entree l'erode ultime
			// ATTENTION a la distance
			calculerVersantsEtiquetes(seq, z);
			// separerVersants(seq, z);
			break;
		case TYPEWSHED_BASINS_MAP:
			calculerLPE(seq, z, 8);
			// nothing more to do
			break;
		case TYPEWSHED_WSHEDONLY:
			calculerVersantsEtiquetes(seq, z);
			separerVersants(seq, z);
			isolerWshed(seq, z);
			break;
		}
	}

	/**
	 * Calculates the "carte des bassins et des versants" of 3D image and the
	 * watershed. Pixels belonging to the watershed have the value 0 in the result
	 * sequence.
	 * 
	 * @param seq       Sequence to be transformed
	 * @param typeWshed Type of the display
	 * 
	 * @throws IllegalArgumentException if the image type is not byte, short, float
	 *                                  or double
	 */
	public static void watershed3D(Sequence seq, int typeWshed) {
		calculerVersantsEtiquetes3D(seq);
		switch (typeWshed) {
		case TYPEWSHED_SEP_BASINS:
			separerVersants3D(seq);
			break;
		case TYPEWSHED_BASINS_MAP:
			// nothing more to do
			break;

		case TYPEWSHED_WSHEDONLY:
			separerVersants3D(seq);
			isolerWshed3D(seq);
			break;
		}
	}

	public static class UnhandledImageTypeException extends RuntimeException {
		static final long serialVersionUID = 0;
		private String text;

		public UnhandledImageTypeException(String txt) {
			text = txt;
		}

		public String toString() {
			return text;
		}
	}
}
