package log.lib;

/*
 * LoG Laplacian of Gaussian filter.
 *
 * @author 	Swiss Federal Institute of Technology Lausanne
 *			Biomedical Imaging Group
 *			Daniel Sage
 *
 *			Creatis - Lyon
 *          Hugo Rositi
 *
 *			ICY Coding Party May 2015 
 * @version 1.0
 *
 */

public class LoG3DLib {

	//private boolean showKernel = false;
	 
	/**
	* Constructor
	*/
	/*public LoG3DLib(boolean showKernel) {
		this.showKernel = showKernel;
	}*/
	
	public LoG3DLib()
	{}
	
	/**
	* Apply a Laplacian of Gaussian o a image or an image sequence (2D).
	* Note the size of the image should be greater than 6*sigma.
	*/
	public double[][] doLoG(double[][] input, double sigmaX, double sigmaY,int sizeX, int sizeY, int sizeZ) 
	{
		if (input == null)
			return null;
		double[][] out = new double[sizeZ][sizeX*sizeY];
		
		for(int z=0;z<sizeZ;z++)
			out[z] = doLoG_Separable(input[z], sigmaX, sigmaY,sizeX, sizeY);
		
		//this.printTab(out[0], nx, ny);		
		return out;
	}
	
	/**
	* Apply a Laplacian of Gaussian to a volume (3D).
	* Note the size of the image should be greater than 6*sigma.
	*/
	public double[][] doLoG(double[][] input, double sigmaX, double sigmaY, double sigmaZ, int sizeX, int sizeY, int sizeZ) 
	{
		if (input == null)
			return null;
		
		double[][] out = doLoG_Separable(input, sigmaX, sigmaY, sigmaZ, sizeX, sizeY, sizeZ);
		
		return out;
	}
	
	
	
	/**
	* Apply a Laplacian of Gaussian 2D.
	* Separable implementation.
	*/
	public double[] doLoG_Separable(double[] input, double sigmaX, double sigmaY, int nx, int ny) {
		if (input == null)
			return null;
		//int nx = sizeX;
		//int ny = sizeY;
		
		int d = 0; // dimension
		d = (sigmaX > 0 ? d+1 : d);
		d = (sigmaY > 0 ? d+1 : d);
		
		if (d == 0)
			return input;
		
		double pd = Math.pow(2*Math.PI, d/2.0);  //2 Pi
		double sx = (sigmaX > 0 ? sigmaX: 1.0);
		double sy = (sigmaY > 0 ? sigmaY: 1.0);
		double cst = 1.0/(pd*sx*sy);

		double kernelFactX[] = createKernelLoG_Fact(sigmaX, cst);	
		double kernelBaseX[] = createKernelLoG_Base(sigmaX);	
		double kernelFactY[] = createKernelLoG_Fact(sigmaY, cst);	
		double kernelBaseY[] = createKernelLoG_Base(sigmaY);	

		/*if (showKernel) {
			double kernel[][] = new double[kernelBaseX.length][kernelFactY.length];
			int nkx = kernelFactX.length;
			int nky = kernelFactY.length;
			for (int x=0; x<nkx; x++)
				for (int y=0; y<nky; y++)
					kernel[x][y] = kernelFactX[x]*kernelBaseY[y] + kernelFactY[y]*kernelBaseX[x];
			ImageWare iKernel = Builder.create(kernel);
			iKernel.show("LoG kernel sigma=" + sigmaX);
		}*/
		double[] outputX = new double[nx*ny];		
		double[] outputY = new double[nx*ny];
		
		for(int i=0; i<nx*ny;i++)  // Save the input.
		{
			outputX[i]=input[i];
			outputY[i]=input[i];
		}

	 	if (sigmaY > 0.0) 
	 	{
			double vinY[] = new double[ny];
			double voutY[] = new double[ny];
			for (int x=0; x<nx; x++) 
			{
				vinY=getColumn(outputX, x, nx, ny);
				convolve(vinY, voutY, kernelFactY);
				setColumn(outputX, voutY, x, nx, ny);
				
				vinY=getColumn(outputY,x,nx,ny);
				convolve(vinY, voutY, kernelBaseY);
				setColumn(outputY, voutY, x, nx, ny);
				
				/*outputX.getY(x, 0, t, vinY);
				convolve(vinY, voutY, kernelFactY);
				outputX.putY(x, 0, t, voutY);
				
				outputY.getY(x, 0, t, vinY);
				convolve(vinY, voutY, kernelBaseY);
				outputY.putY(x, 0, t, voutY);*/
			}
		}
	 	if (sigmaX > 0.0) 
	 	{
			double vinX[] = new double[nx];
			double voutX[] = new double[nx];
			for (int y=0; y<ny; y++) 
			{
				vinX=getRow(outputX, y, nx, ny);
				convolve(vinX, voutX, kernelBaseX);
				setRow(outputX, voutX, y, nx, ny);
				
				vinX=getRow(outputY,y,nx,ny);
				convolve(vinX, voutX, kernelFactX);
				setRow(outputY,voutX,y,nx,ny);
				
				/*outputX.getX(0, y, t, vinX);
				convolve(vinX, voutX, kernelBaseX);
				outputX.putX(0, y, t, voutX);
				
				outputY.getX(0, y, t, vinX);
				convolve(vinX, voutX, kernelFactX);
				outputY.putX(0, y, t, voutX);*/
			}
		}
	 	for(int i=0;i<nx*ny;i++)  // Add the result of the convolution in X and Y
	 		outputX[i]=outputX[i]+outputY[i];
	 	
	 	//outputX.add(outputY);

		return outputX;
	}
	
	/**
	* Apply a Laplacian of Gaussian 3D.
	* Separable implementation.
	*/
	public double[][] doLoG_Separable(double[][] input, double sigmaX, double sigmaY, double sigmaZ, int nx, int ny, int nz) {
		if (input == null)
			return null;
		
		int d = 0; // dimension
		d = (sigmaX > 0 ? d+1 : d);
		d = (sigmaY > 0 ? d+1 : d);
		d = (sigmaZ > 0 ? d+1 : d);
		
		if (d == 0)
			return input;
		
		double pd = Math.pow(2*Math.PI, d/2.0);
		double sx = (sigmaX > 0 ? sigmaX: 1.0);
		double sy = (sigmaY > 0 ? sigmaY: 1.0);
		double sz = (sigmaZ > 0 ? sigmaZ: 1.0);
		double cst = 1.0/(pd*sx*sy*sz);

		double kernelFactX[] = createKernelLoG_Fact(sigmaX, cst);	
		double kernelBaseX[] = createKernelLoG_Base(sigmaX);	
		double kernelFactY[] = createKernelLoG_Fact(sigmaY, cst);	
		double kernelBaseY[] = createKernelLoG_Base(sigmaY);	
		double kernelFactZ[] = createKernelLoG_Fact(sigmaZ, cst);	
		double kernelBaseZ[] = createKernelLoG_Base(sigmaZ);	

		/*if (showKernel) {
			double kernel[][][] = new double[kernelBaseX.length][kernelFactY.length][kernelFactZ.length];
			int nkx = kernelFactX.length;
			int nky = kernelFactY.length;
			int nkz = kernelFactZ.length;
			for (int x=0; x<nkx; x++)
			for (int y=0; y<nky; y++) 
			for (int z=0; z<nkz; z++) {
				kernel[x][y][z] = 
					kernelFactX[x]*kernelBaseY[y]*kernelBaseZ[z] + 
					kernelFactY[y]*kernelBaseX[x]*kernelBaseZ[z] +
					kernelFactZ[z]*kernelBaseX[x]*kernelBaseY[y]; 
			}
			ImageWare iKernel = Builder.create(kernel);
			iKernel.show("LoG kernel Sigma=" + sigmaX);
		}*/

		double[][] outputX = new double[nz][nx*ny];	// on remplit ligne-ligne 
		double[][] outputY = new double[nz][nx*ny];	// on remplit col-col
		double[][] outputZ = new double[nz][nx*ny];	// on remplit z-z 
		
		for(int k =0; k<nz;k++)	// Save the input.
			for(int i=0; i<nx*ny;i++)
			{
				outputX[k][i]=input[k][i];         
				outputY[k][i]=input[k][i];
				outputZ[k][i]=input[k][i];
			}
		
		
 		if (sigmaZ > 0.0) //&& kernelFactZ.length < nz)
 		{ 
	 		double vinZ[] = new double[nz];
	 		double voutZ[] = new double[nz];
			for (int x=0; x<nx; x++)
				for (int y=0; y<ny; y++) 
				{							
					vinZ=getZ(outputX, x, y, nx, ny, nz); 
					convolve(vinZ, voutZ, kernelFactZ);
					setZ(outputX, voutZ, x, y, nx, ny , nz);
				
					vinZ=getZ(outputY, x, y, nx, ny, nz); 
					convolve(vinZ, voutZ, kernelBaseZ);
					setZ(outputY, voutZ, x, y, nx, ny , nz);

					vinZ=getZ(outputZ, x, y, nx, ny, nz);
					convolve(vinZ, voutZ, kernelBaseZ);
					setZ(outputZ, voutZ, x, y, nx, ny , nz);
				}
		}
		
 		if (sigmaY > 0.0) { //&& kernelFactY.length < ny) {
	 		double vinY[] = new double[ny];
	 		double voutY[] = new double[ny];
			for (int x=0; x<nx; x++)
				for (int z=0; z<nz; z++) 
				{
					vinY=getColumn(outputX[z], x, nx, ny);
					convolve(vinY, voutY, kernelBaseY);
					setColumn(outputX[z], voutY, x, nx, ny);
					
					
					vinY=getColumn(outputY[z], x, nx, ny);
					convolve(vinY, voutY, kernelFactY);
					setColumn(outputY[z], voutY, x, nx, ny);
	
					vinY=getColumn(outputZ[z], x, nx, ny);
					convolve(vinY, voutY, kernelBaseY);
					setColumn(outputZ[z], voutY, x, nx, ny);
				}
		}
		
 		if (sigmaX > 0.0) { // && kernelFactX.length < nx) {
	 		double vinX[] = new double[nx];
	 		double voutX[] = new double[nx];
			for (int y=0; y<ny; y++)
				for (int z=0; z<nz; z++) 
				{
					
					vinX=getRow(outputX[z], y, nx, ny);
					convolve(vinX, voutX, kernelBaseX);
					setRow(outputX[z], voutX, y, nx, ny);
					
					vinX=getRow(outputY[z], y, nx, ny);
					convolve(vinX, voutX, kernelBaseX);
					setRow(outputY[z], voutX, y, nx, ny);
	
					vinX=getRow(outputZ[z], y, nx, ny);
					convolve(vinX, voutX, kernelFactX);
					setRow(outputZ[z], voutX, y, nx, ny);
	
				}
		}
 		
 		for(int k=0;k<nz;k++)
	 		for(int i=0;i<nx*ny;i++)
	 			outputX[k][i]=outputX[k][i]+outputY[k][i]+outputZ[k][i];
		
		return outputX;
	}

	/**
	* exp(-(x^2)/(sigma^2))
	*/
	public double[] createKernelLoG_Base(double sigma) {
	
		if (sigma <= 0.0) {
			double[] kernel = new double[1];
			kernel[0] = 1.0;
			return kernel;
		}
		
		double s2 = sigma*sigma;
		double dem = 2.0*s2;
		int size = (int)Math.round(((int)(sigma*3.0))*2.0 + 1);	// always odd size
		int size2 = size/2;
		double[] kernel = new double[size];
		
		double x;
		for(int k=0; k<size; k++) {
			x = (k-size2)*(k-size2);
			kernel[k] = Math.exp(-x/dem);
		}
		return kernel;
	}
	
	

	/**
	* cst*(x^2/(sigma^4)-1/(sigma^2))*exp(-(x^2)/(sigma^2))
	*/
	public double[] createKernelLoG_Fact(double sigma, double cst) {
		
		if (sigma <= 0.0) {
			double[] kernel = new double[1];
			kernel[0] = 1.0;
			return kernel;
		}
		
		double s2 = sigma*sigma;
		double s4 = s2*s2;
		double dem = 2.0*s2;
		int size = (int)Math.round(((int)(sigma*3.0))*2.0 + 1);	// always odd size
		int size2 = size/2;
		double[] kernel = new double[size];
		
		double x;
		for(int k=0; k<size; k++) {
			x = (k-size2)*(k-size2);
			kernel[k] = cst * (x/s4-1.0/s2) * Math.exp(-x/dem);
		}
		return kernel;
	}


	private double[] convolve(double vin[], double vout[], double kernel[]) {
		int n = vin.length;
		int nk = kernel.length;
		int kc = nk/2;
		int	period = (n <= 1 ? 1: 2*n - 2);
		
		int im;
		for (int i=0; i<n; i++) {
			double sum=0.0;
			for (int k=0; k<nk; k++) {
				im = i + k - kc;
				while (im < 0)
					im += period;
				while (im >= n) {
					im = period - im;
					im = (im < 0 ? -im : im);
				}
				sum += kernel[k] * vin[im];
			}
			vout[i] = sum;
		}
		return vout;
	}

	/**
	*
	private double[] convolve(double vin[], double kernel[]) {
		int n = vin.length;
		int nk = kernel.length;
		int kc = nk/2;
		double[] vout = new double[n];
		
		int im;
		for (int i=0; i<n; i++) {
			double sum=0.0;
			for (int k=0; k<nk; k++) {
				im = i + k - kc;
				while (im < 0)
					im = -im;
				while (im >= n)
					im = 2*n - im-1;
				sum += kernel[k] * vin[im];
			}
			vout[i] = sum;
		}
		return vout;
	}
	*/
	
	/**
	 * Return a specific row of a 1D array of 2D dataset.
	 */
	public double[] getRow(double[] input, int rowInd, int nx, int ny)
	{
		if (rowInd >= ny)
				return null;
		double[] row = new double[nx];
		
		int j=0;
		
		for(int i=rowInd*nx;i<rowInd*nx+nx;i++)
		{
			row[j]=input[i];
			j++;
		}			
		
		return row;
	}
	
	/**
	 * Return a specific column of a 1D array of 2D dataset.
	 */
	public double[] getColumn(double[] input, int colInd, int nx, int ny)
	{
		if (colInd >= nx)
			return null;
		
	double[] col = new double[ny];
	
	int j=0;
	
	for(int i=colInd;i<(ny-1)*nx+colInd+1;i=i+nx)
	{
		col[j]=input[i];
		j++;
	}			
	
	return col;
	}
	
	/**
	 * Return a specific Z vector of a 1D array of 3D dataset.
	 */
	public double[] getZ(double[][] input, int x, int y, int nx, int ny, int nz)
	{
		if (x > nx || y>ny)
			return null;
		
		double[] z = new double[nz];
	
		for(int i=0;i<nz;i++)
			z[i]=input[i][x+nx*y];
	
		return z;
	}
	
	
	/**
	 * Replaces a specific Z vector of a 1D array of 3D dataset.
	 */
	public void setZ(double[][] input, double[] zpro, int x, int y, int nx, int ny, int nz)
	{
		if (x<nx && y<ny)	
			for(int i=0;i<nz;i++)
				input[i][x+y*nx]=zpro[i];
	}
	
	
	/**
	 * Replaces a specific row of a 1D array(2D dataset) from a 1D array
	 */
	public void setRow(double[] input, double[] row, int rowInd, int nx, int ny)
	{
		if (rowInd<ny)
		{
			int j=0;
			for(int i=rowInd*nx;i<rowInd*nx+nx;i++)
			{
				input[i]=row[j];
				j++;
			}
		}
	}
	
	/**
	 * Replaces a specific colmun of a 2D array from a 1D array
	 */
	public void setColumn(double[] input, double[] col, int colInd, int nx, int ny)
	{
		if (colInd<nx)
		{
			int j=0;
			
			for(int i=colInd;i<(ny-1)*nx+colInd+1;i=i+nx)
			{
				input[i]=col[j];
				j++;
			}
		}
	}
	
	public void printTab(double[] tab, int nx, int ny)
	{
		for(int i=0;i<ny;i++)
		{
			for(int j=0;j<nx;j++)
			{
				System.out.print(tab[i*nx+j]+" ");
			}
			System.out.println();
		}
	}
	
	/**
	* Apply a Laplacian of Gaussian 2D.
	* Non Separable implementation.
	*/
	/*private ImageWare doLoG_NonSeparable(ImageWare input, double sigmaX, double sigmaY) {
		if (input == null)
			return null;
			
		int nx = input.getSizeX();
		int ny = input.getSizeY();
		int n = input.getSizeZ();
		double kernel[][] = createKernelLoG_NonSeparable(sigmaX);	
		int dimension = kernel.length;
		double neigh[][] = new double[dimension][dimension];
		
		double pix;
		ImageWare slice = Builder.create(nx, ny, 1, input.getType());
		ImageWare output = Builder.create(nx, ny, n, input.getType());
		double p = 0;
		for (int t=0; t<n; t++) {
			input.getXY(0, 0, t, slice);
			for (int x=0; x<nx; x++)
			for (int y=0; y<ny; y++) {
				pix = 0.0;
				slice.getNeighborhoodXY(x, y, 0, neigh, ImageWare.MIRROR);
				for (int k=0; k<dimension; k++) 
				for (int l=0; l<dimension; l++)
					pix += neigh[k][l]*kernel[k][l];
				output.putPixel(x, y, t, pix);
			}
		}
		if (showKernel) {
			ImageWare iKernel = Builder.create(kernel);
			iKernel.show("LoG kernel sigma=" + sigmaX);
		}
		return output;
	}*/
	
	
	/**
	* Apply a Laplacian of Gaussian 3D.
	* Non Separable implementation.
	*/
	/*private ImageWare doLoG_NonSeparable(ImageWare input, double sigmaX, double sigmaY, double sigmaZ) {
		if (input == null)
			return null;
			
		int nx = input.getSizeX();
		int ny = input.getSizeY();
		int nz = input.getSizeZ();
		double kernel[][][] = createKernelLoG_NonSeparable3(sigmaX);	
		int dimx = kernel.length;
		double neigh[][][] = new double[dimx][dimx][dimx];
		
		double pix;
		ImageWare output = Builder.create(nx, ny, nz, input.getType());
		double p = 0;
		for (int z=0; z<nz; z++)
		for (int x=0; x<nx; x++)
		for (int y=0; y<ny; y++) {
			pix = 0.0;
			input.getNeighborhoodXYZ(x, y, z, neigh, ImageWare.MIRROR);
			for (int k=0; k<dimx; k++) 
			for (int l=0; l<dimx; l++)
			for (int m=0; m<dimx; m++)
				pix += neigh[k][l][m]*kernel[k][l][m];
			output.putPixel(x, y, z, pix);
		}
		if (showKernel) {
			ImageWare iKernel = Builder.create(kernel);
			iKernel.show("LoG kernel sigma=" + sigmaX);
		}
		return output;
	}*/

	
	/**
	*/
	/*
	private double[][] createKernelLoG_NonSeparable(double sigma) {
	
		double cst = -(1.0 / (Math.PI*Math.pow(sigma,4)));
		double dem = 2.0 * Math.pow(sigma,2);
		int size = (int)(sigma*6.0);
		double x, y;
		double[][] kernel = new double[size][size];
		int halfsize = size / 2;
		
		for(int k=0; k<size; k++)
		for(int l=0; l<size; l++) {
			x = (k-halfsize)*(k-halfsize);
			y = (l-halfsize)*(l-halfsize);
			kernel[k][l] = cst * (1.0-(x+y)/dem) * Math.exp(-(x+y)/dem);
		}
		return kernel;
	}
	*/
	/**
	*/
	/*
	private double[][][] createKernelLoG_NonSeparable3(double sigma) {
	
		double cst = -(1.0 / (Math.PI*Math.pow(sigma,4)));
		double dem = 2.0 * Math.pow(sigma,2);
		int size = (int)(sigma*6.0);
		double x, y, z;
		double[][][] kernel = new double[size][size][size];
		int halfsize = size / 2;
		
		for(int k=0; k<size; k++)
		for(int l=0; l<size; l++)
		for(int m=0; m<size; m++) {
			x = (k-halfsize)*(k-halfsize);
			y = (l-halfsize)*(l-halfsize);
			z = (m-halfsize)*(m-halfsize);
			kernel[k][l][m] = cst * (1.0-(x+y+z)/dem) * Math.exp(-(x+y+z)/dem);
		}
		return kernel;
	}
	*/
	
	
	
	
 }
