package plugins.vannary.morphomaths.blocks;

import icy.plugin.abstract_.PluginActionable;
import icy.sequence.Sequence;
import icy.sequence.SequenceUtil;
import icy.system.IcyHandledException;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzVar;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarListener;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.vannary.morphomaths.MorphOp;

/**
 * Performs a binary dilation on the active sequence or the block input sequence.
 * @author Daniel Felipe Gonzalez Obando
 */
public class Dilation extends PluginActionable implements Block {

	private static double[][][] STRUCTURING_ELEMENT_3D = new double[][][] { { { 1, 1, 1 }, { 1, 1, 1 }, { 1, 1, 1 } },
			{ { 1, 1, 1 }, { 1, 1, 1 }, { 1, 1, 1 } }, { { 1, 1, 1 }, { 1, 1, 1 }, { 1, 1, 1 } } };

	private static double[][] STRUCTURING_ELEMENT_2D = new double[][] { { 1, 1, 1 }, { 1, 1, 1 }, { 1, 1, 1 } };

	private EzVarSequence inputSequence;
	private EzVarInteger inputDistance;
	private EzVarBoolean inputOverwriteResult;
	private EzVarBoolean input3dModality;
	private EzVarInteger inputProcessedZ;

	@Override
	public void declareInput(VarList inputMap) {
		inputSequence = new EzVarSequence("Sequence");
		inputDistance = new EzVarInteger("Distance", 1, 0, Integer.MAX_VALUE, 1);
		inputOverwriteResult = new EzVarBoolean("Results on input", false);
		input3dModality = new EzVarBoolean("Compute in 3D", false);
		inputProcessedZ = new EzVarInteger("Z value");

		inputProcessedZ.getVariable().setEnabled(false);
		input3dModality.addVarChangeListener(new EzVarListener<Boolean>() {

			@Override
			public void variableChanged(EzVar<Boolean> source, Boolean newValue) {
				if (newValue) {
					inputProcessedZ.getVariable().setEnabled(false);
				} else {
					inputProcessedZ.getVariable().setEnabled(true);
				}
			}
		});

		inputMap.add(inputSequence.name, inputSequence.getVariable());
		inputMap.add(inputDistance.name, inputDistance.getVariable());
		inputMap.add(inputOverwriteResult.name, inputOverwriteResult.getVariable());
		inputMap.add(input3dModality.name, input3dModality.getVariable());
		inputMap.add(inputProcessedZ.name, inputProcessedZ.getVariable());
	}

	private EzVarSequence outputSequence;

	@Override
	public void declareOutput(VarList outputMap) {
		outputSequence = new EzVarSequence("Dilation Sequence");
		outputMap.add(outputSequence.name, outputSequence.getVariable());
	}

	private Sequence transformedSequence;

	@Override
	public void run() {
	    Sequence inSequence;
        if (inputSequence != null)
        {
            inSequence = inputSequence.getValue(true);
        }
        else
        {
            inSequence = getActiveSequence();
        }
        if (inSequence == null)
            throw new IcyHandledException("No sequence selected!");
        
        int distance;
        boolean isReusingSequence;
        boolean is3D;
        int processedZ;
        
        if (inputSequence != null) {
            distance = inputDistance.getValue();
            isReusingSequence = inputOverwriteResult.getValue();
            is3D = input3dModality.getValue();
            processedZ = inputProcessedZ.getValue();
        } else {
            distance = 1;
            isReusingSequence = false;
            is3D = inSequence.getSizeZ() > 1;
            processedZ = 0;
        }

		if (isReusingSequence) {
			transformedSequence = inSequence;
		} else {
			transformedSequence = SequenceUtil.getCopy(inSequence);
			transformedSequence.setName(inSequence.getName() + "_Dilation");
		}

		if (distance < 0) {
			throw new RuntimeException("Distance must be non-negative");
		}

		for (int i = 0; i < distance && !Thread.interrupted(); i++) {
			if (is3D) {
				MorphOp.dilateGreyScale3D(transformedSequence, STRUCTURING_ELEMENT_3D, 1, 1, 1);
			} else {
				if (processedZ >= 0 && processedZ < transformedSequence.getSizeZ()) {
					MorphOp.dilateGreyScale(transformedSequence, processedZ, STRUCTURING_ELEMENT_2D, 1, 1);
				} else {
					throw new RuntimeException("Invalid z coordinate: " + processedZ + ". Expected [0 - "
							+ transformedSequence.getSizeZ() + ")");
				}
			}

		}

		transformedSequence.dataChanged();
		if (outputSequence != null)
		    outputSequence.setValue(transformedSequence);
		else
		    addSequence(transformedSequence);
	}

}
