/**
 * 
 */
package plugins.stef.roi.bloc.op;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.plugin.interface_.PluginLibrary;
import icy.roi.ROI;
import icy.roi.ROIDescriptor;
import icy.sequence.Sequence;
import icy.system.IcyExceptionHandler;
import icy.type.collection.CollectionUtil;
import icy.util.StringUtil;
import plugins.adufour.blocks.tools.roi.ROIBlock;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.gui.model.ValueSelectionModel;
import plugins.adufour.vars.lang.VarDouble;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.lang.VarString;
import plugins.adufour.vars.util.VarException;
import plugins.kernel.roi.descriptor.measure.ROIInteriorDescriptor;
import plugins.stef.roi.bloc.RoiBlocks;

/**
 * Block to filter a set of ROIs using a specific descriptor.
 * 
 * @author Stephane
 */
public class FilterROI extends Plugin implements ROIBlock, PluginLibrary, PluginBundled
{
    public static enum CompareOperator
    {
        EQUAL
        {
            @Override
            public String toString()
            {
                return "Equal to";
            }
        },
        NOT_EQUAL
        {
            @Override
            public String toString()
            {
                return "Not equal to";
            }
        },
        LOWER
        {
            @Override
            public String toString()
            {
                return "Lower than";
            }
        },
        LOWER_OR_EQUAL
        {
            @Override
            public String toString()
            {
                return "Lower or equal than";
            }
        },
        GREATER
        {
            @Override
            public String toString()
            {
                return "Greater than";
            }
        },
        GREATER_OR_EQUAL
        {
            @Override
            public String toString()
            {
                return "Greater or equal than";
            }
        }
    }

    protected final VarROIArray roiSet = new VarROIArray("ROI(s)", null);
    protected final VarSequence sequence = new VarSequence("Sequence", null);
    protected final VarString descriptors = new VarString("Filter on", "");
    protected final VarEnum<CompareOperator> operator = new VarEnum<CompareOperator>("Accept if",
            CompareOperator.EQUAL);
    protected final VarDouble value = new VarDouble("Value", 0d);
    protected final VarROIArray output = new VarROIArray("Filtered ROI(s)");

    @Override
    public void run()
    {
        try
        {
            final List<ROI> result = filterROIs(CollectionUtil.asList(roiSet.getValue()), sequence.getValue(),
                    descriptors.getValue(), operator.getValue(), value.getValue().doubleValue());

            output.setValue(result.toArray(new ROI[result.size()]));
        }
        catch (IllegalArgumentException e)
        {
            throw new VarException(descriptors, e.getMessage());
        }
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        final List<ROIDescriptor> roiDescriptors = new ArrayList<ROIDescriptor>(
                ROIDescriptor.getDescriptors().keySet());
        // build list of descriptor id
        final List<String> descriptorsId = new ArrayList<String>();
        String sizeDescriptorId = null;

        for (ROIDescriptor descriptor : roiDescriptors)
        {
            final String id = descriptor.getId();

            // keep trace of size descriptor
            if (StringUtil.equals(ROIInteriorDescriptor.ID, id))
                sizeDescriptorId = id;

            descriptorsId.add(id);
        }

        // alpha sort
        Collections.sort(descriptorsId);

        // initialize descriptors field
        descriptors.setDefaultEditorModel(new ValueSelectionModel<String>(
                descriptorsId.toArray(new String[descriptorsId.size()]), sizeDescriptorId, false));

        inputMap.add("roi", roiSet);
        inputMap.add("sequence", sequence);
        inputMap.add("descriptors", descriptors);
        inputMap.add("operator", operator);
        inputMap.add("value", value);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("out", output);
    }

    @Override
    public String getMainPluginClassName()
    {
        return RoiBlocks.class.getName();
    }

    /**
     * Filter a set of ROIs using a specific descriptor and the compare operation (keep ROIS where result of comparison is true).
     * 
     * @param rois
     *        input ROIS to filter
     * @param sequence
     *        input sequence used for ROI descriptor computation (can be null if not required)
     * @param descriptorId
     *        the {@link ROIDescriptor} id to use for filtering
     * @param op
     *        Compare operation to use
     * @param value
     *        the value for the comparison
     * @return filtered list of ROIS
     * @throws IllegalArgumentException
     *         if given ROI descriptor id is not found
     */
    public static List<ROI> filterROIs(Collection<ROI> rois, Sequence sequence, String descriptorId, CompareOperator op,
            double value) throws IllegalArgumentException
    {
        final List<ROI> result = new ArrayList<ROI>();
        final ROIDescriptor roiDescriptor = ROIDescriptor.getDescriptor(ROIDescriptor.getDescriptors().keySet(),
                descriptorId);

        if (roiDescriptor == null)
            throw new IllegalArgumentException("Cannot found '" + descriptorId + "' ROI descriptor !");

        for (ROI roi : rois)
        {
            if (roi != null)
            {
                try
                {
                    final Object res = roiDescriptor.compute(roi, sequence);
                    boolean accepted;

                    // we only support Number result for filtering
                    if (res instanceof Number)
                    {
                        final double computedValue = ((Number) res).doubleValue();

                        switch (op)
                        {
                            default:
                            case EQUAL:
                                accepted = computedValue == value;
                                break;
                            case NOT_EQUAL:
                                accepted = computedValue != value;
                                break;
                            case LOWER:
                                accepted = computedValue < value;
                                break;
                            case LOWER_OR_EQUAL:
                                accepted = computedValue <= value;
                                break;
                            case GREATER:
                                accepted = computedValue > value;
                                break;
                            case GREATER_OR_EQUAL:
                                accepted = computedValue >= value;
                                break;
                        }
                    }
                    else
                        accepted = true;

                    if (accepted)
                        result.add(roi);
                }
                catch (Exception e)
                {
                    IcyExceptionHandler.showErrorMessage(e, false, true);
                    // if we can't compute the descriptor we accept the ROI by default...
                    result.add(roi);
                }
            }
        }

        return result;
    }
}