package plugins.stef.roi.bloc.op;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.plugin.interface_.PluginLibrary;
import icy.roi.ROI;
import icy.type.collection.CollectionUtil;
import plugins.adufour.blocks.tools.roi.ROIBlock;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarROIArray;
import plugins.stef.roi.bloc.RoiBlocks;

/**
 * Block computing logical operation between 2 groups of ROIs to retain only a sub part of the first group depending the selected Logical Operation.
 * Examples<br>
 * - Logic operator A_CONTAINING_B means it will keep all ROIs from group A which contains at least one ROI from group B.<br>
 * - Logic operator A_NOT_CONTAINED_IN_B means it will keep all ROIs from group A which are not contained by any ROI from group B.
 * 
 * @author Stephane
 */
public class LogicalOperationROI extends Plugin implements ROIBlock, PluginLibrary, PluginBundled
{
    public static enum LogicOperator
    {
        A_CONTAINED_IN_B, A_CONTAINING_B, A_INTERSECTING_B, A_NOT_CONTAINED_IN_B, A_NOT_CONTAINING_B,
        A_NOT_INTERSECTING_B
    }

    protected VarROIArray roiSetA = new VarROIArray("ROI(s) group A", null);
    protected VarROIArray roiSetB = new VarROIArray("ROI(s) group B", null);
    protected VarEnum<LogicOperator> op = new VarEnum<LogicOperator>("Keep", LogicOperator.A_CONTAINED_IN_B);
    protected VarROIArray output = new VarROIArray("Result");

    @Override
    public void run()
    {
        final List<ROI> result = doLogicalOperation(CollectionUtil.asList(roiSetA.getValue()),
                CollectionUtil.asList(roiSetB.getValue()), op.getValue());

        output.setValue(result.toArray(new ROI[result.size()]));
    }

    @Override
    public void declareInput(VarList inputMap)
    {
        inputMap.add("roiA", roiSetA);
        inputMap.add("roiB", roiSetB);
        inputMap.add("op", op);
    }

    @Override
    public void declareOutput(VarList outputMap)
    {
        outputMap.add("out", output);
    }

    @Override
    public String getMainPluginClassName()
    {
        return RoiBlocks.class.getName();
    }

    /**
     * Apply a logical operation between 2 set of ROIS and obtain the result.
     * 
     * @param roiSetA
     *        first set of ROI
     * @param roiSetB
     *        second set of ROI
     * @param logicOp
     *        logical operation to apply between the 2 sets of ROI
     * @return result ROIs from the given logical operation between the 2 sets of ROI
     */
    public static List<ROI> doLogicalOperation(Collection<ROI> roiSetA, Collection<ROI> roiSetB, LogicOperator logicOp)
    {
        final List<ROI> result = new ArrayList<ROI>();

        for (ROI roiA : roiSetA)
        {
            if (roiA != null)
            {
                boolean cond = (logicOp == LogicOperator.A_NOT_CONTAINED_IN_B)
                        || (logicOp == LogicOperator.A_NOT_CONTAINING_B)
                        || (logicOp == LogicOperator.A_NOT_INTERSECTING_B);
                boolean done = false;

                for (ROI roiB : roiSetB)
                {
                    switch (logicOp)
                    {
                        default:
                        case A_CONTAINED_IN_B:
                        case A_NOT_CONTAINED_IN_B:
                            if (roiB.contains(roiA))
                            {
                                done = true;
                                cond = (logicOp == LogicOperator.A_CONTAINED_IN_B);
                            }
                            break;

                        case A_CONTAINING_B:
                        case A_NOT_CONTAINING_B:
                            if (roiA.contains(roiB))
                            {
                                done = true;
                                cond = (logicOp == LogicOperator.A_CONTAINING_B);
                            }
                            break;

                        case A_INTERSECTING_B:
                        case A_NOT_INTERSECTING_B:
                            if (roiA.intersects(roiB))
                            {
                                done = true;
                                cond = (logicOp == LogicOperator.A_INTERSECTING_B);
                            }
                            break;
                    }

                    // can stop here
                    if (done)
                        break;
                }

                // condition verified ?
                if (cond)
                    result.add(roiA.getCopy());
            }
        }

        return result;
    }
}
