package plugins.lagache.matchtracks;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;

import plugins.fab.trackmanager.TrackSegment;


/**
 * Links two lists of objects based on the LAP framework described in Jaqaman
 * <i>et al.</i>, Nature Methods, <b>2008</b>.
 * 
 * @author Jean-Yves Tinevez - 2014
 * 
 * @param <K>
 *            the type of the source objects to link.
 * @param <J>
 *            the type of the target objects to link.
 */
public class SparseLinker{
	private Map< TrackSegment, TrackSegment > assignments;

	private Map< TrackSegment, Double > costs;

	private final SparseLinkingCostMatrixCreator costMatrixCreator;
	

	/**
	 * Creates a new linker for the specified cost matrix creator. See Jaqaman
	 * <i>et al.</i>, Nature Methods, <b>2008</b>, Figure 1b.
	 * 
	 * @param costMatrixCreator
	 *            the class in charge of creating linking costs.
	 * @param logger
	 *            a logger that will receive progress messages.
	 */

	public SparseLinker( final SparseLinkingCostMatrixCreator costMatrixCreator )
	{
		this.costMatrixCreator=costMatrixCreator;
	}

	/**
	 * Returns the resulting assignments from this algorithm.
	 * <p>
	 * It takes the shape of a map, such that if <code>source</code> is a key of
	 * the map, it is assigned to <code>target = map.get(source)</code>.
	 * 
	 * @return the assignment map.
	 * @see #getAssignmentCosts()
	 */	
	public Map< TrackSegment, TrackSegment > getResult()
	{
		return assignments;
	}

	/**
	 * Returns the costs associated to the assignment results.
	 * <p>
	 * It takes the shape of a map, such that if <code>source</code> is a key of
	 * the map, its assignment as a cost <code>cost = map.get(source)</code>.
	 * 
	 * @return the assignment costs.
	 * @see #getResult()
	 */
	public Map< TrackSegment, Double > getAssignmentCosts()
	{
		return costs;
	}

	
		public boolean process()
	{
		final long start = System.currentTimeMillis();

		/*
		 * Generate the cost matrix
		 */

		
		final SparseCostMatrix tl = SparseLinkingCostMatrixCreator.getResult();
		final List< TrackSegment > matrixRows = SparseLinkingCostMatrixCreator.getSourceList();
		final List< TrackSegment > matrixCols = SparseLinkingCostMatrixCreator.getTargetList();

		if ( matrixCols.isEmpty() || matrixRows.isEmpty() )
		{
			assignments = Collections.emptyMap();
			costs = Collections.emptyMap();
			final long end = System.currentTimeMillis();		
			return true;
		}

		/*
		 * Complement the cost matrix with alternative no linking cost matrix.
		 */

	
		final int nCols = tl.getNCols();
		final int nRows = tl.getNRows();

		/*
		 * Top right
		 */

		final double[] cctr = new double[ nRows ];
		final int[] kktr = new int[ nRows ];
		for ( int i = 0; i < nRows; i++ )
		{
			kktr[ i ] = i;
			cctr[ i ] = costMatrixCreator.getAlternativeCostForSource( matrixRows.get( i ) );
		}
		final int[] numbertr = new int[ nRows ];
		Arrays.fill( numbertr, 1 );
		final SparseCostMatrix tr = new SparseCostMatrix( cctr, kktr, numbertr, nRows );

		/*
		 * Bottom left
		 */
		final double[] ccbl = new double[ nCols ];
		final int[] kkbl = new int[ nCols ];
		for ( int i = 0; i < kkbl.length; i++ )
		{
			kkbl[ i ] = i;
			ccbl[ i ] = costMatrixCreator.getAlternativeCostForTarget( matrixCols.get( i ) );
		}
		final int[] numberbl = new int[ nCols ];
		Arrays.fill( numberbl, 1 );
		final SparseCostMatrix bl = new SparseCostMatrix( ccbl, kkbl, numberbl, nCols );

		/*
		 * Bottom right.
		 * 
		 * Alt. cost is the overall min of alternative costs. This deviate or
		 * extend a bit the u-track code.
		 */
		
		final double minCost = Math.min( min( ccbl ), min( cctr ) );
		final SparseCostMatrix br = tl.transpose();
		br.fillWith( minCost );

		/*
		 * Stitch them together
		 */
		final SparseCostMatrix full = ( tl.hcat( tr ) ).vcat( bl.hcat( br ) );
		
		/*
		 * Solve the full cost matrix.
		 */
		final LAPJV solver = new LAPJV( full );
		solver.process();
		
		final int[] assgn = solver.getResult();
		assignments = new HashMap< TrackSegment, TrackSegment >();
		costs = new HashMap< TrackSegment, Double >();
		for ( int i = 0; i < assgn.length; i++ )
		{
			final int j = assgn[ i ];
			if ( i < matrixRows.size() && j < matrixCols.size() )
			{
				final TrackSegment source = matrixRows.get( i );
				final TrackSegment target = matrixCols.get( j );
				assignments.put( source, target );

				final double cost = full.get( i, j, Double.POSITIVE_INFINITY );
				costs.put( source, Double.valueOf( cost ) );
			}
		}

				return true;
	}
		public double min(double[] input){
			if (input.length==0){return Double.MIN_VALUE;}
			java.util.Arrays.sort(input);
			return input[0];
		}

	}
