package plugins.lagache.matchtracks;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import plugins.fab.trackmanager.TrackSegment;


/**
 * A {@link CostMatrixCreator} that can generate a cost matrix from a list of
 * sources, a list of targets and a {@link CostFunction} that can generate a
 * cost for any combination.
 * 
 * @author Jean-Yves Tinevez - 2014
 * 
 * @param <K>
 * @param <J>
 */
public class SparseLinkingCostMatrixCreator
{

	private static final String BASE_ERROR_MSG = "[JaqamanLinkingCostMatrixCreator] ";

	private final ArrayList<TrackSegment> sources;

	private final ArrayList<TrackSegment> targets;
	
	private final ArrayList<HashMap<TrackSegment,double[]>> backward_start;
	private final ArrayList<HashMap<TrackSegment,double[]>> forward_end;
	private final int initial_time;
	private final int last_time;
	private static SparseCostMatrix scm;
	
	private final double costThreshold;
	private final Integer time_window;

	private static List< TrackSegment > sourceList;

	private static List< TrackSegment > targetList;

	private double alternativeCost;

	private final double alternativeCostFactor;

	private final double percentile;

	public SparseLinkingCostMatrixCreator( final ArrayList<TrackSegment> sources, final ArrayList<TrackSegment> targets, final ArrayList<HashMap<TrackSegment,double[]>> backward_start,final ArrayList<HashMap<TrackSegment,double[]>> forward_end, final double costThreshold, final Integer time_window,final double alternativeCostFactor, final double percentile, final int initial_time,int last_time )
	{
		this.sources = sources;
		this.targets = targets;
		this.backward_start = backward_start;
		this.forward_end = forward_end;
		this.costThreshold = costThreshold;
		this.time_window = time_window;
		this.alternativeCostFactor = alternativeCostFactor;
		this.percentile = percentile;
		this.initial_time=initial_time;
		this.last_time=last_time;
	}

	
	public boolean checkInput()
	{
		if ( null == sources || !sources.iterator().hasNext() )
		{			
			return false;
		}
		if ( null == targets || !targets.iterator().hasNext() )
		{		
			return false;
		}
		return true;
	}

	public boolean process()
	{		

		final List< TrackSegment > accSources = new ArrayList< TrackSegment >();
		final List< TrackSegment > accTargets = new ArrayList< TrackSegment >();
		final ResizableDoubleArray costs = new ResizableDoubleArray();

		for ( final TrackSegment source : sources )
		{
			for ( final TrackSegment target : targets )
			{

				final double cost = compute_shortest_distance(source, target, backward_start, forward_end, initial_time,last_time,costThreshold,time_window);
				if ( cost < costThreshold )
				{
					accSources.add( source );
					accTargets.add( target );
					costs.add( cost );
				}
			}
		}
		costs.trimToSize();

		/*
		 * Check if accepted source or target lists are empty and deal with it.
		 */

		if ( accSources.isEmpty() || accTargets.isEmpty() )
		{

			sourceList = Collections.emptyList();
			targetList = Collections.emptyList();
			alternativeCost = Double.NaN;
			scm = null;
			/*
			 * CAREFUL! We return null if no acceptable links are found.
			 */
		}
		else
		{

			final DefaultCostMatrixCreator cmCreator = new DefaultCostMatrixCreator( accSources, accTargets, costs.data, alternativeCostFactor, percentile );
			if ( !cmCreator.checkInput() || !cmCreator.process() )
			{			
				return false;
			}
			
			scm = cmCreator.getResult();
			sourceList = cmCreator.getSourceList();
			targetList = cmCreator.getTargetList();
			alternativeCost = cmCreator.computeAlternativeCosts();
		}


		return true;
	}


	
	/**
	 * Returns the cost matrix generated.
	 * <p>
	 * Careful, it can be <code>null</code> if not acceptable costs have been
	 * found for the specified configuration. In that case, the lists returned
	 * by {@link #getSourceList()} and {@link #getTargetList()} are empty.
	 * 
	 * @return a new {@link SparseCostMatrix} or <code>null</code>.
	 */	
	public static SparseCostMatrix getResult()
	{
		return scm;
	}

	
	public static List< TrackSegment > getSourceList()
	{
		return sourceList;
	}
	
	public static List< TrackSegment > getTargetList()
	{
		return targetList;
	}


	

	public double getAlternativeCostForSource( final TrackSegment source )
	{
		return alternativeCost;
	}
	
	public double getAlternativeCostForTarget( final TrackSegment target )
	{
		return alternativeCost;
	}
	
	public static double compute_shortest_distance(TrackSegment ts1,TrackSegment ts2,ArrayList<HashMap<TrackSegment,double[]>> backward_start,ArrayList<HashMap<TrackSegment,double[]>> forward_end,int initial_time,int last_time,double threshold,Integer time_window)
	{double min_distance = threshold+1;
	int last_time_1 = ts1.getLastDetection().getT();
	int ini_time_2 = ts2.getFirstDetection().getT();
	if ((last_time_1>ini_time_2)|(last_time_1+time_window<ini_time_2)){return Double.MAX_VALUE;}
	else{
		for (int t = last_time_1;t<ini_time_2;t++){
			double[] pos_last_t = forward_end.get(t-initial_time).get(ts1);
			double[] pos_ini_t = backward_start.get(last_time-t).get(ts2);
			double temp = Math.sqrt(Math.pow(pos_last_t[0]-pos_ini_t[0], 2)+Math.pow(pos_last_t[1]-pos_ini_t[1], 2));
			if (temp<min_distance){
			min_distance=temp;
			}
		}
	}
	if (min_distance<threshold){
		return min_distance;}
	else {return Double.MAX_VALUE;}				
	}
	
}
