/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.shape.utils;

import icy.gui.frame.progress.ProgressFrame;
import icy.image.IcyBufferedImage;
import icy.type.DataType;

import java.util.Collections;
import java.util.LinkedList;

import plugins.big.bigsnakeutils.icy.snake3D.Snake3DNode;
import vtk.vtkOBBTree;
import vtk.vtkPoints;
import vtk.vtkPolyData;

/**
 * Class that encapsulates some computational aspects of the conversion from
 * quad meshes to binary images.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class Binarizer3D implements Runnable {

	private final int z_;
	private ProgressFrame pFrame_ = null;
	private final int width_;
	private final int height_;
	private IcyBufferedImage maskImage_;
	private final double[] bounds_;
	private vtkOBBTree tree_ = null;

	public Binarizer3D(int width, int height, ProgressFrame pFrame, int z,
			vtkPolyData mesh, double[] bounds) {

		height_ = height;
		width_ = width;
		z_ = z;
		pFrame_ = pFrame;
		bounds_ = bounds;
		tree_ = new vtkOBBTree();
		tree_.SetDataSet(mesh);
		tree_.BuildLocator();

	}

	// ----------------------------------------------------------------------------

	@Override
	public void run() {
		maskImage_ = new IcyBufferedImage(width_, height_, 1, DataType.BYTE);

		byte[] binaryMaskX = rasterizeInX();
		byte[] binaryMaskY = rasterizeInY();
		byte[] binaryMask = new byte[width_ * height_];

		for (int i = 0; i < binaryMask.length; i++) {
			if (binaryMaskX[i] == Byte.MAX_VALUE
					&& binaryMaskY[i] == Byte.MAX_VALUE) {
				binaryMask[i] = Byte.MAX_VALUE;
			}
		}

		maskImage_.setDataXYAsByte(0, binaryMask);
		pFrame_.incPosition();
	}

	// ----------------------------------------------------------------------------

	private byte[] rasterizeInX() {
		byte[] binaryMask = new byte[width_ * height_];

		int y0 = Math.max((int) Math.floor(bounds_[2]), 0);
		int y1 = Math.min((int) Math.ceil(bounds_[3]), height_ - 1);

		for (int y = y0; y <= y1; y++) {
			double[] LineP0 = { 0, y, z_ };
			double[] LineP1 = { width_ - 1, y, z_ };

			vtkPoints IntersectPoints = new vtkPoints();
			tree_.IntersectWithLine(LineP0, LineP1, IntersectPoints, null);

			LinkedList<Snake3DNode> intersectionList = mutate(IntersectPoints);
			if (!intersectionList.isEmpty()) {
				Collections.sort(intersectionList);
				boolean in = false;
				Snake3DNode firstPoint = intersectionList.getFirst();
				Snake3DNode previousPoint = null;
				if (intersectionList.size() == 1) {
					binaryMask[(int) Math.ceil(firstPoint.x) + width_
							* ((int) Math.floor(firstPoint.y))] = 1;
				} else {
					for (Snake3DNode point : intersectionList) {
						if (point != firstPoint) {
							if (in) {
								for (int x = (int) Math.round(previousPoint.x); x <= Math
										.round(point.x); x++) {
									binaryMask[x + width_ * y] = Byte.MAX_VALUE;
								}
							}
						}
						previousPoint = point;
						in = !in;
					}
				}
			}
		}
		return binaryMask;
	}

	// ----------------------------------------------------------------------------

	private byte[] rasterizeInY() {
		byte[] binaryMask = new byte[width_ * height_];

		int x0 = Math.max((int) Math.floor(bounds_[0]), 0);
		int x1 = Math.min((int) Math.ceil(bounds_[1]), width_ - 1);

		for (int x = x0; x <= x1; x++) {
			double[] LineP0 = { x, 0, z_ };
			double[] LineP1 = { x, height_ - 1, z_ };

			vtkPoints IntersectPoints = new vtkPoints();
			tree_.IntersectWithLine(LineP0, LineP1, IntersectPoints, null);

			LinkedList<Snake3DNode> intersectionList = mutate(IntersectPoints);
			if (!intersectionList.isEmpty()) {
				Collections.sort(intersectionList);
				boolean in = false;
				Snake3DNode firstPoint = intersectionList.getFirst();
				Snake3DNode previousPoint = null;
				if (intersectionList.size() == 1) {
					binaryMask[(int) Math.ceil(firstPoint.x) + width_
							* ((int) Math.floor(firstPoint.y))] = 1;
				} else {
					for (Snake3DNode point : intersectionList) {
						if (point != firstPoint) {
							if (in) {
								for (int y = (int) Math.round(previousPoint.y); y <= Math
										.round(point.y); y++) {
									binaryMask[x + width_ * y] = Byte.MAX_VALUE;
								}
							}
						}
						previousPoint = point;
						in = !in;
					}
				}
			}
		}
		return binaryMask;
	}

	// ----------------------------------------------------------------------------

	private LinkedList<Snake3DNode> mutate(vtkPoints points) {
		LinkedList<Snake3DNode> intersectionList = new LinkedList<Snake3DNode>();
		for (int i = 0; i < points.GetNumberOfPoints(); i++) {
			double[] coords = points.GetPoint(i);
			intersectionList.add(new Snake3DNode(coords[0], coords[1],
					coords[2]));
		}
		return intersectionList;
	}

	// ----------------------------------------------------------------------------

	public IcyBufferedImage getMask() {
		return maskImage_;
	}
}
