package plugins.ylemontag.sequencecomparator.gui;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvasEvent;
import icy.canvas.IcyCanvasEvent.IcyCanvasEventType;
import icy.canvas.IcyCanvasListener;
import icy.gui.main.MainAdapter;
import icy.gui.main.MainEvent;
import icy.gui.viewer.Viewer;
import icy.gui.viewer.ViewerAdapter;
import icy.gui.viewer.ViewerEvent;
import icy.gui.viewer.ViewerEvent.ViewerEventType;
import icy.gui.viewer.ViewerListener;
import icy.main.Icy;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Stroke;
import java.lang.ref.WeakReference;
import java.util.HashSet;

import javax.swing.JTabbedPane;

import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.annotations.XYLineAnnotation;
import org.jfree.chart.plot.DefaultDrawingSupplier;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.data.UnknownKeyException;
import org.jfree.data.category.DefaultCategoryDataset;
import org.jfree.data.xy.XYSeries;
import org.jfree.data.xy.XYSeriesCollection;

import plugins.ylemontag.sequencecomparator.ErrorMeasure;

/**
 * 
 * @author Yoann Le Montagner
 *
 * Component displaying global comparison results
 */
public class ErrorMeasureComponent extends JTabbedPane
{
	private static final long serialVersionUID = 1L;
	
	private HashSet<ErrorMeasure> _data;
	private String _distanceName;
	private DefaultCategoryDataset _errorAllCollection;
	private XYSeriesCollection     _errorZCollection;
	private XYSeriesCollection     _errorTCollection;
	private DefaultCategoryDataset _errorCCollection;
	private JFreeChart _errorAllChart;
	private JFreeChart _errorZChart;
	private JFreeChart _errorTChart;
	private JFreeChart _errorCChart;
	private WeakReference<Viewer> _currentViewer;
	private ViewerListener _currentViewerListener;
	private WeakReference<IcyCanvas> _currentCanvas;
	private IcyCanvasListener _currentCanvasListener;
	private int _currentZ;
	private int _currentT;
	private XYLineAnnotation _cursorZ;
	private XYLineAnnotation _cursorT;
	
	/**
	 * Constructor
	 */
	public ErrorMeasureComponent()
	{
		super(JTabbedPane.TOP);
		_data = new HashSet<ErrorMeasure>();
		_distanceName = "";
		_errorAllCollection = new DefaultCategoryDataset();
		_errorZCollection   = new XYSeriesCollection    ();
		_errorTCollection   = new XYSeriesCollection    ();
		_errorCCollection   = new DefaultCategoryDataset();
		PlotOrientation poV = PlotOrientation.VERTICAL;
		_errorAllChart = ChartFactory.createBarChart   ("", null      , "Error", _errorAllCollection, poV, true, true, false);
		_errorZChart   = ChartFactory.createXYLineChart("", "Depth"   , "Error", _errorZCollection  , poV, true, true, false);
		_errorTChart   = ChartFactory.createXYLineChart("", "Time"    , "Error", _errorTCollection  , poV, true, true, false);
		_errorCChart   = ChartFactory.createBarChart   ("", "Channels", "Error", _errorCCollection  , poV, true, true, false);
		_errorZChart.getXYPlot().setDrawingSupplier(new CustomDrawingSupplier());
		_errorTChart.getXYPlot().setDrawingSupplier(new CustomDrawingSupplier());
		_currentZ = 0;
		_currentT = 0;
		updateTabVisibility("Overall error", _errorAllChart, true);
		Icy.getMainInterface().addListener(new MainAdapter()
		{	
			@Override
			public void viewerFocused(MainEvent event) {
				onFocusedViewerChanged((Viewer)event.getSource());
			}
		});
		onFocusedViewerChanged(Icy.getMainInterface().getFocusedViewer());
	}
	
	/**
	 * Current distance name
	 */
	public String getDistanceName()
	{
		return _distanceName;
	}
	
	/**
	 * Change the distance name
	 */
	public void setDistanceName(String distanceName)
	{
		_distanceName = distanceName;
		_errorAllChart.setTitle(distanceName);
		_errorZChart  .setTitle(distanceName);
		_errorTChart  .setTitle(distanceName);
		_errorCChart  .setTitle(distanceName);		
	}
	
	/**
	 * Add a new set of measure
	 */
	public void addMeasure(ErrorMeasure m)
	{
		_data.add(m);
		refresh0DSerie(_errorAllCollection, m, m.errorAll);
		refresh1DSerie(_errorZCollection  , m, m.errorZ  );
		refresh1DSerie(_errorTCollection  , m, m.errorT  );
		refresh1DSerie(_errorCCollection  , m, m.errorC  );
		updateMaxZ();
		updateMaxT();
		updateMaxC();
	}
	
	/**
	 * Remove a set of measure
	 */
	public void removeMeasure(ErrorMeasure m)
	{
		_data.remove(m);
		clean0DSerie(_errorAllCollection, m);
		clean1DSerie(_errorZCollection  , m);
		clean1DSerie(_errorTCollection  , m);
		clean1DSerie(_errorCCollection  , m);
		updateMaxZ();
		updateMaxT();
		updateMaxC();
	}
	
	/**
	 * Remove all the sets of measure
	 */
	public void removeAllMeasures()
	{
		_data.clear();
		_errorAllCollection.clear();
		_errorZCollection  .removeAllSeries();
		_errorTCollection  .removeAllSeries();
		_errorCCollection  .clear();
		updateMaxZ();
		updateMaxT();
		updateMaxC();
	}
	
	/**
	 * Action triggered when the focused viewer change
	 */
	private void onFocusedViewerChanged(Viewer newViewer)
	{
		if(_currentViewer!=null && _currentViewerListener!=null) {
			Viewer previousViewer = _currentViewer.get();
			if(previousViewer!=null) {
				previousViewer.removeListener(_currentViewerListener);
			}
		}
		_currentViewer         = null;
		_currentViewerListener = null;
		if(newViewer==null) {
			return;
		}
		_currentViewer         = new WeakReference<Viewer>(newViewer);
		_currentViewerListener = new ViewerAdapter()
		{	
			@Override
			public void viewerChanged(ViewerEvent event) {
				if(event.getType()==ViewerEventType.CANVAS_CHANGED) {
					onFocusedCanvasChanged(event.getSource());
				}
			}
		};
		newViewer.addListener(_currentViewerListener);
		onFocusedCanvasChanged(newViewer);
	}
	
	/**
	 * Action triggered when the focused canvas change
	 */
	private void onFocusedCanvasChanged(Viewer focusedViewer)
	{
		if(_currentCanvas!=null && _currentCanvasListener!=null) {
			IcyCanvas previousCanvas = _currentCanvas.get();
			if(previousCanvas!=null) {
				previousCanvas.removeCanvasListener(_currentCanvasListener);
			}
		}
		_currentCanvas         = null;
		_currentCanvasListener = null;
		if(focusedViewer==null) {
			return;
		}
		IcyCanvas newCanvas = focusedViewer.getCanvas();
		if(newCanvas==null) {
			return;
		}
		_currentCanvas         = new WeakReference<IcyCanvas>(newCanvas);
		_currentCanvasListener = new IcyCanvasListener()
		{
			@Override
			public void canvasChanged(IcyCanvasEvent event) {
				if(event.getType()==IcyCanvasEventType.POSITION_CHANGED) {
					onFocusedPositionChanged(event.getSource(), false);
				}
			}
		};
		newCanvas.addCanvasListener(_currentCanvasListener);
		onFocusedPositionChanged(newCanvas, true);
	}
	
	/**
	 * Action triggered when the position change on the focused canvas
	 */
	private void onFocusedPositionChanged(IcyCanvas focusedCanvas, boolean forceRefresh)
	{
		int newZ = focusedCanvas.getPositionZ();
		int newT = focusedCanvas.getPositionT();
		if(forceRefresh || _currentZ!=newZ) {
			_currentZ = newZ;
			moveCursorZ(newZ);
		}
		if(forceRefresh || _currentT!=newT) {
			_currentT = newT;
			moveCursorT(newT);
		}
	}

	/**
	 * Maximum extent in the Z direction
	 */
	private void updateMaxZ()
	{
		int currentMax = 0;
		for(ErrorMeasure m : _data) {
			currentMax = Math.max(currentMax, m.errorZ.length);
		}
		updateTabVisibility("Error Z", _errorZChart, currentMax>1);
	}
	
	/**
	 * Maximum extent in the T direction
	 */
	private void updateMaxT()
	{
		int currentMax = 0;
		for(ErrorMeasure m : _data) {
			currentMax = Math.max(currentMax, m.errorT.length);
		}
		updateTabVisibility("Error T", _errorTChart, currentMax>1);
	}
	
	/**
	 * Maximum extent in the C direction
	 */
	private void updateMaxC()
	{
		int currentMax = 0;
		for(ErrorMeasure m : _data) {
			currentMax = Math.max(currentMax, m.errorC.length);
		}
		updateTabVisibility("Error C", _errorCChart, currentMax>1);
	}
	
	/**
	 * Display or hide the tab corresponding to the given chart
	 */
	private void updateTabVisibility(String label, JFreeChart chart, boolean visible)
	{
		int previousTabIndex = indexOfTab(label);
		if(visible) {
			if(previousTabIndex<0) {
				ChartPanel newPanel = new ChartPanel(chart);
				addTab(label, newPanel);
			}
		}
		else {
			try {
				remove(previousTabIndex);
			}
			catch(IndexOutOfBoundsException err) {}
		}
	}
	
	/**
	 * Refresh the 0D serie corresponding to 'seq' in 'target'
	 */
	private void refresh0DSerie(DefaultCategoryDataset target, ErrorMeasure m, double data)
	{
		target.setValue(data, m, "");
	}
	
	/**
	 * Refresh the 1D serie corresponding to 'seq' in 'target'
	 */
	private void refresh1DSerie(XYSeriesCollection target, ErrorMeasure m, double[] data)
	{
		XYSeries serie;
		try {
			serie = target.getSeries(m);
		}
		catch(UnknownKeyException err) {
			serie = new XYSeries(m);
			target.addSeries(serie);
		}
		serie.clear();
		for(int k=0; k<data.length; ++k) {
			serie.add(k, data[k]);
		}
	}
	
	/**
	 * Refresh the 1D serie corresponding to 'seq' in 'target'
	 */
	private void refresh1DSerie(DefaultCategoryDataset target, ErrorMeasure m, double[] data)
	{
		for(int k=0; k<data.length; ++k) {
			target.setValue(data[k], m, "ch "+k);
		}
	}
	
	/**
	 * Remove a 0D serie corresponding to the sequence 'seq' from 'target'
	 */
	private void clean0DSerie(DefaultCategoryDataset target, ErrorMeasure m)
	{
		try {
			target.removeValue(m, "");
		}
		catch(UnknownKeyException err) {}
	}
	
	/**
	 * Remove a 1D serie corresponding to the sequence 'seq' from 'target'
	 */
	private void clean1DSerie(XYSeriesCollection target, ErrorMeasure m)
	{
		try {
			XYSeries serie = target.getSeries(m);
			target.removeSeries(serie);
		}
		catch(UnknownKeyException err) {}
	}
	
	/**
	 * Remove a 1D serie corresponding to the sequence 'seq' from 'target'
	 */
	private void clean1DSerie(DefaultCategoryDataset target, ErrorMeasure m)
	{
		try {
			target.removeRow(m);
		}
		catch(UnknownKeyException err) {}
	}
	
	/**
	 * Change the position of the current Z cursor
	 */
	private void moveCursorZ(int z)
	{
		if(_cursorZ!=null) {
			_errorZChart.getXYPlot().removeAnnotation(_cursorZ);
		}
		_cursorZ = makeCursor(z);
		_errorZChart.getXYPlot().addAnnotation(_cursorZ);
	}
	
	/**
	 * Change the position of the current T cursor
	 */
	private void moveCursorT(int t)
	{
		if(_cursorT!=null) {
			_errorTChart.getXYPlot().removeAnnotation(_cursorT);
		}
		_cursorT = makeCursor(t);
		_errorTChart.getXYPlot().addAnnotation(_cursorT);
	}
	
	/**
	 * Create a new vertical line
	 */
	private XYLineAnnotation makeCursor(int pos)
	{
		Stroke stroke = new BasicStroke(1, BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND, 1,
			new float[] {5, 4}, 0);
		return new XYLineAnnotation(pos, -1e10, pos, 1e10, stroke, Color.BLACK);
	}
	
	/**
	 * Drawing supplier
	 */
	private static class CustomDrawingSupplier extends DefaultDrawingSupplier
	{
		private static final long serialVersionUID = 1L;

		/**
		 * Constructor
		 */
		public CustomDrawingSupplier()
		{
			super();
		}
		
		/**
		 * Change the default stroke parameter
		 */
		public Stroke getNextStroke()
		{
			return new BasicStroke(2);
		}
	}
}
