/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.util.HashMap;

import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.border.TitledBorder;

/**
 * Panel that supports collapsing some elements.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class CollapsiblePanel extends JPanel {

	/**
	 * Determines if a de-serialized file is compatible with this class.
	 * 
	 * Maintainers must change this value if and only if the new version of this
	 * class is not compatible with old versions. See Sun docs for <a
	 * href=http://java.sun.com/products/jdk/1.1/docs/guide
	 * /serialization/spec/version.doc.html> details. </a>
	 * 
	 * Not necessary to include in first version of the class, but included here
	 * as a reminder of its importance.
	 */
	private static final long serialVersionUID = 6110247451353671228L;

	/** Title of the panel. */
	protected String title_;

	/** Maps that stores the visibility state of each component. */
	private final HashMap<Component, DetailPanelMode> visibility_;

	/** Detail mode in which the pane is (collapsed or expanded). */
	protected DetailPanelMode activeMode_;

	// ============================================================================
	// PUBLIC METHODS

	public CollapsiblePanel(String title) {
		title_ = title;
		activeMode_ = DetailPanelMode.BASICS;
		visibility_ = new HashMap<Component, DetailPanelMode>();

		setBorder(BorderFactory.createTitledBorder(title_));
		setLayout(new BorderLayout());
		addMouseListener(mouseListener);
	}

	// ----------------------------------------------------------------------------

	public void setTitle(String title) {
		firePropertyChange("title", title_, title_ = title);
	}

	// ----------------------------------------------------------------------------

	public Component add(Component comp, DetailPanelMode mode) {
		comp.addComponentListener(contentComponentListener);
		Component r = super.add(comp);
		visibility_.put(r, mode);
		updateBorderTitle();
		return r;
	}

	// ----------------------------------------------------------------------------

	public Component add(String name, Component comp, DetailPanelMode mode) {
		comp.addComponentListener(contentComponentListener);
		Component r = super.add(name, comp);
		visibility_.put(r, mode);
		updateBorderTitle();
		return r;
	}

	// ----------------------------------------------------------------------------

	public Component add(Component comp, int index, DetailPanelMode mode) {
		comp.addComponentListener(contentComponentListener);
		Component r = super.add(comp, index);
		visibility_.put(r, mode);
		updateBorderTitle();
		return r;
	}

	// ----------------------------------------------------------------------------

	public void add(Component comp, Object constraints, DetailPanelMode mode) {
		comp.addComponentListener(contentComponentListener);
		super.add(comp, constraints);
		visibility_.put(comp, mode);
		updateBorderTitle();
	}

	// ----------------------------------------------------------------------------

	public void add(Component comp, Object constraints, int index,
			DetailPanelMode mode) {
		comp.addComponentListener(contentComponentListener);
		super.add(comp, constraints, index);
		visibility_.put(comp, mode);
		updateBorderTitle();
	}

	// ----------------------------------------------------------------------------

	@Override
	public void remove(int index) {
		Component comp = getComponent(index);
		comp.removeComponentListener(contentComponentListener);
		visibility_.remove(comp);
		super.remove(index);
	}

	// ----------------------------------------------------------------------------

	@Override
	public void remove(Component comp) {
		comp.removeComponentListener(contentComponentListener);
		visibility_.remove(comp);
		super.remove(comp);
	}

	// ----------------------------------------------------------------------------

	@Override
	public void removeAll() {
		for (Component c : getComponents()) {
			c.removeComponentListener(contentComponentListener);
			visibility_.remove(c);
		}
		super.removeAll();
	}

	// ----------------------------------------------------------------------------

	protected void setVisibility(DetailPanelMode mode) {
		activeMode_ = mode;
		switch (mode) {
		case ADVANCED:
			for (Component c : getComponents()) {
				c.setVisible(true);
			}
			break;
		case BASICS:
			for (Component c : getComponents()) {
				DetailPanelMode componentMode = visibility_.get(c);
				switch (componentMode) {
				case BASICS:
					c.setVisible(true);
					break;
				case ADVANCED:
					c.setVisible(false);
					break;
				}
			}
			break;
		}
		updateBorderTitle();
	}

	// ----------------------------------------------------------------------------

	private void toggleVisibility() {
		switch (activeMode_) {
		case BASICS:
			setVisibility(DetailPanelMode.ADVANCED);
			break;
		case ADVANCED:
			setVisibility(DetailPanelMode.BASICS);
			break;
		}
		setVisibility(activeMode_);
	}

	// ----------------------------------------------------------------------------

	private void updateBorderTitle() {
		TitledBorder border = (TitledBorder) getBorder();
		String arrow = "";
		String tip = "";
		switch (activeMode_) {
		case BASICS:
			arrow = "[+]";
			tip = "(Basics)";
			break;
		case ADVANCED:
			arrow = "[-]";
			tip = "(Advanced)";
			break;
		}
		border.setTitle(arrow + " " + title_ + " " + tip);
		repaint();
	}

	// ----------------------------------------------------------------------------

	MouseListener mouseListener = new MouseAdapter() {
		@Override
		public void mouseClicked(MouseEvent e) {
			toggleVisibility();
		}
	};

	// ----------------------------------------------------------------------------

	ComponentListener contentComponentListener = new ComponentAdapter() {
		@Override
		public void componentShown(ComponentEvent e) {
			updateBorderTitle();
		}

		// ----------------------------------------------------------------------------

		@Override
		public void componentHidden(ComponentEvent e) {
			updateBorderTitle();
		}
	};

}
