/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.roi;

import icy.canvas.Canvas2D;
import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.painter.Anchor2D;
import icy.painter.PathAnchor2D;
import icy.sequence.Sequence;
import icy.type.point.Point5D;
import icy.util.XMLUtil;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Cursor;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.Shape;
import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Ellipse2D;
import java.awt.geom.Line2D;
import java.awt.geom.Path2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Point2D;
import java.util.ArrayList;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import plugins.big.bigsnakeutils.icy.snake2D.Snake2D;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DScale;
import plugins.big.shapedesigner.keeper.SplineCurveKeeper;
import plugins.big.shapedesigner.splinecurve.SplineCurve;
import plugins.kernel.canvas.VtkCanvas;
import plugins.kernel.roi.roi2d.ROI2DPath;

/**
 * Class that handles the interaction and the display of the spline curve.
 * 
 * @version May 3, 2014
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Ulugbek S. Kamilov (kamilov@gmail.com)
 */
public class SplineCurve2DROI extends ROI2DPath {

	/** Spline curve associated to the ROI. */
	private Snake2D splineCurve_ = null;
	/** Keeper associated to the ROI */
	private SplineCurveKeeper keeper_ = null;
	/** Group of all polylines forming the skin of the spline curve. */
	private Snake2DScale[] scales_ = null;

	// ----------------------------------------------------------------------------
	// STATUS FIELDS

	/** Denotes the current state of the spline curve for mouse interaction. */
	private SplineCurveEditMode splineCurveEditMode_ = SplineCurveEditMode.MOVE_SPLINE_CURVE;
	/** Flag that denotes if the spline curve can be modified. */
	private boolean isEditable_;

	// ----------------------------------------------------------------------------
	// COLORS

	/** Default ROI color for the spline curve. */
	private static final Color DEFAULT_NORMAL_COLOR = Color.RED;
	/** Default ROI color for the spline curve when selected. */
	private static final Color DEFAULT_SELECTED_COLOR = Color.GREEN;
	/** Default color for the control points of the middle axis. */
	private static final Color DEFAULT_SYMMETRY_COLOR = Color.BLUE;

	// ----------------------------------------------------------------------------
	// CURSORS

	/** Cursor shown on the screen when moving a control point. */
	private final Cursor moveControlPointCursor_ = Cursor
			.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR);
	/** Cursor shown on the screen when moving the spline curve. */
	private final Cursor moveSplineCurveCursor_ = Cursor
			.getPredefinedCursor(Cursor.HAND_CURSOR);
	/** Cursor shown on the screen when dragging the whole spline curve. */
	private final Cursor dragSplineCurveCursor_ = Cursor
			.getPredefinedCursor(Cursor.HAND_CURSOR);
	/** Cursor shown on the screen when rotating the whole spline curve. */
	private final Cursor rotateSplineCurveCursor_ = Cursor
			.getPredefinedCursor(Cursor.HAND_CURSOR);
	/** Cursor shown on the screen when dilating the whole spline curve. */
	private final Cursor dilateSplineCurveCursor_ = Cursor
			.getPredefinedCursor(Cursor.HAND_CURSOR);

	// ----------------------------------------------------------------------------
	// XML TAGS

	public static final String ID_SNAKE_PARAMETERS = "snake_parameters";

	// ----------------------------------------------------------------------------
	// OTHER

	/**
	 * If <code>true</code> the spline curve stay symmetric when the user move
	 * it with the mouse
	 */
	private boolean symmetric_ = false;

	// ============================================================================
	// SPLINECURVE2DROI METHODS

	// ----------------------------------------------------------------------------
	// PUBLIC METHODS

	/** Default constructor. */
	public SplineCurve2DROI() {
		super(new Path2D.Double());
	}

	// ----------------------------------------------------------------------------

	/** Default. */
	public SplineCurve2DROI(SplineCurve splineCurve, SplineCurveKeeper keeper) {
		super(new Path2D.Double());
		Path2D path = new Path2D.Double();
		Snake2DNode[] nodes = splineCurve.getNodes();
		for (int i = 0; i < nodes.length; i++) {
			if (i == 0) {
				path.moveTo(nodes[i].getX(), nodes[i].getY());
			} else {
				path.lineTo(nodes[i].getX(), nodes[i].getY());
			}
		}

		for (Anchor2D pt : getAnchorsFromShape(path)) {
			addPoint(pt);
		}
		scales_ = splineCurve.getScales();
		isEditable_ = true;
		splineCurve_ = splineCurve;
		keeper_ = keeper;
		setEditMode(SplineCurveEditMode.MOVE_SPLINE_CURVE);
		refreshColors();
	}

	// ----------------------------------------------------------------------------

	public void changePosition(double[] xPosition, double[] yPosition,
			Snake2DScale[] scales) {
		if (xPosition.length != controlPoints.size()
				|| yPosition.length != controlPoints.size()) {
			throw new IllegalArgumentException(
					"the number of modified controlPoints should be the same as the number of original control points");
		}
		int i = 0;
		for (Anchor2D anchor : controlPoints) {
			anchor.setPosition(xPosition[i], yPosition[i]);
			i++;
		}
		scales_ = scales;
	}

	// ----------------------------------------------------------------------------

	public Point2D getControlPoint(int i) {
		if (i < controlPoints.size()) {
			return new Point2D.Double(controlPoints.get(i).getX(),
					controlPoints.get(i).getY());
		} else {
			return null;
		}
	}

	// ----------------------------------------------------------------------------

	private void refreshColors() {
		if (symmetric_) {
			controlPoints.get(0).setColor(DEFAULT_SYMMETRY_COLOR);
			controlPoints.get(controlPoints.size() / 2).setColor(
					DEFAULT_SYMMETRY_COLOR);
		}
	}

	// ----------------------------------------------------------------------------

	public void refreshROI() {
		refreshScales();
		removeAllPoint();
		Path2D path = new Path2D.Double();
		Snake2DNode[] nodes = splineCurve_.getNodes();
		for (int i = 0; i < nodes.length; i++) {
			if (i == 0) {
				path.moveTo(nodes[i].getX(), nodes[i].getY());
			} else {
				path.lineTo(nodes[i].getX(), nodes[i].getY());
			}
		}
		for (Anchor2D pt : getAnchorsFromShape(path)) {
			addPoint(pt);
		}
		refreshColors();
	}

	// ----------------------------------------------------------------------------

	public void refreshScales() {
		scales_ = splineCurve_.getScales();
	}

	// ----------------------------------------------------------------------------

	public void setEditMode(SplineCurveEditMode editMode) {
		splineCurveEditMode_ = editMode;
		for (Anchor2D pt : controlPoints) {
			pt.setSelected(false);
		}
	}

	// ----------------------------------------------------------------------------

	public void setSymmetric(boolean symmetric) {
		symmetric_ = symmetric;
		refreshColors();
	}

	// ----------------------------------------------------------------------------
	// PRIVATE METHODS

	private ArrayList<Anchor2D> getAnchorsFromShape(Shape shape) {
		final PathIterator pathIt = shape.getPathIterator(null);
		final ArrayList<Anchor2D> result = new ArrayList<Anchor2D>();
		final double crd[] = new double[6];
		while (!pathIt.isDone()) {
			final int segType = pathIt.currentSegment(crd);
			PathAnchor2D pt = null;
			if (segType != PathIterator.SEG_CLOSE) {
				pt = (PathAnchor2D) createAnchor(new Point2D.Double(crd[0],
						crd[1]));
				pt.setType(segType);
				result.add(pt);
			}
			pathIt.next();
		}
		return result;
	}

	// ============================================================================
	// ROI2DPATH METHODS

	// ----------------------------------------------------------------------------
	// PUBLIC METHODS

	/** Gets the bounding rectangle of the first polyline of the spline curve. */
	@Override
	public Rectangle computeBounds2D() {
		if (scales_ != null) {
			if (scales_.length >= 1) {
				return scales_[0].getBounds2D().getBounds();
			}
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/** Get the ROI as a boolean bitmap mask for the specified rectangular area. */
	@Override
	public boolean[] getAsBooleanMask(int x, int y, int w, int h,
			boolean inclusive) {
		if (scales_ != null) {
			if (scales_.length >= 1) {
				Snake2DScale scale = scales_[0];
				final boolean[] result = new boolean[w * h];
				int offset = 0;
				for (int j = 0; j < h; j++) {
					for (int i = 0; i < w; i++) {
						result[offset] = scale.contains(x + i, y + j);
						if (inclusive) {
							result[offset] |= scale.intersects(x + i, y + j, 1,
									1);
						}
						offset++;
					}
				}
				return result;
			}
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/** Gets the color of the first polyline forming the skin of the snake. */
	@Override
	public Color getColor() {
		if (scales_ != null) {
			if (scales_.length >= 1) {
				return scales_[0].getColor();
			}
		}
		return DEFAULT_NORMAL_COLOR;
	}

	// ----------------------------------------------------------------------------

	/** Saves a snake from an XML node. */
	@Override
	public boolean saveToXML(Node node) {
		if (splineCurve_ != null) {
			Element snakeParametersElement = XMLUtil.setElement(node,
					ID_SNAKE_PARAMETERS);
			splineCurve_.saveToXML(snakeParametersElement);
		}
		return true;
	}

	// ----------------------------------------------------------------------------

	/** Sets the edit status of the ROI. */
	@Override
	public void setEditable(boolean isEditable) {
		isEditable_ = isEditable;
	}

	// ----------------------------------------------------------------------------
	// PROTECTED METHODS

	@Override
	protected Anchor2D createAnchor(Point2D pos) {
		return new ControlPointPainter(pos, DEFAULT_NORMAL_COLOR,
				DEFAULT_SELECTED_COLOR);
	}

	// ----------------------------------------------------------------------------

	@Override
	protected ROI2DShapePainter createPainter() {
		return new Snake2DPainter();
	}

	// ============================================================================
	// INNER CLASSES

	/**
	 * Class that handles the interaction and the display of the control points
	 * of the snake.
	 * 
	 * @version October 30, 2013
	 * 
	 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
	 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
	 * @author Ulugbek S. Kamilov (kamilov@gmail.com)
	 */
	private class ControlPointPainter extends PathAnchor2D {

		/** Size of the control point on the display. */
		private static final int CONTROL_POINT_SIZE = 8;

		// ============================================================================
		// PUBLIC METHODS

		/** Constructor. */
		public ControlPointPainter(Point2D pos, Color defaultSelectedColor,
				Color overColor) {
			super(pos.getX(), pos.getY(), CONTROL_POINT_SIZE,
					defaultSelectedColor, overColor);
		}

		// ----------------------------------------------------------------------------

		/** Draws the cross of the control point on the canvas. */
		@Override
		public void paint(Graphics2D g, Sequence sequence, IcyCanvas canvas) {
			if (canvas instanceof IcyCanvas2D) {
				final double adjRayX = canvas.canvasToImageLogDeltaX(ray);
				final double adjRayY = canvas.canvasToImageLogDeltaY(ray);
				double stroke = getAdjustedStroke(canvas);
				g.setStroke(new BasicStroke((float) stroke));
				if (selected) {
					g.setColor(selectedColor);
				} else {
					g.setColor(color);
				}
				g.draw(new Line2D.Double(position.x - adjRayX, position.y,
						position.x + adjRayX, position.y));
				g.draw(new Line2D.Double(position.x, position.y - adjRayY,
						position.x, position.y + adjRayY));
			} else {
				System.err.println("Unrecognized IcyCanvas.");
			}
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Class that draws the ROI forming the snake.
	 * 
	 * @version November 13, 2013
	 * 
	 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
	 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
	 * @author Ulugbek S. Kamilov (kamilov@gmail.com)
	 */
	private class Snake2DPainter extends ROI2DShapePainter {

		/** Point in image coordinates where the is clicking clicked. */
		private Point5D.Double pressedImagePoint_ = null;
		/** Point in image coordinates where the is clicking clicked. */
		private Point2D firstControlPointInitPosition_ = null;
		/** Point with the center of gravity of the snake curve. */
		private Point2D splineCurveMassCenter_ = null;

		/** Flag to determine if the mouse pointer is within the snake curve. */
		private boolean isMouseInROI_ = false;

		// ============================================================================
		// PUBLIC METHODS

		/**
		 * Invoked when a key is pressed. This method is used to block the key
		 * events from Icy.
		 */
		@Override
		public void keyPressed(KeyEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			e.consume();
		}

		// ----------------------------------------------------------------------------

		/**
		 * Invoked when the mouse cursor has been moved onto the Icy canvas but
		 * no buttons have been pushed.
		 */
		@Override
		public void mouseMove(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (isEditable_) {
				switch (splineCurveEditMode_) {
				case DILATE_SPLINE_CURVE:
					mouseMoveDilate(e, imagePoint, canvas);
					break;
				case ROTATE_SPLINE_CURVE:
					mouseMoveRotate(e, imagePoint, canvas);
					break;
				default:
					mouseMoveDefault(e, imagePoint, canvas);
					break;
				}
			}
		}

		// ----------------------------------------------------------------------------

		/** Invoked when a mouse button has been pressed on the Icy canvas. */
		@Override
		public void mousePressed(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			pressedImagePoint_ = imagePoint;
			if (isEditable_) {
				boolean isMouseOnControlPoint = false;
				for (Anchor2D pt : controlPoints) {
					if (pt.isOver(canvas, imagePoint.getX(), imagePoint.getY())) {
						isMouseOnControlPoint = true;
					}
				}
				isMouseInROI_ = SplineCurve2DROI.this.contains(imagePoint);
				if (!(isMouseInROI_ && imagePoint.equals(pressedImagePoint_))
						&& !isMouseOnControlPoint) {
					keeper_.deactivateSplineCurve();
				}
				switch (splineCurveEditMode_) {
				case DILATE_SPLINE_CURVE:
					mousePressedDilate(e, imagePoint, canvas);
					break;
				case ROTATE_SPLINE_CURVE:
					mousePressedRotate(e, imagePoint, canvas);
					break;
				default:
					mousePressedDefault(e, imagePoint, canvas);
					break;
				}
			}
		}

		// ----------------------------------------------------------------------------

		/**
		 * Invoked when a mouse button is pressed on the Icy canvas and then
		 * dragged.
		 */
		@Override
		public void mouseDrag(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (isEditable_) {
				switch (splineCurveEditMode_) {
				case DILATE_SPLINE_CURVE:
					mouseDragDilate(e, imagePoint, canvas);
					break;
				case ROTATE_SPLINE_CURVE:
					mouseDragRotate(e, imagePoint, canvas);
					break;
				default:
					mouseDragDefault(e, imagePoint, canvas);
					break;
				}
			}
		}

		// ----------------------------------------------------------------------------

		/** Invoked when a mouse button has been pressed on the Icy canvas. */
		@Override
		public void mouseReleased(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isEditable_) {
				if (!isActiveFor(canvas)) {
					return;
				}
				SplineCurve2DROI.this.beginUpdate();
				try {
					// First check if the mouse is inside the ROI
					isMouseInROI_ = SplineCurve2DROI.this.contains(imagePoint);
					if (isMouseInROI_ && imagePoint.equals(pressedImagePoint_)) {
						keeper_.activateSplineCurve();
					}
				} finally {
					SplineCurve2DROI.this.endUpdate();
				}
			}
		}

		// ============================================================================
		// PROTECTED METHODS

		/**
		 * Draws on the image canvas all the visual representation of the snake
		 * (polylines, control points, etc.).
		 */
		@Override
		protected void drawROI(Graphics2D g, Sequence sequence, IcyCanvas canvas) {
			if (canvas instanceof IcyCanvas2D) {
				final Graphics2D g2 = (Graphics2D) g.create();
				if (scales_ != null) {
					double stroke = getAdjustedStroke(canvas);
					g2.setStroke(new BasicStroke((float) (stroke / 2)));
					int imax = isEditable_ ? scales_.length : 1;
					for (int i = 0; i < imax; i++) {
						g2.setColor(scales_[i].getColor());
						g2.draw(scales_[i]);
					}
				}
				if ((SplineCurve2DROI.this.splineCurveEditMode_ == SplineCurveEditMode.DILATE_SPLINE_CURVE || SplineCurve2DROI.this.splineCurveEditMode_ == SplineCurveEditMode.ROTATE_SPLINE_CURVE)
						&& isMouseInROI_
						&& (splineCurveMassCenter_ != null)
						&& (firstControlPointInitPosition_ != null)) {
					g2.setColor(Color.GREEN);
					final double sizeX = canvas.canvasToImageLogDeltaX(5);
					final double sizeY = canvas.canvasToImageLogDeltaX(5);
					g2.fill(new Ellipse2D.Double(splineCurveMassCenter_.getX()
							- sizeX / 2, splineCurveMassCenter_.getY() - sizeY
							/ 2, sizeX, sizeY));
				}
				if (isEditable_) {
					for (Anchor2D pt : controlPoints) {
						pt.paint(g2, sequence, canvas);
					}
				}
				g2.dispose();
			} else if (canvas instanceof VtkCanvas) {
				super.drawROI(g, sequence, canvas);
			}
		}

		// ============================================================================
		// PRIVATE METHODS

		/** Default action when mouse cursor has been moved onto the Icy canvas. */
		private void mouseMoveDefault(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				isMouseInROI_ = SplineCurve2DROI.this.contains(imagePoint);
				if (!isMouseInROI_) {
					for (Snake2DScale scale : SplineCurve2DROI.this.scales_) {
						isMouseInROI_ = scale.contains(imagePoint.getX(),
								imagePoint.getY());
						if (isMouseInROI_) {
							break;
						}
					}
				}
				for (Anchor2D pt : controlPoints) {
					pt.setSelected(false);
					if (pt.isOver(canvas, imagePoint.getX(), imagePoint.getY())) {
						((Canvas2D) canvas).getViewComponent().setCursor(
								moveControlPointCursor_);
						isMouseInROI_ = false;
					}
				}
				if (isMouseInROI_) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							moveSplineCurveCursor_);
					for (Anchor2D pt : controlPoints) {
						pt.setSelected(true);
					}
				} else {
					for (Anchor2D pt : controlPoints) {
						pt.mouseMove(e, imagePoint, canvas);
					}
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		private void mouseMoveDilate(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				isMouseInROI_ = SplineCurve2DROI.this.contains(imagePoint);
				if (!isMouseInROI_) {
					for (Snake2DScale scale : SplineCurve2DROI.this.scales_) {
						isMouseInROI_ = scale.contains(imagePoint.getX(),
								imagePoint.getY());
						if (isMouseInROI_) {
							break;
						}
					}
				}
				if (isMouseInROI_) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							dilateSplineCurveCursor_);
					for (Anchor2D pt : controlPoints) {
						pt.setSelected(true);
					}
				} else {
					for (Anchor2D pt : controlPoints) {
						pt.setSelected(false);
					}
					splineCurveMassCenter_ = null;
					firstControlPointInitPosition_ = null;
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		private void mouseMoveRotate(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				isMouseInROI_ = SplineCurve2DROI.this.contains(imagePoint);
				if (!isMouseInROI_) {
					for (Snake2DScale scale : SplineCurve2DROI.this.scales_) {
						isMouseInROI_ = scale.contains(imagePoint.getX(),
								imagePoint.getY());
						if (isMouseInROI_) {
							break;
						}
					}
				}
				if (isMouseInROI_) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							rotateSplineCurveCursor_);
					for (Anchor2D pt : controlPoints) {
						pt.setSelected(true);
					}
				} else {
					for (Anchor2D pt : controlPoints) {
						pt.setSelected(false);
					}
					splineCurveMassCenter_ = null;
					firstControlPointInitPosition_ = null;
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		/**
		 * Default actions when a mouse button has been pressed on the Icy
		 * canvas.
		 */
		private void mousePressedDefault(MouseEvent e,
				Point5D.Double imagePoint, IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				if (isMouseInROI_) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							dragSplineCurveCursor_);
					e.consume();
				} else {
					for (Anchor2D pt : controlPoints) {
						pt.mousePressed(e, imagePoint, canvas);
					}
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		private void mousePressedDilate(MouseEvent e,
				Point5D.Double imagePoint, IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				if (isMouseInROI_) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							dilateSplineCurveCursor_);
					splineCurveMassCenter_ = splineCurve_.getCentroid();
					firstControlPointInitPosition_ = (Point2D) SplineCurve2DROI.this
							.getControlPoint(0).clone();
					e.consume();
				} else {
					splineCurveMassCenter_ = null;
					firstControlPointInitPosition_ = null;
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		private void mousePressedRotate(MouseEvent e,
				Point5D.Double imagePoint, IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				if (isMouseInROI_) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							rotateSplineCurveCursor_);
					splineCurveMassCenter_ = splineCurve_.getCentroid();
					firstControlPointInitPosition_ = (Point2D) SplineCurve2DROI.this
							.getControlPoint(0).clone();
					e.consume();
				} else {
					splineCurveMassCenter_ = null;
					firstControlPointInitPosition_ = null;
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		/**
		 * Default action when a mouse button is pressed on the Icy canvas and
		 * then dragged.
		 */
		private void mouseDragDefault(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				for (Anchor2D pt : controlPoints) {
					if (pt.isOver(canvas, imagePoint.getX(), imagePoint.getY())) {
						((Canvas2D) canvas).getViewComponent().setCursor(
								moveControlPointCursor_);
					}
				}
				if (isMouseInROI_ && pressedImagePoint_ != null) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							dragSplineCurveCursor_);
					e.consume();
					double dx = imagePoint.getX() - pressedImagePoint_.getX();
					double dy = imagePoint.getY() - pressedImagePoint_.getY();
					pressedImagePoint_ = imagePoint;
					SplineCurve2DROI.this.translate(dx, dy);
				} else {
					for (Anchor2D pt : controlPoints) {
						if (pt.isSelected()) {
							if (symmetric_) {
								int n = controlPoints.size();
								int i = controlPoints.indexOf(pt);

								if (i != 0 && i != n / 2) {
									pt.mouseDrag(e, imagePoint, canvas);

									Point2D pI = controlPoints.get(i)
											.getPosition();
									Point2D p0 = controlPoints.get(0)
											.getPosition();
									Point2D p1 = controlPoints.get(n / 2)
											.getPosition();

									Point2D.Double axe = new Point2D.Double(
											p1.getX() - p0.getX(), p1.getY()
													- p0.getY());
									Point2D.Double OI = new Point2D.Double(
											pI.getX() - p0.getX(), pI.getY()
													- p0.getY());

									double norm2 = axe.getX() * axe.getX()
											+ axe.getY() * axe.getY();
									double scalar_product = axe.getX()
											* OI.getX() + axe.getY()
											* OI.getY();

									controlPoints.get(n - i).setPosition(
											p0.getX() + 2. / norm2 * axe.getX()
													* scalar_product
													- OI.getX(),
											p0.getY() + 2. / norm2 * axe.getY()
													* scalar_product
													- OI.getY());
								} else {
									Point2D p0_old = controlPoints.get(i)
											.getPosition();
									pt.mouseDrag(e, imagePoint, canvas);

									Point2D p0;
									Point2D p1;

									if (i == 0) {
										p0 = controlPoints.get(0).getPosition();
										p1 = controlPoints.get(n / 2)
												.getPosition();
									} else {
										p1 = controlPoints.get(0).getPosition();
										p0 = controlPoints.get(n / 2)
												.getPosition();
									}

									Point2D.Double axe_old = new Point2D.Double(
											p0_old.getX() - p1.getX(),
											p0_old.getY() - p1.getY());
									Point2D.Double axe = new Point2D.Double(
											p0.getX() - p1.getX(), p0.getY()
													- p1.getY());

									double norm = Math.sqrt(axe_old.getX()
											* axe_old.getX() + axe_old.getY()
											* axe_old.getY());
									axe_old.setLocation(axe_old.getX() / norm,
											axe_old.getY() / norm);
									norm = Math.sqrt(axe.getX() * axe.getX()
											+ axe.getY() * axe.getY());
									axe.setLocation(axe.getX() / norm,
											axe.getY() / norm);

									double scalar_product = axe.getX()
											* axe_old.getX() + axe.getY()
											* axe_old.getY();
									double S = axe_old.getX() * axe.getY()
											- axe_old.getY() * axe.getX();
									double alpha = Math.signum(S)
											* Math.acos(scalar_product);

									double cosa = Math.cos(alpha);
									double sina = Math.sin(alpha);

									for (int k = 1; k < n / 2; k++) {
										Point2D pk = controlPoints.get(k)
												.getPosition();
										Point2D.Double vec = new Point2D.Double(
												pk.getX() - p1.getX(),
												pk.getY() - p1.getY());
										vec.setLocation(
												cosa * vec.getX() - sina
														* vec.getY(),
												sina * vec.getX() + cosa
														* vec.getY());
										controlPoints.get(k).setPosition(
												p1.getX() + vec.getX(),
												p1.getY() + vec.getY());
									}
									for (int k = n / 2 + 1; k < n; k++) {
										Point2D pk = controlPoints.get(k)
												.getPosition();
										Point2D.Double vec = new Point2D.Double(
												pk.getX() - p1.getX(),
												pk.getY() - p1.getY());
										vec.setLocation(
												cosa * vec.getX() - sina
														* vec.getY(),
												sina * vec.getX() + cosa
														* vec.getY());
										controlPoints.get(k).setPosition(
												p1.getX() + vec.getX(),
												p1.getY() + vec.getY());
									}
								}
							} else {
								pt.mouseDrag(e, imagePoint, canvas);
							}
						}
					}
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		private void mouseDragDilate(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				if (isMouseInROI_ && pressedImagePoint_ != null
						&& splineCurveMassCenter_ != null
						&& !controlPoints.isEmpty()) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							dilateSplineCurveCursor_);
					double distInit = (splineCurveMassCenter_.getX() - pressedImagePoint_
							.getX())
							* (splineCurveMassCenter_.getX() - pressedImagePoint_
									.getX())
							+ (splineCurveMassCenter_.getY() - pressedImagePoint_
									.getY())
							* (splineCurveMassCenter_.getY() - pressedImagePoint_
									.getY());
					double distCurrent = (splineCurveMassCenter_.getX() - imagePoint
							.getX())
							* (splineCurveMassCenter_.getX() - imagePoint
									.getX())
							+ (splineCurveMassCenter_.getY() - imagePoint
									.getY())
							* (splineCurveMassCenter_.getY() - imagePoint
									.getY());
					if (distInit == 0) {
						return;
					}
					double mouseDilationFactor = distCurrent / distInit;
					Point2D firstPoint = SplineCurve2DROI.this
							.getControlPoint(0);
					double snakeDilationFactor = ((splineCurveMassCenter_
							.getX() - firstPoint.getX())
							* (splineCurveMassCenter_.getX() - firstPoint
									.getX()) + (splineCurveMassCenter_.getY() - firstPoint
							.getY())
							* (splineCurveMassCenter_.getY() - firstPoint
									.getY()))
							/ ((splineCurveMassCenter_.getX() - firstControlPointInitPosition_
									.getX())
									* (splineCurveMassCenter_.getX() - firstControlPointInitPosition_
											.getX()) + (splineCurveMassCenter_
									.getY() - firstControlPointInitPosition_
									.getY())
									* (splineCurveMassCenter_.getY() - firstControlPointInitPosition_
											.getY()));
					double dilationFactor = Math.sqrt(mouseDilationFactor
							/ snakeDilationFactor);
					for (Anchor2D anchor : controlPoints) {
						Point2D p = anchor.getPosition();
						anchor.setPosition(
								splineCurveMassCenter_.getX()
										+ dilationFactor
										* (p.getX() - splineCurveMassCenter_
												.getX()),
								splineCurveMassCenter_.getY()
										+ dilationFactor
										* (p.getY() - splineCurveMassCenter_
												.getY()));
					}
					e.consume();
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}

		// ----------------------------------------------------------------------------

		private void mouseDragRotate(MouseEvent e, Point5D.Double imagePoint,
				IcyCanvas canvas) {
			if (!isActiveFor(canvas)) {
				return;
			}
			SplineCurve2DROI.this.beginUpdate();
			try {
				if (isMouseInROI_ && (pressedImagePoint_ != null)
						&& (splineCurveMassCenter_ != null)
						&& !controlPoints.isEmpty()) {
					((Canvas2D) canvas).getViewComponent().setCursor(
							rotateSplineCurveCursor_);
					double distMouse = Math
							.sqrt((splineCurveMassCenter_.getX() - imagePoint
									.getX())
									* (splineCurveMassCenter_.getX() - imagePoint
											.getX())
									+ (splineCurveMassCenter_.getY() - imagePoint
											.getY())
									* (splineCurveMassCenter_.getY() - imagePoint
											.getY()));
					if (distMouse == 0) {
						return;
					}
					double cosAngMouse = (imagePoint.getX() - splineCurveMassCenter_
							.getX()) / distMouse;
					double sinAngMouse = (imagePoint.getY() - splineCurveMassCenter_
							.getY()) / distMouse;
					double distMouseInit = Math
							.sqrt((splineCurveMassCenter_.getX() - pressedImagePoint_
									.getX())
									* (splineCurveMassCenter_.getX() - pressedImagePoint_
											.getX())
									+ (splineCurveMassCenter_.getY() - pressedImagePoint_
											.getY())
									* (splineCurveMassCenter_.getY() - pressedImagePoint_
											.getY()));
					if (distMouseInit == 0) {
						return;
					}
					double cosAngMouseInit = (pressedImagePoint_.getX() - splineCurveMassCenter_
							.getX()) / distMouseInit;
					double sinAngMouseInit = (pressedImagePoint_.getY() - splineCurveMassCenter_
							.getY()) / distMouseInit;
					double cosDeltaAngMouse = cosAngMouse * cosAngMouseInit
							+ sinAngMouseInit * sinAngMouse;
					double sinDeltaAngMouse = sinAngMouse * cosAngMouseInit
							- sinAngMouseInit * cosAngMouse;
					double distFirstPointInit = Math
							.sqrt((splineCurveMassCenter_.getX() - firstControlPointInitPosition_
									.getX())
									* (splineCurveMassCenter_.getX() - firstControlPointInitPosition_
											.getX())
									+ (splineCurveMassCenter_.getY() - firstControlPointInitPosition_
											.getY())
									* (splineCurveMassCenter_.getY() - firstControlPointInitPosition_
											.getY()));
					if (distFirstPointInit == 0) {
						return;
					}
					double cosAngPointInit = (firstControlPointInitPosition_
							.getX() - splineCurveMassCenter_.getX())
							/ distFirstPointInit;
					double sinAngPointInit = (firstControlPointInitPosition_
							.getY() - splineCurveMassCenter_.getY())
							/ distFirstPointInit;
					Point2D firstPoint = controlPoints.get(0).getPosition();
					double distFirstPoint = Math.sqrt((splineCurveMassCenter_
							.getX() - firstPoint.getX())
							* (splineCurveMassCenter_.getX() - firstPoint
									.getX())
							+ (splineCurveMassCenter_.getY() - firstPoint
									.getY())
							* (splineCurveMassCenter_.getY() - firstPoint
									.getY()));
					if (distFirstPoint == 0) {
						return;
					}
					double cosAngPoint = (firstPoint.getX() - splineCurveMassCenter_
							.getX()) / distFirstPoint;
					double sinAngPoint = (firstPoint.getY() - splineCurveMassCenter_
							.getY()) / distFirstPoint;
					double cosDeltaAngPoint = cosAngPoint * cosAngPointInit
							+ sinAngPointInit * sinAngPoint;
					double sinDeltaAngPoint = sinAngPoint * cosAngPointInit
							- sinAngPointInit * cosAngPoint;
					double cosRotAngle = cosDeltaAngMouse * cosDeltaAngPoint
							+ sinDeltaAngMouse * sinDeltaAngPoint;
					double sinRotAngle = sinDeltaAngMouse * cosDeltaAngPoint
							- sinDeltaAngPoint * cosDeltaAngMouse;
					for (Anchor2D anchor : controlPoints) {
						Point2D p = anchor.getPosition();
						anchor.setPosition(
								splineCurveMassCenter_.getX()
										+ cosRotAngle
										* (p.getX() - splineCurveMassCenter_
												.getX())
										- sinRotAngle
										* (p.getY() - splineCurveMassCenter_
												.getY()),
								splineCurveMassCenter_.getY()
										+ sinRotAngle
										* (p.getX() - splineCurveMassCenter_
												.getX())
										+ cosRotAngle
										* (p.getY() - splineCurveMassCenter_
												.getY()));
					}
					e.consume();
				}
			} finally {
				SplineCurve2DROI.this.endUpdate();
			}
		}
	}
}
