/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.gui.curve;

import java.awt.Dimension;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.awt.event.FocusListener;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.event.MouseMotionListener;
import java.awt.geom.Point2D;

import javax.swing.JComponent;

import plugins.adufour.vars.gui.VarEditor;

/**
 * VarEditor for Curve objects. Curves are defined by control points. The user
 * can add a control point by left clicking, remove a control point by right
 * clicking and drag a selected point to edit the curve. It's not possible to
 * delete the first and the last point
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 */
public class CurveEditor extends VarEditor<Curve> {
	private MouseListener mouseListener_;
	private MouseMotionListener mouseMotionListener_;
	private FocusListener focusListener_;

	/** Default constructor. */
	public CurveEditor(VarCurve variable) {
		super(variable);
	}

	@Override
	public JComponent getEditorComponent() {
		return (JComponent) super.getEditorComponent();
	}

	@Override
	public Dimension getPreferredSize() {
		return getEditorComponent().getPreferredSize();
	}

	@Override
	public void setComponentToolTipText(String s) {
		getEditorComponent().setToolTipText(s);
	}

	@Override
	public void dispose() {
		super.dispose();
	}

	@Override
	protected JComponent createEditorComponent() {
		final CurveCanvas canvas = new CurveCanvas(variable.getValue().clone());

		mouseListener_ = new MouseListener() {
			@Override
			public void mouseClicked(MouseEvent arg0) {
				int currentIndex = canvas.indexAt(arg0.getX(), arg0.getY());
				canvas.setCurrentIndex(currentIndex);

				// Right click on a control -> delete the control (except if
				// it's the first or the last)
				if (arg0.getButton() != MouseEvent.BUTTON1
						&& currentIndex != -1) {
					int index = canvas.indexAt(arg0.getX(), arg0.getY());
					if (index > 0 && index < canvas.getCurve().count() - 1) {
						canvas.getCurve().removeControlPoint(index);
						canvas.setCurrentIndex(-1);
						updateVariableValue();
					}
				}
			}

			@Override
			public void mouseEntered(MouseEvent arg0) {
				canvas.setShowControls(true);
			}

			@Override
			public void mouseExited(MouseEvent arg0) {
				if (canvas.getCurrentIndex() != -1) {
					return;
				}

				canvas.setShowControls(false);
			}

			@Override
			public void mousePressed(MouseEvent arg0) {
				int currentIndex = canvas.indexAt(arg0.getX(), arg0.getY());
				canvas.setCurrentIndex(currentIndex);

				// Left click on an empty area -> add a control
				if (arg0.getButton() == MouseEvent.BUTTON1
						&& currentIndex == -1) {
					int index = 0;
					Point2D pt0 = canvas.mapToCurveCoordinate(arg0.getX(),
							arg0.getY());

					for (Point2D pt : canvas.getCurve().getControlPoints()) {
						if (pt0.getX() > pt.getX()) {
							index++;
						} else {
							break;
						}
					}

					canvas.getCurve().insertControlPoint(index, pt0);
					canvas.setCurrentIndex(index);
					updateVariableValue();
				}
			}

			@Override
			public void mouseReleased(MouseEvent arg0) {
				canvas.setCurrentIndex(-1);
			}

		};

		mouseMotionListener_ = new MouseMotionListener() {
			@Override
			public void mouseDragged(MouseEvent arg0) {
				int currentIndex = canvas.getCurrentIndex();
				if (currentIndex == -1) {
					return;
				}

				Point2D pt = canvas.mapToCurveCoordinate(arg0.getX(),
						arg0.getY());

				// The first and last point cannot be dragged inside the box
				if (currentIndex == 0) {
					pt.setLocation(0, pt.getY());
				} else if (currentIndex == canvas.getCurve().count() - 1) {
					pt.setLocation(1, pt.getY());
				} else {
					Point2D before = canvas.getCurve().getControlPoint(
							currentIndex - 1);
					Point2D after = canvas.getCurve().getControlPoint(
							currentIndex + 1);
					pt.setLocation(Math.max(
							Math.min(pt.getX(), after.getX() - 0.001),
							before.getX() + 0.001), pt.getY());
				}

				canvas.getCurve().setControlPoint(currentIndex, pt);
				updateVariableValue();
			}

			@Override
			public void mouseMoved(MouseEvent arg0) {
			}

		};

		focusListener_ = new FocusAdapter() {
			@Override
			public void focusLost(FocusEvent e) {
				updateVariableValue();
			}
		};

		return canvas;
	}

	@Override
	protected void setEditorEnabled(boolean enabled) {
		getEditorComponent().setEnabled(enabled);
	}

	/** Update the variable value when the curve is edited. */
	protected void updateVariableValue() {
		variable.setValue(((CurveCanvas) getEditorComponent()).getCurve()
				.clone());
	}

	@Override
	protected void updateInterfaceValue() {
		((CurveCanvas) getEditorComponent()).setCurve(variable.getValue()
				.clone());
	}

	@Override
	protected void activateListeners() {
		((CurveCanvas) getEditorComponent()).addMouseListener(mouseListener_);
		((CurveCanvas) getEditorComponent())
				.addMouseMotionListener(mouseMotionListener_);
		((CurveCanvas) getEditorComponent()).addFocusListener(focusListener_);
	}

	@Override
	protected void deactivateListeners() {
		((CurveCanvas) getEditorComponent())
				.removeMouseListener(mouseListener_);
		((CurveCanvas) getEditorComponent())
				.removeMouseMotionListener(mouseMotionListener_);
		((CurveCanvas) getEditorComponent())
				.removeFocusListener(focusListener_);
	}

	@Override
	public boolean isComponentEnabled() {
		return true;
	}

	@Override
	public boolean isComponentOpaque() {
		return true;
	}
}
