/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.gui;

import icy.gui.frame.progress.AnnounceFrame;

import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.GridLayout;
import java.awt.Insets;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import plugins.big.bigsnakeutils.icy.gui.CollapsiblePanel;
import plugins.big.bigsnakeutils.icy.gui.DetailPanelMode;
import plugins.big.shapedesigner.ShapeDesigner;
import plugins.big.shapedesigner.splinecurve.SplineCurveParameters;

/**
 * Panel in which the user specifies the parameters of the spline curve.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class SplineCurveSettingsPane extends CollapsiblePanel implements
		ChangeListener, ItemListener {

	/**
	 * Determines if a de-serialized file is compatible with this class.
	 * 
	 * Maintainers must change this value if and only if the new version of this
	 * class is not compatible with old versions. See Sun docs for <a
	 * href=http://java.sun.com/products/jdk/1.1/docs/guide
	 * /serialization/spec/version.doc.html> details. </a>
	 * 
	 * Not necessary to include in first version of the class, but included here
	 * as a reminder of its importance.
	 */
	private static final long serialVersionUID = 6164479559453994234L;

	private final JSpinner numControlPointsSpinner_ = new JSpinner();
	private final JCheckBox symmetricCheckbox_ = new JCheckBox("Symmetric");

	private final JLabel numControlPointsLabel_ = new JLabel("Control points");

	private ShapeDesigner shapeDesigner_ = null;

	private final SpinnerNumberModel spinnerNumberModel_;

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public SplineCurveSettingsPane(String title, ShapeDesigner shapeDesigner) {
		super(title);
		shapeDesigner_ = shapeDesigner;

		GridBagLayout gridBagLayout = new GridBagLayout();
		gridBagLayout.columnWidths = new int[] { 120, 150, 0 };
		gridBagLayout.rowHeights = new int[] { 27, 23, 0 };
		gridBagLayout.columnWeights = new double[] { 0.0, 1.0, Double.MIN_VALUE };
		gridBagLayout.rowWeights = new double[] { 0.0, 0.0, Double.MIN_VALUE };
		setLayout(gridBagLayout);

		JPanel numControlPointsLabelPanel = new JPanel();
		GridBagConstraints gbc_numControlPointsLabelPanel = new GridBagConstraints();
		gbc_numControlPointsLabelPanel.anchor = GridBagConstraints.EAST;
		gbc_numControlPointsLabelPanel.insets = new Insets(0, 0, 5, 5);
		gbc_numControlPointsLabelPanel.fill = GridBagConstraints.VERTICAL;
		gbc_numControlPointsLabelPanel.gridx = 0;
		gbc_numControlPointsLabelPanel.gridy = 0;
		add(numControlPointsLabelPanel, gbc_numControlPointsLabelPanel,
				DetailPanelMode.BASICS);
		numControlPointsLabelPanel.setLayout(new GridLayout(0, 1, 0, 0));

		numControlPointsLabelPanel.add(numControlPointsLabel_);

		JPanel numControlPointsSpinnerPanel = new JPanel();
		GridBagConstraints gbc_numControlPointsSpinnerPanel = new GridBagConstraints();
		gbc_numControlPointsSpinnerPanel.insets = new Insets(0, 0, 5, 0);
		gbc_numControlPointsSpinnerPanel.fill = GridBagConstraints.BOTH;
		gbc_numControlPointsSpinnerPanel.gridx = 1;
		gbc_numControlPointsSpinnerPanel.gridy = 0;
		add(numControlPointsSpinnerPanel, gbc_numControlPointsSpinnerPanel,
				DetailPanelMode.BASICS);
		numControlPointsSpinnerPanel.setLayout(new GridLayout(0, 1, 0, 0));

		numControlPointsSpinnerPanel.add(numControlPointsSpinner_);
		spinnerNumberModel_ = new SpinnerNumberModel(new Integer(3),
				new Integer(3), new Integer(20), new Integer(1));
		numControlPointsSpinner_.setModel(spinnerNumberModel_);

		JPanel symmetricCheckboxPanel = new JPanel();
		symmetricCheckboxPanel.setBorder(null);
		GridBagConstraints gbc_symmetricCheckboxPanel = new GridBagConstraints();
		gbc_symmetricCheckboxPanel.fill = GridBagConstraints.BOTH;
		gbc_symmetricCheckboxPanel.gridx = 1;
		gbc_symmetricCheckboxPanel.gridy = 1;
		add(symmetricCheckboxPanel, gbc_symmetricCheckboxPanel,
				DetailPanelMode.BASICS);
		symmetricCheckboxPanel.setLayout(new GridLayout(0, 1, 0, 0));

		symmetricCheckboxPanel.add(symmetricCheckbox_);

		setVisibility(activeMode_);

		numControlPointsSpinner_.addChangeListener(this);
		symmetricCheckbox_.addItemListener(this);
	}

	// ----------------------------------------------------------------------------

	/** Returns the number of control points introduced by the user. */
	public Integer getNumControlPoints() {
		return (Integer) numControlPointsSpinner_.getValue();
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns <code>true</code> if the curve model is symmetric. Otherwise it
	 * returns <code>false</code>.
	 */
	public boolean isSymmetric() {
		return symmetricCheckbox_.isSelected();
	}

	// ----------------------------------------------------------------------------

	/** Sets the spline curve parameters to the interface. */
	public void setSplineCurveParameters(
			SplineCurveParameters splineCurveParameters) {
		numControlPointsSpinner_.setValue(splineCurveParameters.getM());
	}

	// ----------------------------------------------------------------------------
	// LISTENER METHODS

	/** Handles the events of the spinners. */
	@Override
	public void stateChanged(ChangeEvent e) {
		Object source = e.getSource();
		if (source == numControlPointsSpinner_) {
			shapeDesigner_.loadSplineCurveParametersFromInterface();
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Handles the events of the combo boxes and the checkbox. It fires an
	 * update of the active spline curve.
	 */
	@Override
	public void itemStateChanged(ItemEvent e) {
		if (e.getSource() == symmetricCheckbox_) {
			if (symmetricCheckbox_.isSelected()) {
				if (getNumControlPoints() % 2 == 0) {
					shapeDesigner_.loadSplineCurveParametersFromInterface();
				} else {
					new AnnounceFrame(
							"For a symmetric model the spline curve must have an even number of control points.");
					symmetricCheckbox_.setSelected(false);
				}
			}
			if (symmetricCheckbox_.isSelected()) {
				spinnerNumberModel_.setStepSize(2);
			} else {
				spinnerNumberModel_.setStepSize(1);
			}
		}
	}
}
