/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.ellipsoid;

import icy.canvas.IcyCanvas;
import icy.canvas.IcyCanvas2D;
import icy.roi.ROI;
import icy.roi.ROI3D;
import icy.sequence.Sequence;
import icy.type.point.Point3D;
import icy.type.rectangle.Rectangle3D;
import icy.vtk.VtkUtil;

import java.awt.BasicStroke;
import java.awt.Graphics2D;
import java.awt.Shape;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;

import plugins.kernel.canvas.VtkCanvas;
import vtk.vtkActor;
import vtk.vtkCellArray;
import vtk.vtkDoubleArray;
import vtk.vtkPoints;
import vtk.vtkPolyData;
import vtk.vtkPolyDataMapper;
import vtk.vtkRenderer;

/**
 * Class describing a ROI for ellipsoids.
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class EllipsoidROI3D extends ROI3D {

	private final Ellipsoid3D ellipsoid_;

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * Default constructor.
	 * 
	 * @param ellipsoid
	 *            Ellipsoid descriptor associated to this ROI
	 */
	public EllipsoidROI3D(Ellipsoid3D ellipsoid) {
		ellipsoid_ = ellipsoid;
		setName(ellipsoid_.toString());
	}

	// ----------------------------------------------------------------------------

	/** Returns the descriptor associated to this ROI. */
	public Ellipsoid3D getDescriptor() {
		return ellipsoid_;
	}

	// ============================================================================
	// ROI3D METHODS

	@Override
	public Rectangle3D computeBounds3D() {
		double r = Math.max(Math.max(ellipsoid_.a, ellipsoid_.b), ellipsoid_.c);
		return new Rectangle3D.Double(ellipsoid_.x0 - r, ellipsoid_.y0 - r,
				ellipsoid_.z0 - r, 2 * r, 2 * r, 2 * r);
	}

	// ----------------------------------------------------------------------------

	@Override
	public boolean contains(double x, double y, double z) {
		return ellipsoid_.contains(x, y, z);
	}

	// ----------------------------------------------------------------------------

	@Override
	public boolean contains(double x, double y, double z, double sizeX,
			double sizeY, double sizeZ) {
		// TODO Auto-generated method stub
		return false;
	}

	// ----------------------------------------------------------------------------

	@Override
	public boolean contains(double x, double y, double z, double t, double c) {
		return ellipsoid_.getT() == t && ellipsoid_.contains(x, y, z);
	}

	// ----------------------------------------------------------------------------

	@Override
	protected ROIPainter createPainter() {
		return new EllipsoidPainter();
	}

	// ----------------------------------------------------------------------------

	@Override
	public ROI getCopy() {
		return new EllipsoidROI3D((Ellipsoid3D) ellipsoid_.clone());
	}

	// ----------------------------------------------------------------------------

	@Override
	public Point3D getPosition3D() {
		return new Point3D.Double(ellipsoid_.x0, ellipsoid_.y0, ellipsoid_.z0);
	}

	// ----------------------------------------------------------------------------

	@Override
	public double getSurfaceArea() {
		return ellipsoid_.getPerimeter();
	}

	// ----------------------------------------------------------------------------

	@Override
	public int getT() {
		return ellipsoid_.getT();
	}

	// ----------------------------------------------------------------------------

	@Override
	public double getVolume() {
		return ellipsoid_.getVolume();
	}

	// ----------------------------------------------------------------------------

	@Override
	public boolean hasSelectedPoint() {
		// TODO Auto-generated method stub
		return false;
	}

	// ----------------------------------------------------------------------------

	@Override
	public boolean intersects(double x, double y, double z, double t, double c,
			double sizeX, double sizeY, double sizeZ, double sizeT, double sizeC) {
		return true; // TODO
	}

	// ----------------------------------------------------------------------------

	@Override
	public boolean intersects(double x, double y, double z, double sizeX,
			double sizeY, double sizeZ) {
		// TODO Auto-generated method stub
		return true;
	}

	// ============================================================================
	// PAINTER

	protected class EllipsoidPainter extends ROIPainter {
		/**
		 * Number of horizontal lines of the 3D representation of the ellipsoid.
		 */
		private static final int H_LINES_COUNT = 12;

		/**
		 * Number of horizontal lines of the 3D representation of the ellipsoid.
		 */
		private static final int V_LINES_COUNT = 12;
		private boolean isInitialized_ = false;
		private vtkActor actor_ = null;

		@Override
		public void paint(Graphics2D g2, Sequence sequence, IcyCanvas canvas) {
			if (ellipsoid_ == null
					|| ellipsoid_.getT() != canvas.getPositionT()) {
				return;
			}

			if (canvas instanceof IcyCanvas2D) {
				int zSlide = canvas.getPositionZ();

				double stroke = getStroke()
						* (1.0 - Math.abs(ellipsoid_.z0 - zSlide)
								/ Math.max(
										Math.max(ellipsoid_.a, ellipsoid_.b),
										ellipsoid_.c));

				Ellipsoid2D ellipse = ellipsoid_.intersectionAtZ(zSlide);
				if (ellipse != null) {
					g2.setStroke(new BasicStroke((float) stroke));
					g2.setColor(getColor());

					Shape baseShape = new Ellipse2D.Double(ellipse.x0
							- ellipse.a, ellipse.y0 - ellipse.b, 2 * ellipse.a,
							2 * ellipse.b);
					Shape rotatedShape = AffineTransform.getRotateInstance(
							ellipse.alpha, ellipse.x0, ellipse.y0)
							.createTransformedShape(baseShape);
					g2.draw(rotatedShape);
				}
			} else if (canvas instanceof VtkCanvas) {
				final VtkCanvas canvas3d = (VtkCanvas) canvas;
				final vtkRenderer renderer = canvas3d.getRenderer();

				if (!isInitialized_) {
					init3DRenderer(renderer, sequence.getPixelSizeX(),
							sequence.getPixelSizeY(), sequence.getPixelSizeZ());
				}
				if (!VtkUtil.findProp(renderer, actor_)) {
					renderer.AddActor(actor_);
				}
			}
		}

		protected void init3DRenderer(vtkRenderer renderer, double pixelSizeX,
				double pixelSizeY, double pixelSizeZ) {
			renderer.SetGlobalWarningDisplay(0);

			final vtkPoints points = new vtkPoints();
			final double[] coordinates = new double[3 * H_LINES_COUNT
					* V_LINES_COUNT];
			int[][] indices = new int[2 * H_LINES_COUNT * V_LINES_COUNT][2];
			int i = 0, j = 0, offset = H_LINES_COUNT * V_LINES_COUNT;

			final double[] R = AbstractEllipsoid.transposeMatrix(ellipsoid_
					.getRotationMatrix());
			for (int vi = 0; vi < H_LINES_COUNT; vi++) {
				double v = ellipsoid_.beta + Math.PI
						* (2.0 * vi / (H_LINES_COUNT - 1) - 1);
				double cb = Math.cos(v);
				double sb = Math.sin(v);

				for (int ui = 0; ui < V_LINES_COUNT; ui++) {
					double u = ellipsoid_.alpha + ui * Math.PI
							/ (V_LINES_COUNT - 1);
					double ca = Math.cos(u);
					double sa = Math.sin(u);

					double X[] = { ellipsoid_.a * ca * cb,
							ellipsoid_.b * sa * cb, ellipsoid_.c * sb };
					X = AbstractEllipsoid.applyRotation(R, X);

					coordinates[3 * i] = pixelSizeX * (ellipsoid_.x0 + X[0]);
					coordinates[3 * i + 1] = pixelSizeY
							* (ellipsoid_.y0 + X[1]);
					coordinates[3 * i + 2] = pixelSizeZ
							* (ellipsoid_.z0 + X[2]);

					indices[j][0] = i;
					indices[j][1] = (ui != V_LINES_COUNT - 1) ? i + 1 : i + 1
							- V_LINES_COUNT;
					indices[j + offset][0] = i;
					indices[j + offset][1] = (vi != H_LINES_COUNT - 1) ? i
							+ V_LINES_COUNT : i - (H_LINES_COUNT - 1)
							* V_LINES_COUNT;

					i++;
					j++;
				}
			}

			final vtkDoubleArray array = new vtkDoubleArray();
			array.SetJavaArray(coordinates);
			array.SetNumberOfComponents(3);
			points.SetData(array);

			final vtkCellArray cells = VtkUtil.getCells(indices.length - 1,
					VtkUtil.prepareCells(indices));

			vtkPolyData data = new vtkPolyData();
			data.SetPoints(points);
			data.SetLines(cells);

			// Add actor to the renderer
			final vtkPolyDataMapper polyMapper = new vtkPolyDataMapper();
			polyMapper.SetInputData(data);
			actor_ = new vtkActor();
			actor_.SetMapper(polyMapper);
			actor_.GetProperty().SetColor(1, 0, 0);
			renderer.AddActor(actor_);

			isInitialized_ = true;
		}
	}
}
