package plugins.spop.advancefilters;

import icy.image.IcyBufferedImage;
import icy.math.ArrayMath;
import icy.sequence.Sequence;
import icy.type.collection.array.Array1DUtil;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.ezplug.EzVarText;
import plugins.adufour.filtering.Convolution1D;
import plugins.adufour.filtering.Kernels1D;

public class WeickertFilter extends EzPlug implements Block{
	
	/**
	 *	PDE Anisotropic Diffusion
	 *
	 * @author Sorin POP
	 * 
	 * @see J. Weickert  Coherence enhancing diffusion, International Journal of Computer Vision , no.31, pp. 111-127, 1999
	 * 
	 * Computed only on the first color band of data
	 */
	EzVarSequence input = new EzVarSequence("Input");
	EzVarText	type_2D = new EzVarText("Type", new String[] { "2D", "3D" },0, false);
	EzVarInteger iterations = new EzVarInteger("Iterations",10, 1, 10000, 1);
	EzVarDouble step = new EzVarDouble("Discretization step", 0.15, 0.01, 0.25, 0.01);
	//amount of diffusion in the gradient direction
	EzVarDouble alpha = new EzVarDouble("Alpha",0.001,0.001,1,0.001);
	// range for the difference between tensor structure eigenvalue
	// @see Weickert diffusion function
	EzVarDouble cval = new EzVarDouble("C",1,0,65536,0.1);
	//local scale
	EzVarDouble sigma = new EzVarDouble("Sigma",0.5,0.1,10,0.01);
	//integration scale
	EzVarDouble ro = new EzVarDouble("Ro",1,0.1,10,0.01);
	EzGroup group;
	// CED 1D option - classical Coherence Enhanching Diffusion Filter - the diffusion acts mainly on the most homogeneous structure direction
	// CED 2D option - diffusion on tangential plane to structure
	EzVarText	coh_type = new EzVarText("Coherence", new String[] { "CED 1D", "CED 2D" }, 0, false);
	EzVarDouble z_rezolution = new EzVarDouble("Z/X rezolution Ratio",1,0.001,100,0.001);
	
	EzVarSequence output = new EzVarSequence("Output");
	
	long time = 0;
	@Override
	public void initialize() {
		// TODO Auto-generated method stub
		addEzComponent(input);
		addEzComponent(type_2D);
		addEzComponent(iterations);
		addEzComponent(step);		
		addEzComponent(alpha);
		addEzComponent(cval);
		addEzComponent(sigma);
		addEzComponent(ro);
		
		
		group=new EzGroup("3D parameters",coh_type,z_rezolution);
		addEzComponent(group);
		type_2D.addVisibilityTriggerTo(group, "3D");
		setTimeDisplay(true);
		
	}
	@Override
	public void execute() {
		
		double dt= step.getValue();;
		Sequence sequence_out=null;
				
		if(type_2D.getValue().equalsIgnoreCase("2D"))
		{			
			sequence_out=Weickert_2D(input.getValue(true),0,iterations.getValue(),dt,alpha.getValue(),cval.getValue(),sigma.getValue(),ro.getValue());
		}
		else
		{
			int coh_int=1;
			double z_res=z_rezolution.getValue();
			if (coh_type.getValue().equalsIgnoreCase("CED 1D"))
	            coh_int = 0;
	        if (coh_type.getValue().equalsIgnoreCase("CED 2D"))
	            coh_int = 1;
			sequence_out= Weickert_3D(input.getValue(true),0,iterations.getValue(),dt,alpha.getValue(),cval.getValue(),sigma.getValue(),ro.getValue(),coh_int,z_res);
			
			sequence_out.setName("Diffusion Weickert 3D");
		}
		
		 if(getUI()!=null)
			addSequence(sequence_out);
		output.setValue(sequence_out);
				
	}

/**
 * 2D Weickert model
 * @param sequence - input sequence; The input can be 2D,3D or 4D 
 * @param c - band; Only one band is computed
 * @param iter - number of iteration
 * @param dt - discretization step 
 * @param alpha - amount of diffusivity in the mean gradient direction
 * @param C - range of the eigenvalue difference; Default value :1 
 * @param sigma - local scale
 * @param ro - integration scale
 * @return - filtered sequence
 */
public Sequence Weickert_2D(Sequence sequence, int c, int iter, double dt, double alpha, double C, double sigma, double ro){
		
		int z=0;
		
		Sequence seq=new Sequence();
		
		
		int dim_x=sequence.getSizeX();
		int dim_y=sequence.getSizeY();
		int dim_z=sequence.getSizeZ();
		
		//gaussian kernels for local and integration scales
		double[] ker1=  Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigma).getData();
		double[] ker2=  Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(ro).getData();
		// kerX - first order central derivative kernel
		double[] kerX= { -0.5, 0, 0.5 };
        
        
        int time_slice=0;
        int dim_t=sequence.getSizeT();
        //
		for(time_slice=0;time_slice<dim_t;time_slice++)
		{
			z=0;
			double[][] tab = new double[dim_z][dim_x*dim_y];
			double[][] tab_gauss = new double[1][dim_x*dim_y];
			//a,b,c components of 2D structure tensor : J=[a,c;c,b]
			double[][] tab_a = new double[1][dim_x*dim_y];
			double[][] tab_b = new double[1][dim_x*dim_y];
			double[][] tab_c = new double[1][dim_x*dim_y];
			
			double[] tab_out = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice,z,c), false);
			
			//copy the sequence
			for(z=0;z<dim_z;z++)
				tab[z] = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice,z,c), false);
			
			
			for (int t = 0; t < iter; t++)
			{
				for(z=0;z<dim_z;z++)
				{
					//compute the structure tensor components
					try{
					System.arraycopy(tab[z], 0, tab_gauss[0], 0, dim_x*dim_y);
					Convolution1D.convolve(tab_gauss, dim_x,dim_y,ker1, ker1, null);
					
					System.arraycopy(tab_gauss[0], 0, tab_a[0], 0, dim_x*dim_y);
					System.arraycopy(tab_gauss[0], 0, tab_b[0], 0, dim_x*dim_y);
					
					Convolution1D.convolve(tab_a, dim_x,dim_y,kerX, null, null);
					Convolution1D.convolve(tab_b, dim_x,dim_y,null, kerX, null);
					
					ArrayMath.multiply(tab_a[0], tab_b[0],tab_c[0]);
					ArrayMath.multiply(tab_a[0], tab_a[0],tab_a[0]);
					ArrayMath.multiply(tab_b[0], tab_b[0],tab_b[0]);
								
					Convolution1D.convolve(tab_a, dim_x,dim_y,ker2, ker2, null);
					Convolution1D.convolve(tab_b, dim_x,dim_y,ker2, ker2, null);
					Convolution1D.convolve(tab_c, dim_x,dim_y,ker2, ker2, null);
					 } catch (Exception e) {
							// TODO Auto-generated catch block
							e.printStackTrace();
						} 
					
					//
					DiffusionTools.Weickert_core_2D(tab[z],tab_a[0],tab_b[0],tab_c[0],dim_x,dim_y,dt, alpha,C, tab_out);
								
			        System.arraycopy(tab_out, 0, tab[z], 0, dim_x*dim_y);
				}        
				if(getUI()!=null)
					getUI().setProgressBarValue((double)((t+iter*time_slice)/(double)(iter*dim_t)));		        
			}			
			
			for(z=0;z<dim_z;z++)
			{		
				IcyBufferedImage icy_img = new IcyBufferedImage(dim_x, dim_y, 1, sequence.getDataType());
				Array1DUtil.doubleArrayToSafeArray(tab[z], icy_img.getDataXY(0), icy_img.isSignedDataType());
				//icy_img.setDataXY(c, (Object)tab[z]);
				seq.setImage(time_slice, z, icy_img);
			}
			System.gc();
			
		}
		seq.updateComponentsBounds();		
		seq.setName("Diffusion Weickert 2D");
		
		return seq;
		
	}
	
/**
 * 3D Weickert model
 * @param sequence - input sequence; The input can be 3D or 4D 
 * @param c - band; Only one band is computed
 * @param iter - number of iteration
 * @param dt - discretization step 
 * @param alpha - amount of diffusivity in the mean gradient direction
 * @param C - range of the eigenvalue difference; Default value :1 
 * @param sigma - local scale
 * @param ro - integration scale
 * @param coh_type - 0 stands for CED1D and 1 for CED2D
 * @param z_res - Z/X resolution ratio 
 * @return - filtered sequence
 */
public Sequence Weickert_3D(Sequence sequence, int c, int iter, double dt, double alpha, double C, double sigma, double ro, int coh_type, double z_res){
		
		int z=0;
		double hz_inv=(1.0/z_res);
		//integration scale on Z axis 
		double ro_z=ro/z_res;
		//local scale on Z axis 
		double sigma_z=sigma/z_res;
		
		
		int dim_x=sequence.getSizeX();
		int dim_y=sequence.getSizeY();
		int dim_z=sequence.getSizeZ();
		
							
		double[] ker1=  Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigma).getData();
		double[] ker2=  Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(ro).getData();
		double[] ker1z=  Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigma_z).getData();
		double[] ker2z=  Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(ro_z).getData();
		
		double[] kerX= { -0.5, 0, 0.5 };
		// 
        double[] kerZ=new double[kerX.length];
        for(int l_ker=0;l_ker<kerX.length;l_ker++)
        	kerZ[l_ker]=kerX[l_ker]*hz_inv;
        
        Sequence seq=new Sequence();
        int time_slice=0;
        int dim_t=sequence.getSizeT();
		for(time_slice=0;time_slice<dim_t;time_slice++)
		{						
			double[][] tab = new double[dim_z][dim_x*dim_y];
			double[][] tab_gauss = new double[dim_z][dim_x*dim_y];
			//a,b,c,d,e,f components of the 3D structure tensor : J=[a,d,e;d,b,f;e,f,c]
			//      a  d  e
			//   J= d  b  f
			//      e  f  c
			double[][] tab_a = new double[dim_z][dim_x*dim_y];
			double[][] tab_b = new double[dim_z][dim_x*dim_y];
			double[][] tab_c = new double[dim_z][dim_x*dim_y];
			double[][] tab_d = new double[dim_z][dim_x*dim_y];
			double[][] tab_e = new double[dim_z][dim_x*dim_y];
			double[][] tab_f = new double[dim_z][dim_x*dim_y];
					
			for(z=0;z<dim_z;z++)
			{
				tab[z] = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice,z,c), false);
				System.arraycopy(tab[z], 0, tab_gauss[z], 0, dim_x*dim_y);
			}
			
			
			for (int t = 0; t < iter; t++)
			{
				//compute the structure tensor components
				try{
				Convolution1D.convolve(tab_gauss, dim_x,dim_y,ker1, ker1, ker1z);
				
				for(z=0;z<dim_z;z++)
				{	
					
					System.arraycopy(tab_gauss[z], 0, tab_a[z], 0, dim_x*dim_y);
					System.arraycopy(tab_gauss[z], 0, tab_b[z], 0, dim_x*dim_y);
					System.arraycopy(tab_gauss[z], 0, tab_c[z], 0, dim_x*dim_y);
				}
				Convolution1D.convolve(tab_a, dim_x,dim_y,kerX, null, null);
				Convolution1D.convolve(tab_b, dim_x,dim_y,null, kerX, null);
				Convolution1D.convolve(tab_c, dim_x,dim_y,null, null, kerZ);
					
				for(z=0;z<dim_z;z++)
				{
					
					
					ArrayMath.multiply(tab_b[z], tab_c[z],tab_f[z]);
					ArrayMath.multiply(tab_a[z], tab_c[z],tab_e[z]);
					ArrayMath.multiply(tab_a[z], tab_b[z],tab_d[z]);
					
					ArrayMath.multiply(tab_c[z], tab_c[z],tab_c[z]);
					ArrayMath.multiply(tab_b[z], tab_b[z],tab_b[z]);
					ArrayMath.multiply(tab_a[z], tab_a[z],tab_a[z]);
				}
								
				Convolution1D.convolve(tab_a, dim_x,dim_y,ker2, ker2, ker2z);
				Convolution1D.convolve(tab_b, dim_x,dim_y,ker2, ker2, ker2z);
				Convolution1D.convolve(tab_c, dim_x,dim_y,ker2, ker2, ker2z);
				Convolution1D.convolve(tab_d, dim_x,dim_y,ker2, ker2, ker2z);
				Convolution1D.convolve(tab_e, dim_x,dim_y,ker2, ker2, ker2z);
				Convolution1D.convolve(tab_f, dim_x,dim_y,ker2, ker2, ker2z);
				 } catch (Exception e) {
						// TODO Auto-generated catch block
						e.printStackTrace();
					} 
			
				DiffusionTools.Weickert_core_3D(tab,tab_a,tab_b,tab_c,tab_d,tab_e,tab_f,dim_x,dim_y,dim_z,dt, alpha,C,coh_type, hz_inv, tab_gauss);//tab_gauss acts like output
				
				for(z=0;z<dim_z;z++)
					System.arraycopy(tab_gauss[z], 0, tab[z], 0, dim_x*dim_y);
				if(getUI()!=null)
					getUI().setProgressBarValue((double)((t+iter*time_slice)/(double)(iter*dim_t)));
		        
			}
						
			for(z=0;z<dim_z;z++)
			{		
				IcyBufferedImage icy_img = new IcyBufferedImage(dim_x, dim_y, 1, sequence.getDataType());
				Array1DUtil.doubleArrayToSafeArray(tab[z], icy_img.getDataXY(0), icy_img.isSignedDataType());
				seq.setImage(time_slice, z, icy_img);
			}
		}
		seq.updateComponentsBounds();		
		return seq;
		
	}



public void clean()
{
}
@Override
public void declareInput(VarList inputMap) {
	// TODO Auto-generated method stub
	inputMap.add(input.getVariable());
	inputMap.add(type_2D.getVariable());
	inputMap.add(iterations.getVariable());
	inputMap.add(step.getVariable());
	inputMap.add(alpha.getVariable());	
	inputMap.add(cval.getVariable());	
	inputMap.add(sigma.getVariable());	
	inputMap.add(ro.getVariable());
	inputMap.add(coh_type.getVariable());
	inputMap.add(z_rezolution.getVariable());
	
	
}
@Override
public void declareOutput(VarList outputMap) {
	// TODO Auto-generated method stub
	outputMap.add(output.getVariable());
}
	
}
