/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.process.process2D;

import icy.image.IcyBufferedImage;
import icy.type.collection.array.Array1DUtil;

import java.util.Arrays;

/**
 * Class that encapsulated simple 2D image operations.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 */
public class Filters {

	// ============================================================================
	// PUBLIC METHODS

	/** Applies a 3x3 smoothing kernel to the image. */
	public static void filter3x3(IcyBufferedImage im, int[] kernel) {
		double[] pixels = Array1DUtil.arrayToDoubleArray(im.getDataXY(0), im
				.getDataType_().isSigned());
		filter3x3(pixels, im.getWidth(), im.getHeight(), kernel);
		Array1DUtil.doubleArrayToArray(pixels, im.getDataXY(0));
	}

	// ----------------------------------------------------------------------------

	/**
	 * Efficient routine to filter an image expressed as an array of pixels with
	 * a 3x3 kernel.
	 */
	public static void filter3x3(double[] pixels, int width, int height,
			int[] kernel) {
		double v1, v2, v3; // input pixel values around the current pixel
		double v4, v5, v6;
		double v7, v8, v9;
		double scale = 0f;

		double k1 = kernel[0];
		double k2 = kernel[1];
		double k3 = kernel[2];
		double k4 = kernel[3];
		double k5 = kernel[4];
		double k6 = kernel[5];
		double k7 = kernel[6];
		double k8 = kernel[7];
		double k9 = kernel[8];

		for (int element : kernel) {
			scale += element;
		}
		if (scale == 0) {
			scale = 1f;
		}
		scale = 1f / scale; // multiplication factor (multiply is faster than
							// divide)

		int inc = (width - 2) / 25;
		if (inc < 1) {
			inc = 1;
		}

		double[] pixels2 = Arrays.copyOf(pixels, pixels.length);

		for (int y = 1; y < height - 1; y++) {
			int p = 1 + y * width; // points to current pixel
			int p6 = p - 1; // will point to v6, currently lower
			int p3 = p6 - width; // will point to v3, currently lower
			int p9 = p6 + width; // ... to v9, currently lower
			v2 = pixels2[p3];
			v5 = pixels2[p6];
			v8 = pixels2[p9];
			p3++;
			p6++;
			p9++;
			v3 = pixels2[p3];
			v6 = pixels2[p6];
			v9 = pixels2[p9];

			for (int x = 1; x < width - 1; x++, p++) {
				p3++;
				p6++;
				p9++;
				v1 = v2;
				v2 = v3;
				v3 = pixels2[p3];
				v4 = v5;
				v5 = v6;
				v6 = pixels2[p6];
				v7 = v8;
				v8 = v9;
				v9 = pixels2[p9];
				double sum = k1 * v1 + k2 * v2 + k3 * v3 + k4 * v4 + k5 * v5
						+ k6 * v6 + k7 * v7 + k8 * v8 + k9 * v9;
				sum *= scale;
				pixels[p] = sum;
			}
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Applies an affine mapping to the image values such that the minimum value
	 * is mapped to the maximum and vice versa.
	 */
	public static void invert(IcyBufferedImage im) {
		double[] minAndMax = im.getChannelsGlobalTypeBounds();
		double min = minAndMax[0];
		double max = minAndMax[1];
		double[] resultDataBuffer = Array1DUtil.arrayToDoubleArray(
				im.getDataXY(0), im.getDataType_().isSigned());
		for (int i = 0; i < resultDataBuffer.length; i++) {
			resultDataBuffer[i] = min + max - resultDataBuffer[i];
		}
		Array1DUtil.doubleArrayToArray(resultDataBuffer, im.getDataXY(0));
	}
}
