package plugins.ylemontag.complex;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.system.SystemUtil;
import icy.system.thread.Processor;
import icy.type.DataType;
import icy.type.collection.array.Array2DUtil;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Complex conversion functions
 */
public enum ComplexFeature
{
	REAL_PART             ("Real part"               , true ),
	IMAGINARY_PART        ("Imaginary part"          , true ),
	MODULUS               ("Modulus"                 , true ),
	ARGUMENT              ("Argument"                , true ),
	CARTESIAN_REPRESENTION("Cartesian representation", false),
	POLAR_REPRESENTATION  ("Polar representation"    , false);
	
	private String _name;
	private boolean _isScalarFeature;
	
	/**
	 * Constructor
	 */
	private ComplexFeature(String name, boolean isScalarFeature)
	{
		_name = name;
		_isScalarFeature = isScalarFeature;
	}
	
	/**
	 * Feature name
	 */
	public String description()
	{
		return _name;
	}
	
	@Override
	public String toString()
	{
		return _name;
	}
	
	/**
	 * Whether the feature is scalar-valued
	 */
	public boolean isScalarFeature()
	{
		return _isScalarFeature;
	}
	
	/**
	 * Conversion function
	 * @param in Complex-valued input sequence, with an even number of channels
	 * @param representation Whether the input sequence uses a (real,imaginary) or a (modulus,argument) representation
	 */
	public Sequence get(Sequence in, ComplexRepresentation representation)
		throws InvalidComplexEntry
	{
		try {
			return get(in, representation, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException(
				"A complex conversion thread have been unexpectedly interrupted"
			);
		}
	}
	
	/**
	 * Conversion function
	 * @param in1 Real-valued input sequence, representing either a real component or a modulus
	 * @param in2 Real-valued input sequence, representing either an imaginary component or an argument
	 * @param representation Whether the input sequence uses a (real,imaginary) or a (modulus,argument) representation
	 * @remark in1 and in2 must have the same dimensions
	 */
	public Sequence get(Sequence in1, Sequence in2, ComplexRepresentation representation)
		throws InvalidComplexEntry
	{
		try {
			return get(in1, in2, representation, new Controller());
		}
		catch(Controller.CanceledByUser err) {
			throw new RuntimeException(
				"A complex conversion thread have been unexpectedly interrupted"
			);
		}
	}
	
	/**
	 * Conversion function with thread control (1 complex input)
	 * @param in1 Real-valued input sequence, representing either a real component or a modulus
	 * @param in2 Real-valued input sequence, representing either an imaginary component or an argument
	 * @param representation Whether the input sequence uses a (real,imaginary) or a (modulus,argument) representation
	 * @param controller Controller object
	 * @remark in1 and in2 must have the same dimensions
	 */
	public Sequence get(Sequence in, ComplexRepresentation representation, Controller controller)
		throws InvalidComplexEntry, Controller.CanceledByUser
	{
		switch(representation)
		{	
			// The input uses a Cartesian representation
			case CARTESIAN:
				switch(this) {
					case REAL_PART             : return coreExtract(in, new CopyEvenFunctor        (), controller);
					case IMAGINARY_PART        : return coreExtract(in, new CopyOddFunctor         (), controller);
					case MODULUS               : return coreExtract(in, new ModulusFunctor         (), controller);
					case ARGUMENT              : return coreExtract(in, new ArgumentFunctor        (), controller);
					case CARTESIAN_REPRESENTION: return coreExtract(in, new CopyBothFunctor        (), controller);
					case POLAR_REPRESENTATION  : return coreExtract(in, new CartesianToPolarFunctor(), controller);
				}
			
			// The input uses a polar representation
			case POLAR:
				switch(this) {
					case REAL_PART             : return coreExtract(in, new RealFunctor            (), controller);
					case IMAGINARY_PART        : return coreExtract(in, new ImaginaryFunctor       (), controller);
					case MODULUS               : return coreExtract(in, new CopyEvenFunctor        (), controller);
					case ARGUMENT              : return coreExtract(in, new CopyOddFunctor         (), controller);
					case CARTESIAN_REPRESENTION: return coreExtract(in, new PolarToCartesianFunctor(), controller);
					case POLAR_REPRESENTATION  : return coreExtract(in, new CopyBothFunctor        (), controller);
				}
		}
		return null;
	}
	
	/**
	 * Conversion function with thread control (2 real inputs)
	 * @param in Complex-valued input sequence, with an even number of channels
	 * @param representation Whether the input sequence uses a (real,imaginary) or a (modulus,argument) representation
	 * @param controller Controller object
	 */
	public Sequence get(Sequence in1, Sequence in2, ComplexRepresentation representation, Controller controller)
		throws InvalidComplexEntry, Controller.CanceledByUser
	{
		switch(representation)
		{	
			// The input uses a Cartesian representation
			case CARTESIAN:
				switch(this) {
					case REAL_PART             : return coreExtract(in1, in2, new CopyEvenFunctor        (), controller);
					case IMAGINARY_PART        : return coreExtract(in1, in2, new CopyOddFunctor         (), controller);
					case MODULUS               : return coreExtract(in1, in2, new ModulusFunctor         (), controller);
					case ARGUMENT              : return coreExtract(in1, in2, new ArgumentFunctor        (), controller);
					case CARTESIAN_REPRESENTION: return coreExtract(in1, in2, new CopyBothFunctor        (), controller);
					case POLAR_REPRESENTATION  : return coreExtract(in1, in2, new CartesianToPolarFunctor(), controller);
				}
			
			// The input uses a polar representation
			case POLAR:
				switch(this) {
					case REAL_PART             : return coreExtract(in1, in2, new RealFunctor            (), controller);
					case IMAGINARY_PART        : return coreExtract(in1, in2, new ImaginaryFunctor       (), controller);
					case MODULUS               : return coreExtract(in1, in2, new CopyEvenFunctor        (), controller);
					case ARGUMENT              : return coreExtract(in1, in2, new CopyOddFunctor         (), controller);
					case CARTESIAN_REPRESENTION: return coreExtract(in1, in2, new PolarToCartesianFunctor(), controller);
					case POLAR_REPRESENTATION  : return coreExtract(in1, in2, new CopyBothFunctor        (), controller);
				}
		}
		return null;
	}
	
	/**
	 * Functor interface for conversions
	 */
	private static interface Functor
	{
		/**
		 * Targeted feature
		 * @param out Output vector
		 * @param in Input vector (complex-valued)
		 * @param sizeComplexC Number of complex channels (twice the number of channels of the input)
		 * @param sizeXY Number of pixels in the 2D image
		 */
		public void apply1(double[][] out, double[][] in, int sizeComplexC, int sizeXY);
		
		/**
		 * Targeted feature
		 * @param out Output vector
		 * @param in1 Input vector 1 (read-valued)
		 * @param in2 Input vector 2 (read-valued)
		 * @param sizeComplexC Number of complex channels (twice the number of channels of the input)
		 * @param sizeXY Number of pixels in the 2D image
		 */
		public void apply2(double[][] out, double[][] in1, double[][] in2, int sizeComplexC, int sizeXY);
	}
	
	/**
	 * Functor that outputs a scalar feature
	 */
	private abstract static class UnaryFunctor implements Functor
	{
		@Override
		public void apply1(double[][] out, double[][] in, int sizeComplexC, int sizeXY)
		{
			for(int c=0; c<sizeComplexC; ++c) {
				copyXY(out[c], in[2*c], in[2*c+1], sizeXY);
			}
		}

		@Override
		public void apply2(double[][] out, double[][] in1, double[][] in2, int sizeComplexC, int sizeXY)
		{
			for(int c=0; c<sizeComplexC; ++c) {
				copyXY(out[c], in1[c], in2[c], sizeXY);
			}
		}
		
		/**
		 * Core function to implement, that do the conversion between in1 and in2
		 */
		protected abstract void copyXY(double out[], double in1[], double in2[], int sizeXY);
	}
	
	/**
	 * Functor that outputs a complex-valued feature
	 */
	private abstract static class BinaryFunctor implements Functor
	{
		@Override
		public void apply1(double[][] out, double[][] in, int sizeComplexC, int sizeXY)
		{
			for(int c=0; c<sizeComplexC; ++c) {
				copyXY(out[2*c], out[2*c+1], in[2*c], in[2*c+1], sizeXY);
			}
		}

		@Override
		public void apply2(double[][] out, double[][] in1, double[][] in2, int sizeComplexC, int sizeXY)
		{
			for(int c=0; c<sizeComplexC; ++c) {
				copyXY(out[2*c], out[2*c+1], in1[c], in2[c], sizeXY);
			}
		}
		
		/**
		 * Core function to implement, that do the conversion between in1 and in2
		 */
		protected abstract void copyXY(double out1[], double[] out2, double in1[], double in2[], int sizeXY);
	}
	
	/**
	 * Extract the real component
	 */
	private static class RealFunctor extends UnaryFunctor
	{	
		@Override
		protected void copyXY(double out[], double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out[xy] = ComplexUtil.real(in1[xy], in2[xy]);
			}
		}
	}
	
	/**
	 * Extract the imaginary component
	 */
	private static class ImaginaryFunctor extends UnaryFunctor
	{
		@Override
		protected void copyXY(double out[], double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out[xy] = ComplexUtil.imaginary(in1[xy], in2[xy]);
			}
		}		
	}
	
	/**
	 * Extract the modulus
	 */
	private static class ModulusFunctor extends UnaryFunctor
	{
		@Override
		protected void copyXY(double out[], double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out[xy] = ComplexUtil.modulus(in1[xy], in2[xy]);
			}
		}
	}
	
	/**
	 * Extract the argument
	 */
	private static class ArgumentFunctor extends UnaryFunctor
	{
		@Override
		protected void copyXY(double out[], double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out[xy] = ComplexUtil.argument(in1[xy], in2[xy]);
			}
		}
	}
	
	/**
	 * Conversion Cartesian to polar
	 */
	private static class CartesianToPolarFunctor extends BinaryFunctor
	{
		@Override
		protected void copyXY(double out1[], double[] out2, double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				double currentIn1 = in1[xy];
				double currentIn2 = in2[xy];
				out1[xy] = ComplexUtil.modulus (currentIn1, currentIn2);
				out2[xy] = ComplexUtil.argument(currentIn1, currentIn2);
			}
		}
	}
	
	/**
	 * Conversion polar to Cartesian
	 */
	private static class PolarToCartesianFunctor extends BinaryFunctor
	{
		@Override
		protected void copyXY(double out1[], double[] out2, double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				double currentIn1 = in1[xy];
				double currentIn2 = in2[xy];
				out1[xy] = ComplexUtil.real     (currentIn1, currentIn2);
				out2[xy] = ComplexUtil.imaginary(currentIn1, currentIn2);
			}
		}
	}
	
	/**
	 * Copy the even channels in the output
	 */
	private static class CopyEvenFunctor extends UnaryFunctor
	{
		@Override
		protected void copyXY(double out[], double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out[xy] = in1[xy];
			}
		}
	}
	
	/**
	 * Copy the odd channels in the output
	 */
	private static class CopyOddFunctor extends UnaryFunctor
	{
		@Override
		protected void copyXY(double out[], double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out[xy] = in2[xy];
			}
		}
	}
	
	/**
	 * Copy the both channels in the output
	 */
	private static class CopyBothFunctor extends BinaryFunctor
	{
		@Override
		protected void copyXY(double out1[], double[] out2, double in1[], double in2[], int sizeXY)
		{
			for(int xy=0; xy<sizeXY; ++xy) {
				out1[xy] = in1[xy];
				out2[xy] = in2[xy];
			}
		}
	}
	
	/**
	 * Core function for component extraction (1 input)
	 */
	private Sequence coreExtract(Sequence in, Functor fun, Controller controller)
		throws InvalidComplexEntry, Controller.CanceledByUser
	{
		// Check the input
		if(in==null) {
			throw new InvalidComplexEntry("The input sequence cannot be null");
		}
		if(in.getSizeC()%2!=0) {
			throw new InvalidComplexEntry("The input sequence must have an even number of channels");
		}
		int sizeX = in.getSizeX();
		int sizeY = in.getSizeY();
		int sizeZ = in.getSizeZ();
		int sizeT = in.getSizeT();
		int sizeComplexC = in.getSizeC()/2;
		
		// Allocations
		Sequence out = allocateSequence(sizeX, sizeY, sizeZ, sizeT, sizeComplexC, in.getName());
		final JobCommons1 jc1 = new JobCommons1();
		jc1.in = in;
		jc1.out = out;
		jc1.fun = fun;
		jc1.sizeComplexC = sizeComplexC;
		jc1.sizeXY = sizeX*sizeY;
		jc1.controller = controller;
		
		// Core compute
		return coreCompute(out, sizeT, sizeZ, jc1, new JobBuilder() {
			
			@Override
			public Job build(int t, int z) {
				return new Job1(t, z, jc1);
			}
		});
	}
	
	/**
	 * Core function for component extraction (2 inputs)
	 */
	private Sequence coreExtract(Sequence in1, Sequence in2, Functor fun, Controller controller)
		throws InvalidComplexEntry, Controller.CanceledByUser
	{
		// Check the input
		if(in1==null || in2==null) {
			throw new InvalidComplexEntry("The input sequences cannot be null");
		}
		if(!(
			in1.getSizeX()==in2.getSizeX() &&
			in1.getSizeY()==in2.getSizeY() &&
			in1.getSizeZ()==in2.getSizeZ() &&
			in1.getSizeT()==in2.getSizeT() &&
			in1.getSizeC()==in2.getSizeC()
		)) {
			throw new InvalidComplexEntry("The input sequences must have the same size");
		}
		int sizeX = in1.getSizeX();
		int sizeY = in1.getSizeY();
		int sizeZ = in1.getSizeZ();
		int sizeT = in1.getSizeT();
		int sizeComplexC = in1.getSizeC();
		
		// Allocations
		String baseName = "(" + in1.getName() + "," + in2.getName() + ")";
		Sequence out = allocateSequence(sizeX, sizeY, sizeZ, sizeT, sizeComplexC, baseName);
		final JobCommons2 jc2 = new JobCommons2();
		jc2.in1 = in1;
		jc2.in2 = in2;
		jc2.out = out;
		jc2.fun = fun;
		jc2.sizeComplexC = sizeComplexC;
		jc2.sizeXY = sizeX*sizeY;
		jc2.controller = controller;
		
		// Core compute
		return coreCompute(out, sizeT, sizeZ, jc2, new JobBuilder() {
			
			@Override
			public Job build(int t, int z) {
				return new Job2(t, z, jc2);
			}
		});
	}
	
	/**
	 * Allocate the result sequence
	 */
	private Sequence allocateSequence(int sizeX, int sizeY, int sizeZ, int sizeT,
		int sizeComplexC, String baseName)
	{
		int sizeC = _isScalarFeature ? sizeComplexC : sizeComplexC*2;
		Sequence retVal = new Sequence();
		retVal.setName(baseName + " -> " + _name);
		retVal.beginUpdate();
		for(int t=0; t<sizeT; ++t) {
			for(int z=0; z<sizeZ; ++z) {
				IcyBufferedImage frame = new IcyBufferedImage(sizeX, sizeY, sizeC, DataType.DOUBLE);
				retVal.setImage(t, z, frame);
			}
		}
		retVal.endUpdate();
		return retVal;
	}
	
	/**
	 * Core computation
	 */
	private static Sequence coreCompute(Sequence out, int sizeT, int sizeZ, JobCommons jc, JobBuilder builder)
		throws Controller.CanceledByUser
	{
		int jobCount = sizeT*sizeZ;
		if(jobCount==1) {
			Job job = builder.build(0, 0);
			job.doTheJob();
		}
		else {
			Processor processor = new Processor(jobCount, SystemUtil.getAvailableProcessors());
			for(int t=0; t<sizeT; ++t) {
				for(int z=0; z<sizeZ; ++z) {
					processor.addTask(builder.build(t, z));
				}
			}
			processor.waitAll();
			if(jc.cancelException!=null) {
				throw jc.cancelException;
			}
		}
		out.dataChanged();
		return out;
	}
	
	/**
	 * Parameters common to all jobs
	 */
	private static class JobCommons
	{
		Sequence out;
		Functor fun;
		int sizeComplexC;
		int sizeXY;
		Controller controller;
		Controller.CanceledByUser cancelException;
	}
	
	/**
	 * Common parameters for 1 argument conversions
	 */
	private static class JobCommons1 extends JobCommons
	{
		Sequence in;
	}
	
	/**
	 * Common parameters for 2 argument conversions
	 */
	private static class JobCommons2 extends JobCommons
	{
		Sequence in1;
		Sequence in2;
	}
	
	/**
	 * Base job class
	 */
	private abstract static class Job implements Runnable
	{
		protected int _t;
		protected int _z;
		private JobCommons _jc;
		
		/**
		 * Constructor
		 */
		public Job(int t, int z, JobCommons jc)
		{
			_t = t;
			_z = z;
			_jc = jc;
		}

		@Override
		public void run()
		{
			try {
				doTheJob();
			}
			catch(Controller.CanceledByUser err) {
				_jc.cancelException = err;
			}
		}
		
		/**
		 * Return the data (converted to double if necessary) corresponding to the
		 * given (t,z) coordinates
		 */
		protected static double[][] getInputDataXYC(int t, int z, Sequence in)
		{
			DataType type = in.getDataType_();
			if(type==DataType.DOUBLE)
				return in.getDataXYCAsDouble(t, z);
			else
				return Array2DUtil.arrayToDoubleArray(in.getDataXYC(t, z), type.isSigned());
		}
		
		/**
		 * Actual job
		 */
		public abstract void doTheJob() throws Controller.CanceledByUser;
	}
	
	/**
	 * Job class (1 argument)
	 */
	private static class Job1 extends Job
	{	
		private JobCommons1 _jc1;
		
		/**
		 * Constructor
		 */
		public Job1(int t, int z, JobCommons1 jc1)
		{
			super(t, z, jc1);
			_jc1 = jc1;
		}

		@Override
		public void doTheJob() throws Controller.CanceledByUser
		{
			_jc1.controller.checkPoint();
			double[][] dataIn  = getInputDataXYC(_t, _z, _jc1.in);
			double[][] dataOut = _jc1.out.getDataXYCAsDouble(_t, _z);
			_jc1.fun.apply1(dataOut, dataIn, _jc1.sizeComplexC, _jc1.sizeXY);
		}
	}
	
	/**
	 * Job class (1 arguments)
	 */
	private static class Job2 extends Job
	{
		private JobCommons2 _jc2;
		
		/**
		 * Constructor
		 */
		public Job2(int t, int z, JobCommons2 jc2)
		{
			super(t, z, jc2);
			_jc2 = jc2;
		}

		@Override
		public void doTheJob() throws Controller.CanceledByUser
		{
			_jc2.controller.checkPoint();
			double[][] dataIn1 = getInputDataXYC(_t, _z, _jc2.in1);
			double[][] dataIn2 = getInputDataXYC(_t, _z, _jc2.in2);
			double[][] dataOut = _jc2.out.getDataXYCAsDouble(_t, _z);
			_jc2.fun.apply2(dataOut, dataIn1, dataIn2, _jc2.sizeComplexC, _jc2.sizeXY);
		}
	}
	
	/**
	 * Job builder interface
	 */
	private static interface JobBuilder
	{
		/**
		 * Build a job corresponding the given (t,z) coordinates
		 */
		public Job build(int t, int z);
	}
}
