package plugins.tprovoost.note;

import icy.canvas.Layer;
import icy.gui.component.ColorComponent;
import icy.gui.component.button.IcyButton;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.gui.frame.progress.ToolTipFrame;
import icy.gui.main.MainAdapter;
import icy.gui.main.MainEvent;
import icy.gui.viewer.Viewer;
import icy.main.Icy;
import icy.painter.Anchor2D;
import icy.painter.Painter;
import icy.preferences.IcyPreferences;
import icy.preferences.XMLPreferences;
import icy.resource.icon.IcyIcon;
import icy.sequence.Sequence;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceEvent.SequenceEventSourceType;
import icy.sequence.SequenceListener;
import icy.util.XMLUtil;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;

import javax.swing.BoxLayout;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.WindowConstants;
import javax.swing.border.EmptyBorder;
import javax.swing.border.TitledBorder;
import javax.swing.event.CaretEvent;
import javax.swing.event.CaretListener;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

public class AnnotationFrame extends IcyFrame implements SequenceListener {

	private static AnnotationFrame singleton = new AnnotationFrame();

	// PREFERENCES
	private XMLPreferences prefs = IcyPreferences.pluginsRoot().node("Annotation");
	private static String PREF_FONT_SIZE = "fontSize";
	private static String PREF_COLOR_RGB = "color";
	private static String PREF_FONT = "font";

	// GUI
	private JTextField tfSize;
	private JTextArea textArea;
	private JComboBox comboFont;
	private Font[] allFonts = GraphicsEnvironment.getLocalGraphicsEnvironment().getAllFonts();
	private int fontSize;
	private Color color;
	private JComboBox comboStyle;
	private Note currentNote = null;

	private ColorComponent colorComp;

	private JComboBox cboxPainters;

	private boolean editing = false;

	private AnnotationFrame() {
		super("Note", true, true, true, true);
		// ----------------
		// Load Preferences
		// ----------------
		fontSize = prefs.getInt(PREF_FONT_SIZE, 12);
		color = new Color(prefs.getInt(PREF_COLOR_RGB, 0));
		int idxFont = prefs.getInt(PREF_FONT, 0);

		// --------------
		// INITIALIZE GUI
		// --------------
		JPanel mainPanel = new JPanel();
		add(mainPanel);
		mainPanel.setLayout(new BorderLayout());

		JPanel panelCenter = new JPanel();
		panelCenter.setBorder(new TitledBorder("Notes:"));
		panelCenter.setLayout(new BorderLayout());

		mainPanel.add(panelCenter, BorderLayout.CENTER);

		JPanel panelText = new JPanel();
		panelText.setBorder(new EmptyBorder(4, 4, 4, 4));
		panelCenter.add(panelText);
		panelText.setLayout(new BorderLayout());

		JLabel lblText = new JLabel("Text:");
		panelText.add(lblText, BorderLayout.NORTH);

		textArea = new JTextArea("");
		textArea.addCaretListener(new CaretListener() {

			@Override
			public void caretUpdate(CaretEvent e) {
				Viewer v = Icy.getMainInterface().getFocusedViewer();
				if (v != null && currentNote != null) {
					currentNote.setText(textArea.getText());
					currentNote.refreshROI();
				}
			}

		});
		JScrollPane jScrollPane1 = new JScrollPane(textArea);
		panelText.add(jScrollPane1);

		String[] allFontsStr = new String[allFonts.length];
		for (int i = 0; i < allFontsStr.length; ++i)
			allFontsStr[i] = allFonts[i].getFontName();

		JPanel panelSouth = new JPanel();
		panelCenter.add(panelSouth, BorderLayout.SOUTH);
		panelSouth.setLayout(new BoxLayout(panelSouth, BoxLayout.Y_AXIS));

		JPanel panelFont = new JPanel();
		panelFont.setBorder(new EmptyBorder(4, 0, 4, 0));
		panelSouth.add(panelFont);
		panelFont.setLayout(new BorderLayout(0, 0));
		comboFont = new JComboBox(allFontsStr);
		if (idxFont < allFontsStr.length)
			comboFont.setSelectedIndex(idxFont);
		comboFont.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				if (currentNote != null) {
					Font f = currentNote.getFont();
					currentNote.setFont(new Font((String) comboFont.getSelectedItem(), f.getStyle(), f.getSize()));
					currentNote.refreshROI();
				}
				textArea.setFont(new Font((String) comboFont.getSelectedItem(), comboStyle.getSelectedIndex(), textArea.getFont().getSize()));
				prefs.putInt(PREF_FONT, comboFont.getSelectedIndex());
			}

		});

		JLabel lblFont = new JLabel("Font:");
		panelFont.add(lblFont, BorderLayout.NORTH);
		panelFont.add(comboFont);

		JPanel panelOptions = new JPanel();
		panelSouth.add(panelOptions);
		panelOptions.setLayout(new BoxLayout(panelOptions, BoxLayout.X_AXIS));

		JPanel panelSize = new JPanel();
		panelSize.setBorder(new EmptyBorder(0, 0, 4, 0));
		panelOptions.add(panelSize);

		tfSize = new JTextField();
		tfSize.setHorizontalAlignment(SwingConstants.RIGHT);
		tfSize.setText("" + fontSize);
		tfSize.setColumns(3);
		tfSize.addKeyListener(new KeyAdapter() {

			@Override
			public void keyPressed(KeyEvent e) {
				if (e.getKeyCode() == KeyEvent.VK_ENTER) {
					try {
						fontSize = Integer.valueOf(tfSize.getText()).intValue();
						if (currentNote != null) {
							Font f = currentNote.getFont();
							currentNote.setFont(new Font(f.getFontName(), f.getStyle(), fontSize));
							currentNote.refreshROI();
						}
						prefs.putInt(PREF_FONT_SIZE, fontSize);
					} catch (NumberFormatException e1) {
					}
				}
			}
		});
		panelSize.setLayout(new GridLayout(0, 1, 0, 0));

		JLabel lblSize = new JLabel("Size:");
		panelSize.add(lblSize);
		panelSize.add(tfSize);

		JPanel panelStyle = new JPanel();
		panelStyle.setBorder(new EmptyBorder(0, 4, 4, 4));
		panelOptions.add(panelStyle);

		comboStyle = new JComboBox();
		comboStyle.setModel(new DefaultComboBoxModel(new String[] { "Regular", "Bold", "Italic", "Bold/Italic" }));
		comboStyle.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				if (currentNote != null) {
					Font f = currentNote.getFont();
					currentNote.setFont(new Font(f.getFontName(), comboStyle.getSelectedIndex(), f.getSize()));
					currentNote.refreshROI();
				}
				textArea.setFont(new Font((String) comboFont.getSelectedItem(), comboStyle.getSelectedIndex(), textArea.getFont().getSize()));
			}
		});
		panelStyle.setLayout(new GridLayout(0, 1, 0, 0));

		JLabel lblStyle = new JLabel("Style:");
		panelStyle.add(lblStyle);
		panelStyle.add(comboStyle);

		JPanel panelColor = new JPanel();
		panelColor.setBorder(new EmptyBorder(0, 4, 4, 4));

		JLabel lblColor = new JLabel("Color:");
		panelColor.add(lblColor);

		colorComp = new ColorComponent(color);
		colorComp.setPreferredSize(new Dimension(80, 18));
		colorComp.addMouseListener(new MouseAdapter() {

			@Override
			public void mousePressed(MouseEvent e) {
				final JColorChooser tcc = new JColorChooser(color);
				tcc.getSelectionModel().addChangeListener(new ChangeListener() {

					@Override
					public void stateChanged(ChangeEvent e) {
						color = tcc.getColor();
						colorComp.setColor(color);
						colorComp.repaint();
						if (currentNote != null) {
							currentNote.setColor(color);
							currentNote.refreshROI();
						}
						prefs.putInt(PREF_COLOR_RGB, color.getRGB());
					}
				});
				JDialog jd = JColorChooser.createDialog(Icy.getMainInterface().getMainFrame(), "Choose color", true, tcc, null, null);
				jd.setVisible(true);
			}
		});
		panelColor.setLayout(new GridLayout(0, 1, 0, 0));
		panelColor.add(colorComp);
		colorComp.setLayout(new GridLayout(0, 1, 0, 0));

		panelOptions.add(panelColor);

		JPanel panelAlignment = new JPanel();
		panelAlignment.setBorder(new EmptyBorder(4, 0, 4, 0));
		panelSouth.add(panelAlignment);
		panelAlignment.setLayout(new GridLayout(1, 3, 0, 0));

		JButton btnLeft = new JButton("Left");
		panelAlignment.add(btnLeft);

		JButton btnCenter = new JButton("Cent");
		btnCenter.setEnabled(false);
		panelAlignment.add(btnCenter);

		JButton btnRight = new JButton("Right");
		btnRight.setEnabled(false);
		panelAlignment.add(btnRight);

		JButton btnJustified = new JButton("Just");
		btnJustified.setEnabled(false);
		panelAlignment.add(btnJustified);

		JPanel panelButtons = new JPanel();
		panelButtons.setBorder(new EmptyBorder(0, 0, 4, 0));
		panelSouth.add(panelButtons);

		panelButtons.setLayout(new GridLayout(0, 1, 0, 0));

		JButton btnCreate = new JButton("Create");
		panelButtons.add(btnCreate);
		btnCreate.setToolTipText("Create a Note.");

		JPanel panelPainters = new JPanel();
		panelPainters.setBorder(new TitledBorder("Painters:"));
		panelPainters.setLayout(new GridLayout(0, 1, 0, 0));
		mainPanel.add(panelPainters, BorderLayout.NORTH);

		JPanel panel = new JPanel();
		panel.setBorder(new EmptyBorder(0, 0, 0, 0));
		panelPainters.add(panel);
		panel.setLayout(new BorderLayout(0, 0));

		cboxPainters = new JComboBox();
		panel.add(cboxPainters);

		IcyButton btnReload = new IcyButton(new IcyIcon("rot_unclock"));
		panel.add(btnReload, BorderLayout.EAST);
		btnReload.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				updateComboPainters();
			}
		});

		final JButton btnCreateNewPainter = new JButton("Create New Painter for Notes");
		panelPainters.add(btnCreateNewPainter);
		btnCreateNewPainter.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				Sequence s = Icy.getMainInterface().getFocusedSequence();
				if (s != null) {
					PainterNotes painter = new PainterNotes();
					s.addPainter(painter);
					renamePainter(painter, "Painter Notes");
					updateComboPainters();
				}
			}
		});

		btnCreate.addActionListener(new ActionListener() {

			@Override
			public void actionPerformed(ActionEvent e) {
				Sequence s = Icy.getMainInterface().getFocusedSequence();
				if (s != null) {
					Note n = new Note(s, textArea.getText());
					Anchor2D r = new Anchor2D(s.getWidth() / 2, s.getHeight() / 2);
					n.setAnchor(r);
					n.setColor(colorComp.getColor());
					n.setFont(new Font((String) comboFont.getSelectedItem(), comboStyle.getSelectedIndex(), fontSize));
					n.refreshROI();
					PainterNotes painter = (PainterNotes) cboxPainters.getSelectedItem();
					if (painter == null) {
						painter = new PainterNotes();
						s.addPainter(painter);
						renamePainter(painter, "Painter Notes");
						updateComboPainters();
					}
					painter.addNote(n);
					setCurrentNote(n);
				}
			}
		});
		setSize(240, 440);

		setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
		addFrameListener(new IcyFrameAdapter() {
			@Override
			public void icyFrameClosing(IcyFrameEvent e) {
				setVisible(false);
			}
		});
		addToMainDesktopPane();
		requestFocus();

		for (Sequence s : Icy.getMainInterface().getSequences()) {
			loadSequencePaintersNotes(s);
			s.addListener(AnnotationFrame.this);
		}
		updateComboPainters();
		Icy.getMainInterface().addListener(new MainAdapter() {

			@Override
			public void sequenceOpened(MainEvent event) {
				super.sequenceOpened(event);
				Sequence s = (Sequence) event.getSource();
				loadSequencePaintersNotes(s);
				updateComboPainters();
				s.addListener(AnnotationFrame.this);
			}

			@Override
			public void sequenceClosed(MainEvent event) {
				super.sequenceClosed(event);
				Sequence s = (Sequence) event.getSource();
				saveSequencePaintersNotes(s);
				updateComboPainters();
			}

			@Override
			public void sequenceFocused(MainEvent event) {
				updateComboPainters();
			}
		});
	}

	protected void updateComboPainters() {
		Object previouslySelected = cboxPainters.getSelectedItem();
		Sequence s = Icy.getMainInterface().getFocusedSequence();
		ArrayList<PainterNotes> notes = new ArrayList<PainterNotes>();

		if (s != null) {
			for (Painter p : s.getPainters()) {
				if (p instanceof PainterNotes)
					notes.add((PainterNotes) p);
			}
		}
		cboxPainters.setModel(new DefaultComboBoxModel(notes.toArray()));
		cboxPainters.setSelectedItem(previouslySelected);
		if (cboxPainters.getSelectedItem() == null && cboxPainters.getItemCount() > 0)
			cboxPainters.setSelectedIndex(0);
		cboxPainters.repaint();
	}

	protected void saveSequencePaintersNotes(Sequence s) {
		Node nSeq = s.setNode("Annotations");
		XMLUtil.removeAllChildren(nSeq);
		updateComboPainters();
		for (Painter painter : s.getPainters()) {
			if (painter instanceof PainterNotes) {
				PainterNotes pn = (PainterNotes) painter;
				Element elem = XMLUtil.addElement(nSeq, "PainterNote");
				elem.setAttribute("Name", pn.toString());
				for (Note note : pn.getNotes()) {
					note.saveToXML(XMLUtil.addElement(elem, "Note"));
				}
			}
		}
	}

	protected void loadSequencePaintersNotes(Sequence s) {
		Node nSeq = s.getNode("Annotations");
		ArrayList<PainterNotes> painterNotesList = new ArrayList<PainterNotes>();
		if (nSeq != null) {
			for (Element elementPainterNotes : XMLUtil.getSubElements(nSeq, "PainterNote")) {
				PainterNotes pn = new PainterNotes();
				String name = elementPainterNotes.getAttribute("Name");
				for (Element elementNote : XMLUtil.getSubElements(elementPainterNotes, "Note")) {
					Note note = new Note(s);
					note.loadFromXML(elementNote);
					pn.addNote(note);
				}
				painterNotesList.add(pn);
				s.addPainter(pn);
				if (name.contentEquals(""))
					renamePainter(pn, "Painter Notes");
				else
					renamePainter(pn, name);
			}
		}
	}

	public Note getCurrentPainter() {
		return currentNote;
	}

	/**
	 * Assign current note and add the anchor to the sequence.<br/>
	 * Remove the previous anchor is necessary.
	 * 
	 * @param seq
	 * @param note
	 */
	public void setCurrentNote(Note note) {
		// Remove previous painter
		if (currentNote != null)
			currentNote.attachedSequence.removePainter(currentNote.getAnchor());

		// Assign new value
		this.currentNote = note;

		if (note != null) {
			note.attachedSequence.addPainter(note.getAnchor());
			renamePainter(note.getAnchor(), "Anchor");

			// Set values in frame
			String s = note.getText();
			if (!textArea.getText().equals(s))
				textArea.setText(note.getText());
			Font font = note.getFont();
			Color color = note.getColor();
			comboFont.setSelectedItem(font.getFontName());
			comboStyle.setSelectedIndex(font.getStyle());
			tfSize.setText("" + font.getSize());
			colorComp.setColor(color);
			textArea.setFont(new Font(font.getFontName(), font.getStyle(), textArea.getFont().getSize()));
			this.color = color;
		}
	}

	public static void renamePainter(Painter painter, String name) {
		for (Sequence s : Icy.getMainInterface().getSequences())
			for (Viewer v : s.getViewers()) {
				Layer l = v.getCanvas().getLayer(painter);
				if (l != null)
					l.setName(name);
			}
	}

	@Override
	public void setVisible(boolean value) {
		super.setVisible(value);
		if (value) {
			new ToolTipFrame("<html><h3>Version warning</h3><body>This version is no longer compatible with previous XML saved notes.<br/>"
					+ "To fix this issue, please go on the documentation page of the plugin on Icy Website.</body></html>","annotationWarning");
		} else {
			setCurrentNote(null);
		}
	}

	public static AnnotationFrame getAnnotationFrame() {
		return singleton;
	}

	@Override
	public void sequenceChanged(SequenceEvent sequenceEvent) {
		if (sequenceEvent.getSourceType() == SequenceEventSourceType.SEQUENCE_PAINTER) {
			updateComboPainters();
		}
	}

	@Override
	public void sequenceClosed(Sequence sequence) {
		sequence.removeListener(this);
	}

	boolean isEditing() {
		return editing;
	}

	void setEditing(boolean b) {
		editing = b;
	}

}
