/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.process.process2D;

import icy.image.IcyBufferedImage;
import icy.type.collection.array.Array1DUtil;

/**
 * Fast implementation of a Gaussian filtering based on successive iterations of
 * a bilateral exponential filter.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 */
public class FastGaussianFilter {

	/** Minimum tolerance value. */
	private static double TINY = Float.intBitsToFloat(0x33FFFFFF);

	/**
	 * Number of iterations of the exponential filter used to approximate the
	 * Gaussian filter.
	 */
	private static int DEFAULTEXPONENTIALFILTERITERATIONS = 15;

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * Applies a fast implementation of a 2D Gaussian filter with a given
	 * standard deviation.
	 */
	public static void smooth(IcyBufferedImage im, double standardDeviation) {
		smooth(im, standardDeviation, DEFAULTEXPONENTIALFILTERITERATIONS, TINY);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Applies a fast implementation of a 2D Gaussian filter with a given
	 * standard deviation and tolerance.
	 */
	public static void smooth(IcyBufferedImage im, double standardDeviation,
			int iterationsOfExponentialFilter, double tolerance) {
		double[] pixels = Array1DUtil.arrayToDoubleArray(im.getDataXY(0), im
				.getDataType_().isSigned());
		smooth(pixels, im.getWidth(), im.getHeight(), standardDeviation,
				iterationsOfExponentialFilter, tolerance);
		Array1DUtil.doubleArrayToArray(pixels, im.getDataXY(0));
	}

	// ----------------------------------------------------------------------------

	/**
	 * Applies a fast implementation of a 2D Gaussian filter with a given
	 * standard deviation.
	 */
	public static void smooth(double[] pixels, int width, int height,
			double standardDeviation) {
		smooth(pixels, width, height, standardDeviation,
				DEFAULTEXPONENTIALFILTERITERATIONS, TINY);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Applies a fast implementation of a 2D Gaussian filter with a given
	 * standard deviation and tolerance.
	 */
	public static void smooth(double[] pixels, int width, int height,
			double standardDeviation, int iterationsOfExponentialFilter,
			double tolerance) {
		double[] row = new double[width];
		for (int y = 0; y < height; y++) {
			extractRow(pixels, y, row);
			gaussianFilter(row, standardDeviation,
					iterationsOfExponentialFilter, tolerance);
			putRow(pixels, y, row);
		}
		double[] column = new double[height];
		for (int x = 0; x < width; x++) {
			extractColumn(pixels, width, x, column);
			gaussianFilter(column, standardDeviation,
					iterationsOfExponentialFilter, tolerance);
			putColumn(pixels, width, x, column);
		}
	}

	// ============================================================================
	// PRIVATE METHODS

	/** Extracts a particular columns of an image expressed as a single array. */
	private static void extractColumn(double[] array, int width, int x,
			double[] column) {
		for (int k = 0; k < column.length; k++) {
			column[k] = array[x];
			x += width;
		}
	}

	// ----------------------------------------------------------------------------

	/** Extracts a particular row of an image expressed as a single array. */
	private static void extractRow(double[] array, int y, double[] row) {
		y *= row.length;
		for (int k = 0; k < row.length; k++) {
			row[k] = array[y++];
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Applies a fast implementation of a 1D Gaussian filter with a given
	 * standard deviation and tolerance.
	 */
	private static void gaussianFilter(double[] line, double standardDeviation,
			int iterationsOfExponentialFilter, double tolerance) {
		if (standardDeviation <= 0.0) {
			return;
		}

		double variance = standardDeviation * standardDeviation;
		double e = 1.0
				+ (Math.sqrt(iterationsOfExponentialFilter) - Math
						.sqrt(iterationsOfExponentialFilter + 2.0 * variance))
				* Math.sqrt(iterationsOfExponentialFilter) / variance;
		for (int i = 0; i < iterationsOfExponentialFilter; i++) {
			double sum = 0.0;
			double e1 = 1.0;
			double e2 = Math.pow(e, 2.0 * line.length - 1.0);
			int horizon = line.length;
			if (0.0 < tolerance) {
				horizon = 2 + (int) (Math.log(tolerance) / Math
						.log(Math.abs(e)));
				horizon = horizon < line.length ? horizon : line.length;
			}
			for (int k = 0; k < horizon; k++) {
				sum += (e1 + e2) * line[k];
				e1 *= e;
				e2 /= e;
			}
			line[0] += sum * e / (1.0 - Math.pow(e, 2.0 * line.length));
			for (int k = 1; k < line.length; k++) {
				line[k] += e * line[k - 1];
			}
			line[line.length - 1] /= 1.0 - e;
			for (int k = line.length - 2; 0 <= k; k--) {
				line[k] += e * line[k + 1];
			}
		}
		double normalisation = Math.pow(1.0 - e,
				2.0 * iterationsOfExponentialFilter);
		for (int k = 0; k < line.length; k++) {
			line[k] *= normalisation;
		}
	}

	// ----------------------------------------------------------------------------

	/** Sets a particular columns of an image expressed as a single array. */
	private static void putColumn(double[] array, int width, int x,
			double[] column) {
		for (double element : column) {
			array[x] = element;
			x += width;
		}
	}

	// ----------------------------------------------------------------------------

	/** Sets a particular row of an image expressed as a single array. */
	private static void putRow(double[] array, int y, double[] row) {
		y *= row.length;
		for (double element : row) {
			array[y++] = element;
		}
	}
}
