package net.imglib2.algorithm.binary;

import java.util.Vector;

import net.imglib2.Cursor;
import net.imglib2.RandomAccess;
import net.imglib2.converter.Converter;
import net.imglib2.exception.IncompatibleTypeException;
import net.imglib2.img.Img;
import net.imglib2.img.ImgFactory;
import net.imglib2.multithreading.Chunk;
import net.imglib2.multithreading.SimpleMultiThreading;
import net.imglib2.type.Type;
import net.imglib2.type.logic.BitType;

/**
 * Collection of static utilities meant to generate {@link BitType} images from
 * {@link Comparable} images.
 *
 * @author Jean-Yves Tinevez
 */
public class Thresholder
{

	/**
	 * Returns a new boolean {@link Img} generated by thresholding the values of
	 * the source image.
	 *
	 * @param source
	 *            the image to threshold.
	 * @param threshold
	 *            the threshold.
	 * @param above
	 *            if <code>true</code>, the target value will be true for source
	 *            values above the threshold, <code>false</code> otherwise.
	 * @param numThreads
	 *            the number of threads to use for thresholding.
	 * @return a new {@link Img} of type {@link BitType} and of same dimension
	 *         that the source image.
	 */
	public static final < T extends Type< T > & Comparable< T >> Img< BitType > threshold( final Img< T > source, final T threshold, final boolean above, final int numThreads )
	{
		final ImgFactory< T > factory = source.factory();
		try
		{
			final ImgFactory< BitType > bitFactory = factory.imgFactory( new BitType() );
			final Img< BitType > target = bitFactory.create( source, new BitType() );

			final Converter< T, BitType > converter;
			if ( above )
			{
				converter = new Converter< T, BitType >()
				{
					@Override
					public void convert( final T input, final BitType output )
					{
						output.set( input.compareTo( threshold ) > 0 );
					}
				};
			}
			else
			{
				converter = new Converter< T, BitType >()
				{
					@Override
					public void convert( final T input, final BitType output )
					{
						output.set( input.compareTo( threshold ) < 0 );
					}
				};
			}

			final Vector< Chunk > chunks = SimpleMultiThreading.divideIntoChunks( target.size(), numThreads );
			final Thread[] threads = SimpleMultiThreading.newThreads( numThreads );

			if ( target.iterationOrder().equals( source.iterationOrder() ) )
			{
				for ( int i = 0; i < threads.length; i++ )
				{
					final Chunk chunk = chunks.get( i );
					threads[ i ] = new Thread( "Thresholder thread " + i )
					{
						@Override
						public void run()
						{
							final Cursor< BitType > cursorTarget = target.cursor();
							cursorTarget.jumpFwd( chunk.getStartPosition() );
							final Cursor< T > cursorSource = source.cursor();
							cursorSource.jumpFwd( chunk.getStartPosition() );
							for ( long steps = 0; steps < chunk.getLoopSize(); steps++ )
							{
								cursorTarget.fwd();
								cursorSource.fwd();
								converter.convert( cursorSource.get(), cursorTarget.get() );
							}
						}
					};
				}
			}
			else
			{
				for ( int i = 0; i < threads.length; i++ )
				{
					final Chunk chunk = chunks.get( i );
					threads[ i ] = new Thread( "Thresholder thread " + i )
					{
						@Override
						public void run()
						{
							final Cursor< BitType > cursorTarget = target.cursor();
							cursorTarget.jumpFwd( chunk.getStartPosition() );
							final RandomAccess< T > ra = source.randomAccess( target );
							for ( long steps = 0; steps < chunk.getLoopSize(); steps++ )
							{
								cursorTarget.fwd();
								ra.setPosition( cursorTarget );
								converter.convert( ra.get(), cursorTarget.get() );
							}
						}
					};
				}
			}

			SimpleMultiThreading.startAndJoin( threads );
			return target;
		}
		catch ( final IncompatibleTypeException e )
		{
			e.printStackTrace();
			return null;
		}
	}

}