/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner;

import icy.gui.component.BorderedPanel;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.gui.frame.progress.AnnounceFrame;
import icy.gui.main.ActiveViewerListener;
import icy.gui.main.GlobalViewerListener;
import icy.gui.viewer.Viewer;
import icy.gui.viewer.ViewerEvent;
import icy.main.Icy;
import icy.plugin.abstract_.PluginActionable;
import icy.resource.ResourceUtil;
import icy.sequence.Sequence;

import java.awt.CardLayout;
import java.awt.Container;
import java.awt.GridLayout;
import java.awt.Polygon;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map.Entry;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.swing.AbstractAction;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.KeyStroke;
import javax.swing.SwingConstants;

import jxl.write.WriteException;
import jxl.write.biff.RowsExceededException;
import plugins.big.shapedesigner.gui.PlugInFrame;
import plugins.big.shapedesigner.gui.SplineCurveSettingsPane;
import plugins.big.shapedesigner.gui.ToolTipsMessages;
import plugins.big.shapedesigner.io.IOXML;
import plugins.big.shapedesigner.keeper.KeepersList;
import plugins.big.shapedesigner.keeper.SplineCurveKeeper;
import plugins.big.shapedesigner.roi.ROIParser;
import plugins.big.shapedesigner.roi.SplineCurveEditMode;
import plugins.big.shapedesigner.splinecurve.SplineCurve;
import plugins.big.shapedesigner.splinecurve.SplineCurveParameters;

/**
 * Plug-in for Icy that allows the user to design spline shapes by moving
 * control points.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class ShapeDesigner extends PluginActionable implements ActionListener,
		ActiveViewerListener, GlobalViewerListener, KeyListener {

	/** Reference to the main plug-in class. */
	private ShapeDesigner mainPlugin_ = null;

	// ----------------------------------------------------------------------------
	// SPLINE CURVES

	/** Map that associates each open image with a list of spline curves. */
	private final HashMap<Sequence, KeepersList> keepersListTable_ = new HashMap<Sequence, KeepersList>();
	/** Lock used to preserve synchronization among different open images. */
	private final Lock keepersListLock_ = new ReentrantLock();
	/**
	 * Index that counts the number of spline curves added since the plug-in was
	 * created.
	 */
	private int nSplineCurves_ = 0;

	// ----------------------------------------------------------------------------
	// IMAGES AND VIEWERS

	/**
	 * Map that keeps track of which <code>Viewer</code> have a
	 * <code>KeyListeners</code> attached.
	 */
	private final HashMap<Viewer, Boolean> hasKeyListenerTable_ = new HashMap<Viewer, Boolean>();

	// ----------------------------------------------------------------------------
	// GUI

	/** Main frame of the plug-in. */
	private final PlugInFrame plugInMainFrame_ = new PlugInFrame(this);
	/** Pane that contains the settings related to the spline curves. */
	private SplineCurveSettingsPane splineCurveSettingsPane_ = null;
	/** Panel that contains the settings of the active spline curve. */
	private final JPanel splineCurveParametersPanel_ = new BorderedPanel();
	/** Label in which the tooltips are shown. */
	private final JLabel toolTipMessagesLabel_ = new JLabel(
			ToolTipsMessages.getToolTipMessage(0), SwingConstants.CENTER);

	// ----------------------------------------------------------------------------
	// ACTION BUTTONS

	/**
	 * Button that sets the interaction mode with the spline curve to
	 * translation mode (control points or whole spline curve).
	 */
	private final JButton moveSplineCurveButton_ = new JButton("");
	/**
	 * Button that sets the interaction mode with the spline curve to
	 * resizing/scaling mode.
	 */
	private final JButton resizeSplineCurveButton_ = new JButton("");
	/**
	 * Button that sets the interaction mode with the spline curve to rotation
	 * mode.
	 */
	private final JButton rotateSplineCurveButton_ = new JButton("");
	/**
	 * Button that fires the process to create a new spline curve from the
	 * interface parameters.
	 */
	private final JButton createSplineCurveButton_ = new JButton("");
	/**
	 * Button that fires the process of deletion of the active spline curve of
	 * the focused image.
	 */
	private final JButton deleteSplineCurveButton_ = new JButton("");

	/** Size of the icons in the menu. */
	private static int ICONSIZE = 20;

	// ----------------------------------------------------------------------------
	// OTHER

	/** Determines the way the user can interact with the spline curve. */
	private SplineCurveEditMode editingMode_ = null;

	// ============================================================================
	// PUBLIC METHODS

	/** Method executed when launching the plug-in. */
	@Override
	public void run() {

		mainPlugin_ = this;
		splineCurveSettingsPane_ = new SplineCurveSettingsPane("Spline curve",
				mainPlugin_);

		Viewer existedViewer = getActiveViewer();
		if (existedViewer != null) {
			addKeyListenerToViewer(existedViewer);
		}

		// create the main Panel
		Container mainPane = plugInMainFrame_.getContentPane();
		mainPane.setLayout(new BoxLayout(mainPane, BoxLayout.PAGE_AXIS));

		// create the spline curve parameters Panel
		splineCurveParametersPanel_.setLayout(new CardLayout());
		splineCurveParametersPanel_.add("test1", splineCurveSettingsPane_);
		mainPane.add(splineCurveParametersPanel_);

		JPanel interactionPanelHeader = new JPanel();
		interactionPanelHeader.setLayout(new BoxLayout(interactionPanelHeader,
				BoxLayout.LINE_AXIS));
		JLabel splineCurveEditLabel = new JLabel("Spline curve editing");
		interactionPanelHeader.add(splineCurveEditLabel);
		mainPane.add(interactionPanelHeader);

		// create the bar of interaction buttons
		JPanel interactionPanel = new JPanel();

		moveSplineCurveButton_.setIcon(ResourceUtil.getAlphaIcon(
				"cursor_arrow.png", ICONSIZE));
		moveSplineCurveButton_.setToolTipText("Move spline curve");
		moveSplineCurveButton_.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				setEditingMode(SplineCurveEditMode.MOVE_SPLINE_CURVE);
			}
		});

		resizeSplineCurveButton_.setIcon(ResourceUtil.getAlphaIcon(
				"expand.png", ICONSIZE));
		resizeSplineCurveButton_.setToolTipText("Dilate spline curve");
		resizeSplineCurveButton_.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				setEditingMode(SplineCurveEditMode.DILATE_SPLINE_CURVE);
			}
		});

		rotateSplineCurveButton_.setIcon(ResourceUtil.getAlphaIcon(
				"rot_unclock.png", ICONSIZE));
		rotateSplineCurveButton_.setToolTipText("Rotate spline curve");
		rotateSplineCurveButton_.addActionListener(new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				setEditingMode(SplineCurveEditMode.ROTATE_SPLINE_CURVE);
			}
		});

		// default editing mode
		setEditingMode(SplineCurveEditMode.MOVE_SPLINE_CURVE);

		interactionPanel.setLayout(new GridLayout(1, 3));
		interactionPanel.add(moveSplineCurveButton_);
		interactionPanel.add(resizeSplineCurveButton_);
		interactionPanel.add(rotateSplineCurveButton_);

		mainPane.add(interactionPanel);

		// create the action panel
		JPanel actionPanelHeader = new JPanel();
		actionPanelHeader.setLayout(new BoxLayout(actionPanelHeader,
				BoxLayout.LINE_AXIS));
		JLabel actionLabel = new JLabel("Spline curve actions");
		actionPanelHeader.add(actionLabel);
		mainPane.add(actionPanelHeader);

		JPanel actionPane = new JPanel();
		actionPane.setLayout(new GridLayout(1, 3));

		createSplineCurveButton_.setIcon(ResourceUtil.getAlphaIcon("plus.png",
				ICONSIZE));
		createSplineCurveButton_.setToolTipText("Add a new spline curve");

		deleteSplineCurveButton_.setIcon(ResourceUtil.getAlphaIcon("trash.png",
				ICONSIZE));
		deleteSplineCurveButton_.setToolTipText("Remove active spline curve");

		actionPane.add(createSplineCurveButton_);
		actionPane.add(deleteSplineCurveButton_);

		createSplineCurveButton_.addActionListener(this);
		deleteSplineCurveButton_.addActionListener(this);

		// assign DELETE as key binding for deleteSplineCurveButton_
		deleteSplineCurveButton_.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW)
				.put(KeyStroke.getKeyStroke(KeyEvent.VK_DELETE, 0),
						"doDeleteAction");
		deleteSplineCurveButton_.getActionMap().put("doDeleteAction",
				new AbstractAction() {
					private static final long serialVersionUID = -4139948974574644979L;

					@Override
					public void actionPerformed(ActionEvent e) {
						deleteSplineCurveButton_.doClick();
					}
				});

		// assign BACKSPACE as key binding for deleteSplineCurveButton_ (for MAC
		// users)
		deleteSplineCurveButton_.getInputMap(JComponent.WHEN_IN_FOCUSED_WINDOW)
				.put(KeyStroke.getKeyStroke(KeyEvent.VK_BACK_SPACE, 0),
						"doDeleteAction");
		deleteSplineCurveButton_.getActionMap().put("doDeleteAction",
				new AbstractAction() {
					private static final long serialVersionUID = 1720306735845428242L;

					@Override
					public void actionPerformed(ActionEvent e) {
						deleteSplineCurveButton_.doClick();
					}
				});
		mainPane.add(actionPane);

		// create a pane for the selected sequence display
		JPanel sequencePane = new JPanel();
		sequencePane.add(toolTipMessagesLabel_);

		toolTipMessagesLabel_.addMouseListener(new MouseAdapter() {
			@Override
			public void mouseClicked(MouseEvent e) {
				toolTipMessagesLabel_.setText(ToolTipsMessages
						.getToolTipMessage());
			}
		});

		mainPane.add(sequencePane);

		Icy.getMainInterface().addActiveViewerListener(this);
		Icy.getMainInterface().addGlobalViewerListener(this);

		plugInMainFrame_.addFrameListener(new IcyFrameAdapter() {
			@Override
			public void icyFrameClosed(IcyFrameEvent e) {
				Icy.getMainInterface().removeActiveViewerListener(mainPlugin_);
				Icy.getMainInterface().removeGlobalViewerListener(mainPlugin_);
			}
		});

		plugInMainFrame_.pack();

		plugInMainFrame_.addFrameListener(new IcyFrameAdapter() {
			@Override
			public void icyFrameInternalized(IcyFrameEvent e) {
				plugInMainFrame_.pack();
			}

			@Override
			public void icyFrameExternalized(IcyFrameEvent e) {
				plugInMainFrame_.pack();
			}
		});

		addIcyFrame(plugInMainFrame_);
		plugInMainFrame_.center();
		plugInMainFrame_.setVisible(true);

	}

	// ----------------------------------------------------------------------------

	/**
	 * Prepares the plug-in to be terminated. All memory is freed, and auxiliary
	 * plug-ins are closed.
	 */
	public void terminatePlugin() {
		removeAllSplineCurves();
		removeAllKeyListeners();
	}

	// ----------------------------------------------------------------------------
	// SPLINE CURVES CREATION/DESTRUCTION MANAGEMENT

	/**
	 * Returns <code>true</code> if <code>SplineCurveKeeper</code> passed is the
	 * active one.
	 */
	public boolean isActiveSplineCurve(SplineCurveKeeper keeper) {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			return false;
		}
		keepersListLock_.lock();
		try {
			if (keepersListTable_.containsKey(focusedSequence)) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				if (keepersList != null) {
					return keepersList.isActiveSplineCurveKeeper(keeper);
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
		return false;
	}

	// ----------------------------------------------------------------------------

	/** Makes a particular spline curve active and responsive to interactions. */
	public void activateSplineCurve(SplineCurveKeeper keeper) {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			return;
		}
		keepersListLock_.lock();
		try {
			if (keepersListTable_.containsKey(focusedSequence)) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				if (keepersList != null) {
					boolean success = keepersList
							.activateSplineCurveKeeper(keeper);
					splineCurveSettingsPane_.setSplineCurveParameters(keeper
							.getSplineCurveParameters());
					keeper.setSymmetric(splineCurveSettingsPane_.isSymmetric());
					if (!success) {
						System.err
								.println("activateSplineCurve: The SplineCurveKeeper could not be activated.");
					}
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Adds a new spline curve to the active <code>Sequence</code> and makes it
	 * active and responsive to interactions.
	 */
	private void addAndActivateSplineCurve() {
		if (nSplineCurves_ == 0) {
			Sequence focusedSequence = getActiveSequence();
			if (focusedSequence == null) {
				new AnnounceFrame(
						"No image detected. Please, open an image first.");
				return;
			}

			Polygon initialROI = ROIParser.parseFirstFreeROI(focusedSequence
					.getROIs());

			SplineCurveParameters splineCurveParameters = new SplineCurveParameters(
					splineCurveSettingsPane_.getNumControlPoints());

			SplineCurve splineCurve = new SplineCurve(focusedSequence,
					splineCurveParameters, initialROI);
			SplineCurveKeeper keeper = new SplineCurveKeeper(focusedSequence,
					splineCurve, mainPlugin_);
			keeper.setSymmetric(splineCurveSettingsPane_.isSymmetric());
			nSplineCurves_++;
			keeper.setID("" + nSplineCurves_);
			keeper.setSplineCurveEditMode(getEditingMode());
			keepersListLock_.lock();
			try {
				if (keepersListTable_.containsKey(focusedSequence)) {
					KeepersList list = keepersListTable_.get(focusedSequence);
					list.addAndActivateKeeper(keeper);
				} else {
					KeepersList list = new KeepersList();
					list.addAndActivateKeeper(keeper);
					keepersListTable_.put(focusedSequence, list);
				}
			} finally {
				keepersListLock_.unlock();
			}
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Makes a particular spline curve inactive and not responsive to
	 * interaction.
	 */
	public void deactivateSplineCurve(SplineCurveKeeper keeper) {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			return;
		}
		keepersListLock_.lock();
		try {
			if (keepersListTable_.containsKey(focusedSequence)) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				if (keepersList != null) {
					boolean success = keepersList
							.deactivateSplineCurveKeeper(keeper);
					if (!success) {
						System.err
								.println("deactivateSplineCurve: The SplineCurveKeeper could not be activated.");
					}
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/** Removes the active spline curve from the active <code>Sequence</code>. */
	private void removeActiveSplineCurve() {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			new AnnounceFrame("No image detected. Please, open an image first.");
			return;
		}
		keepersListLock_.lock();
		try {
			if (keepersListTable_.containsKey(focusedSequence)) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				if (keepersList != null) {
					keepersList.removeActiveSplineCurveKeeper();
					nSplineCurves_--;
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Clears the spline curves from memory and detaches the ROI's from the
	 * image.
	 */
	private void removeAllSplineCurves() {
		keepersListLock_.lock();
		try {
			for (Entry<Sequence, KeepersList> entry : keepersListTable_
					.entrySet()) {
				KeepersList keepersList = entry.getValue();
				if (keepersList != null) {
					if (!keepersList.isEmpty()) {
						keepersList.removeAllSplineCurveKeepers();
					}
				}
			}
			keepersListTable_.clear();
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Retrieves the spline curve parameters from the GUI and sets them to the
	 * active spline curve.
	 */
	public void loadSplineCurveParametersFromInterface() {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			return;
		}
		keepersListLock_.lock();
		try {
			if (keepersListTable_.containsKey(focusedSequence)) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				if (keepersList != null) {
					SplineCurveKeeper activeSplineCurveKeeper = keepersList
							.getActiveSplineCurveKeeper();
					if (activeSplineCurveKeeper != null) {
						SplineCurveParameters splineCurveParameters = new SplineCurveParameters(
								splineCurveSettingsPane_.getNumControlPoints());
						activeSplineCurveKeeper
								.setSplineCurveParameters(splineCurveParameters);
						activeSplineCurveKeeper
								.setSymmetric(splineCurveSettingsPane_
										.isSymmetric());
					}
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------
	// I/O

	/** Load the spline curves from an XML file. */
	public void loadSplineCurvesFromXML() {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			new AnnounceFrame("No sequence loaded. Please open an image first.");
			return;
		}

		keepersListLock_.lock();
		try {
			if (keepersListTable_.containsKey(focusedSequence)) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				keepersList.deactivateActiveSplineCurveKeeper();
			}
		} finally {
			keepersListLock_.unlock();
		}

		try {
			ArrayList<SplineCurveKeeper> loadedKeepers = IOXML
					.loadSplineCurvesFromXML(focusedSequence, this);
			for (SplineCurveKeeper loadedKeeper : loadedKeepers) {
				if (loadedKeeper != null) {
					SplineCurveParameters splineCurveParameters = loadedKeeper
							.getSplineCurveParameters();
					splineCurveSettingsPane_
							.setSplineCurveParameters(splineCurveParameters);
					nSplineCurves_++;
					loadedKeeper.setID("" + nSplineCurves_);
					loadedKeeper.setSplineCurveEditMode(getEditingMode());
					loadedKeeper.setSymmetric(splineCurveSettingsPane_
							.isSymmetric());
					keepersListLock_.lock();
					try {
						if (keepersListTable_.containsKey(focusedSequence)) {
							KeepersList list = keepersListTable_
									.get(focusedSequence);
							list.addAndActivateKeeper(loadedKeeper);
						} else {
							KeepersList list = new KeepersList();
							list.addAndActivateKeeper(loadedKeeper);
							keepersListTable_.put(focusedSequence, list);
						}
					} finally {
						keepersListLock_.unlock();
					}
					deactivateSplineCurve(loadedKeeper);
				}
			}
		} catch (Exception e) {
			new AnnounceFrame(
					"The spline curves could not be loaded. XML file is corrupt or non-valid.");
			System.err
					.println("The spline curves could not be loaded. XML file is corrupt or non-valid.");
			e.printStackTrace();
		}
	}

	// ----------------------------------------------------------------------------

	/** Saves the spline curves to an XML file. */
	public void saveSplineCurvesToXML() {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			new AnnounceFrame("No image detected. Please, open an image first.");
			return;
		}

		if (nSplineCurves_ <= 0) {
			new AnnounceFrame(
					"No spline curves detected. Please, add one first.");
			return;
		}

		try {
			IOXML.saveSplineCurvesToXML(focusedSequence);
		} catch (RowsExceededException e) {
			new AnnounceFrame(
					"An error occurred while saving the spline curves. XML file is corrupt or non-valid.");
			System.err
					.println("An error occurred while saving the spline curves. XML file is corrupt or non-valid.");
			e.printStackTrace();
		} catch (WriteException e) {
			new AnnounceFrame(
					"An error occurred while saving the spline curves. XML file is corrupt or non-valid.");
			System.err
					.println("An error occurred while saving the spline curves. XML file is corrupt or non-valid.");
			e.printStackTrace();
		} catch (Exception e) {
			new AnnounceFrame(
					"An error occurred while saving the spline curves. XML file is corrupt or non-valid.");
			System.err
					.println("An error occurred while saving the spline curves. XML file is corrupt or non-valid.");
			e.printStackTrace();
		}
	}

	// ----------------------------------------------------------------------------
	// GUI METHODS

	/** Retrieves the interaction method specified on the GUI. */
	private SplineCurveEditMode getEditingMode() {
		if (moveSplineCurveButton_.isSelected()) {
			return SplineCurveEditMode.MOVE_SPLINE_CURVE;
		}
		if (resizeSplineCurveButton_.isSelected()) {
			return SplineCurveEditMode.DILATE_SPLINE_CURVE;
		}
		if (rotateSplineCurveButton_.isSelected()) {
			return SplineCurveEditMode.ROTATE_SPLINE_CURVE;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	/** Sets an interaction method to the GUI. */
	private void setEditingMode(SplineCurveEditMode editingMode) {
		switch (editingMode) {
		case DILATE_SPLINE_CURVE:
			resizeSplineCurveButton_.setSelected(true);
			moveSplineCurveButton_.setSelected(false);
			rotateSplineCurveButton_.setSelected(false);
			break;
		case MOVE_SPLINE_CURVE:
			moveSplineCurveButton_.setSelected(true);
			resizeSplineCurveButton_.setSelected(false);
			rotateSplineCurveButton_.setSelected(false);
			break;
		case ROTATE_SPLINE_CURVE:
			rotateSplineCurveButton_.setSelected(true);
			resizeSplineCurveButton_.setSelected(false);
			moveSplineCurveButton_.setSelected(false);
			break;
		}
		if (editingMode == editingMode_) {
			return;
		}
		editingMode_ = editingMode;
		keepersListLock_.lock();
		try {
			for (KeepersList keeperList : keepersListTable_.values()) {
				if (keeperList != null) {
					keeperList.setSplineCurveEditMode(editingMode);
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------
	// SPLINE CURVE EDITING

	/**
	 * Shifts the active spline curve within the active <code>Sequence</code> by
	 * a vertical and horizontal offset.
	 */
	private void translateActiveSplineCurve(int dx, int dy) {
		Sequence focusedSequence = getActiveSequence();
		if (focusedSequence == null) {
			return;
		}

		keepersListLock_.lock();
		try {
			if (keepersListTable_.get(focusedSequence) != null) {
				KeepersList keepersList = keepersListTable_
						.get(focusedSequence);
				if (keepersList != null) {
					SplineCurveKeeper activeKeeper = keepersList
							.getActiveSplineCurveKeeper();
					activeKeeper.shiftSplineCurve(dx, dy);
				}
			}
		} finally {
			keepersListLock_.unlock();
		}
	}

	// ----------------------------------------------------------------------------
	// KEYLISTENER METHODS

	/**
	 * Adds a <code>KeyListener</code> to a <code>Viewer</code> if it has not
	 * one already.
	 */
	private void addKeyListenerToViewer(Viewer viewer) {
		if (!hasKeyListenerTable_.containsKey(viewer)) {
			hasKeyListenerTable_.put(viewer, true);
			viewer.addKeyListener(this);
		}
	}

	// ----------------------------------------------------------------------------

	/** Removes a <code>KeyListener</code> from a <code>>Viewer</code>. */
	private void removeKeyListenerFromViewer(Viewer viewer) {
		viewer.removeKeyListener(this);
	}

	// ----------------------------------------------------------------------------

	/** Removes all instances of <code>KeyListener</code> from all images. */
	private void removeAllKeyListeners() {
		ArrayList<Sequence> sequences = getSequences();
		for (Sequence sequence : sequences) {
			ArrayList<Viewer> viewers = sequence.getViewers();
			for (Viewer viewer : viewers) {
				removeKeyListenerFromViewer(viewer);
			}
		}
	}

	// ============================================================================
	// LISTENER METHODS

	// ----------------------------------------------------------------------------
	// ACTION LISTENER METHODS

	@Override
	public void actionPerformed(ActionEvent e) {
		if (e.getSource() == createSplineCurveButton_) {
			addAndActivateSplineCurve();
		} else if (e.getSource() == deleteSplineCurveButton_) {
			removeActiveSplineCurve();
		}
	}

	// ----------------------------------------------------------------------------
	// ACTIVEVIEWERLISTENER METHODS

	@Override
	public void viewerActivated(Viewer viewer) {
		if (viewer != null) {
			addKeyListenerToViewer(viewer);
		}
	}

	// ----------------------------------------------------------------------------

	@Override
	public void viewerDeactivated(Viewer viewer) {
	}

	// ----------------------------------------------------------------------------

	@Override
	public void activeViewerChanged(ViewerEvent event) {
	}

	// ----------------------------------------------------------------------------
	// GLOBALVIEWERLISTENER METHODS

	@Override
	public void viewerOpened(Viewer viewer) {
	}

	// ----------------------------------------------------------------------------

	@Override
	public void viewerClosed(Viewer viewer) {
		if (viewer != null) {
			removeKeyListenerFromViewer(viewer);
		}
	}

	// ----------------------------------------------------------------------------
	// KEYLISTENER METHODS

	/** Interactions with respect to <code>KeyEvent</code>. */
	@Override
	public void keyPressed(KeyEvent e) {
		int key = e.getKeyCode();

		if (key == KeyEvent.VK_DELETE || key == KeyEvent.VK_BACK_SPACE) {
			removeActiveSplineCurve();
		} else if (key == KeyEvent.VK_LEFT) {
			translateActiveSplineCurve(-1, 0);
		} else if (key == KeyEvent.VK_RIGHT) {
			translateActiveSplineCurve(1, 0);
		} else if (key == KeyEvent.VK_UP) {
			translateActiveSplineCurve(0, -1);
		} else if (key == KeyEvent.VK_DOWN) {
			translateActiveSplineCurve(0, 1);
		}
	}

	// ----------------------------------------------------------------------------

	@Override
	public void keyReleased(KeyEvent e) {
	}

	// ----------------------------------------------------------------------------

	@Override
	public void keyTyped(KeyEvent e) {
	}

}
