/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake3d.snake;

import icy.util.XMLUtil;

import org.w3c.dom.Element;

/**
 * Class that wraps the parameters of the E-Snake.
 * 
 * @version May 3, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class SphereSnakeParameters {

	/**
	 * If <code>true</code> indicates that the snake will keep iterating till
	 * the optimizer decides so.
	 */
	private boolean immortal_ = true;

	/** Number of spline vector coefficients. */
	private int M_ = 0;

	/** Energy tradeoff factor. */
	private double alpha_ = 0;

	/** Surface stiffness. */
	private double gamma_ = 0;

	/** Maximum number of iterations allowed when the immortal is false. */
	private int maxLife_ = 0;

	/** Indicates the energy function of the snake. **/
	private SphereSnakeEnergyType energyType_;

	/** Indicates the type of features to detect (bright or dark), **/
	private SphereSnakeTargetType detectType_;

	/** Label of the XML tag containing the maximum number of iterations. */
	public static final String ID_MAX_LIFE = "maxLife";

	/**
	 * Label of the XML tag informing if the number of iterations of the snake
	 * is limited or not.
	 */
	public static final String ID_IMMORTAL = "immortal";

	/**
	 * Label of the XML tag containing the number of control points of the
	 * snake.
	 */
	public static final String ID_M = "M";

	/**
	 * Label of the XML tag containing the energy trade-off between the contour
	 * energy and the region energy.
	 */
	public static final String ID_ALPHA = "alpha";

	/**
	 * Label of the XML tag containing the surface stiffness.
	 */
	public static final String ID_GAMMA = "gamma";

	/**
	 * Label of the XML tag informing about the type of energy the snake uses
	 * (contour, region or mixture).
	 */
	public static final String ID_ENERGY_TYPE = "energyType";

	/**
	 * Label of the XML tag informing about the brightness of the target to
	 * segment (brighter or darker than the background).
	 */
	public static final String ID_DETECT_TYPE = "detectType";

	// ============================================================================
	// PUBLIC METHODS

	/** Default constructor. */
	public SphereSnakeParameters(int maxLife, int M, double alpha,
			double gamma, boolean immortal, SphereSnakeTargetType detectType,
			SphereSnakeEnergyType energyType) {
		setMaxLife(maxLife);
		setM(M);
		setAlpha(alpha);
		setGamma(gamma);
		setImmortal(immortal);
		setDetectType(detectType);
		setEnergyType(energyType);
	}

	// ----------------------------------------------------------------------------

	@Override
	public String toString() {
		return new String("[E-Snake parameters: immortal = " + immortal_
				+ ", M = " + M_ + ", alpha = " + alpha_ + ", gamma = " + gamma_
				+ ", maximum number of iterations = " + maxLife_
				+ ", energy type = " + energyType_ + ", detect type = "
				+ detectType_ + "]");
	}

	// ----------------------------------------------------------------------------

	public void saveToXML(Element node) {
		XMLUtil.setElementDoubleValue(node, SphereSnakeParameters.ID_MAX_LIFE,
				maxLife_);
		XMLUtil.setElementIntValue(node, SphereSnakeParameters.ID_M, M_);
		XMLUtil.setElementDoubleValue(node, SphereSnakeParameters.ID_ALPHA,
				alpha_);
		XMLUtil.setElementDoubleValue(node, SphereSnakeParameters.ID_GAMMA,
				gamma_);
		XMLUtil.setElementBooleanValue(node, SphereSnakeParameters.ID_IMMORTAL,
				immortal_);
		switch (energyType_) {
		case CONTOUR:
			XMLUtil.setElementValue(node, SphereSnakeParameters.ID_ENERGY_TYPE,
					"CONTOUR");
			break;
		case REGION:
			XMLUtil.setElementValue(node, SphereSnakeParameters.ID_ENERGY_TYPE,
					"REGION");
			break;
		case MIXTURE:
			XMLUtil.setElementValue(node, SphereSnakeParameters.ID_ENERGY_TYPE,
					"MIXTURE");
			break;
		}
		switch (detectType_) {
		case BRIGHT:
			XMLUtil.setElementValue(node, SphereSnakeParameters.ID_DETECT_TYPE,
					"BRIGHT");
			break;
		case DARK:
			XMLUtil.setElementValue(node, SphereSnakeParameters.ID_DETECT_TYPE,
					"DARK");
			break;
		}
	}

	// ============================================================================
	// GETTERS AND SETTERS

	public double getAlpha() {
		return alpha_;
	}

	// ----------------------------------------------------------------------------

	public void setAlpha(double alpha) {
		alpha_ = alpha;
	}

	// ----------------------------------------------------------------------------

	public double getGamma() {
		return gamma_;
	}

	// ----------------------------------------------------------------------------

	public void setGamma(double gamma) {
		gamma_ = gamma;
	}

	// ----------------------------------------------------------------------------

	public boolean isImmortal() {
		return immortal_;
	}

	// ----------------------------------------------------------------------------

	public void setImmortal(boolean immortal) {
		immortal_ = immortal;
	}

	// ----------------------------------------------------------------------------

	public int getM() {
		return M_;
	}

	// ----------------------------------------------------------------------------

	public void setM(int M) {
		M_ = M;
	}

	// ----------------------------------------------------------------------------

	public SphereSnakeTargetType getDetectType() {
		return detectType_;
	}

	// ----------------------------------------------------------------------------

	public void setDetectType(SphereSnakeTargetType detectType) {
		detectType_ = detectType;
	}

	// ----------------------------------------------------------------------------

	public SphereSnakeEnergyType getEnergyType() {
		return energyType_;
	}

	// ----------------------------------------------------------------------------

	public void setEnergyType(SphereSnakeEnergyType energyType) {
		energyType_ = energyType;
	}

	// ----------------------------------------------------------------------------

	public int getMaxLife() {
		return maxLife_;
	}

	// ----------------------------------------------------------------------------

	public void setMaxLife(int maxLife) {
		maxLife_ = maxLife;
	}
}
