/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 *     Daniel Schmitter (daniel.schmitter@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.shape.priorshapes;

import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.process.process1D.BSplineBasis;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.PriorShape;
import Jama.Matrix;

/**
 * Factory class that constructs shape projectors.
 * 
 * @version October 27, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Daniel Schmitter (daniel.schmitter@epfl.ch)
 */
public class ShapeProjectorBuilder {

	/** Number of spline vector coefficients. */
	private int M_ = 0;

	// ============================================================================
	// PUBLIC METHODS

	/** Constructor. */
	public ShapeProjectorBuilder(int M) {
		M_ = M;
	}

	// ----------------------------------------------------------------------------

	/** Create an affine (+ translation) shape-projector for a particular shape-space. */
	public ShapeProjector buildShapeProjectorAffine(PriorShape shape) {
		Snake2DNode[] shapeNodes = shape.getNodes(M_);
		double[][] VcArray = new double[3][M_];
		for (int i = 0; i < M_; i++) {
			VcArray[0][i] = shapeNodes[i].x;
			VcArray[1][i] = shapeNodes[i].y;
			VcArray[2][i] = 1;
		}
		Matrix Vc = new Matrix(VcArray);
		Matrix VcT = Vc.transpose();
		Matrix projector_ = VcT.times(((Vc.times(VcT)).inverse()).times(Vc));
		Matrix orthoProjector_ = Matrix.identity(projector_.getRowDimension(),
				projector_.getColumnDimension()).minus(projector_);
		return new ShapeProjector(shape, projector_, orthoProjector_);
	}
	
	/** Create a similarity (+ translation) shape-projector for a particular shape-space. */
	public ShapeProjector buildShapeProjectorSimilarity(PriorShape shape) {
		
		Snake2DNode[] shapeNodes = shape.getNodes(M_);
		double[][] VcArray = new double[4][2 * M_];
		for (int i = 0; i < M_; i++) {
			VcArray[0][i] = shapeNodes[i].x;
			VcArray[0][i + M_] = shapeNodes[i].y;
			
			VcArray[1][i] = -shapeNodes[i].y;
			VcArray[1][i + M_] = shapeNodes[i].x;
			
			VcArray[2][i] = 1;
			VcArray[2][i + M_] = 0;
			
			VcArray[3][i] = 0;
			VcArray[3][i + M_] = 1;
		}
		
		// Autocorrelation matrix
		double[][] AMatArray = new double[M_][M_];
		for (int i = 0; i < M_; i++) {
			for (int j=0; j < M_; j++){
				AMatArray[i][j] = BSplineBasis.autocorrelationESpline3(i-j, M_);
			}
		}
		
		// Augmented autocorrelation matrix
		double[][] BMatArray = new double[2 * M_][2 * M_];
		for (int i = 0; i < M_; i++) {
			for (int j=0; j < M_; j++){
				BMatArray[i][j] = AMatArray[i][j];
				BMatArray[M_ + i][M_ + j] = AMatArray[i][j];
			}
		}
		
		Matrix VcT = new Matrix(VcArray);
		Matrix Vc = VcT.transpose();
		
		Matrix B = new Matrix(BMatArray);			
		
		Matrix G = VcT.times(B.times(Vc));		 
		Matrix projector_ = Vc.times(G.inverse().times(VcT.times(B)));
		
		return new ShapeProjector(shape, projector_, null, B);
	}
}
