package plugins.tprovoost.Microscopy.MicroscopeLive;

import icy.canvas.IcyCanvas;
import icy.gui.dialog.ConfirmDialog;
import icy.gui.dialog.MessageDialog;
import icy.painter.Overlay;
import icy.preferences.IcyPreferences;
import icy.preferences.XMLPreferences;
import icy.sequence.Sequence;

import java.awt.event.KeyEvent;
import java.awt.event.MouseEvent;
import java.awt.geom.Point2D;

import plugins.tprovoost.Microscopy.MicroManager.MicroManager;
import plugins.tprovoost.Microscopy.MicroManager.tools.StageMover;

public class LiveOverlay extends Overlay
{
    Point2D lastImagePoint;
    XMLPreferences prefs = IcyPreferences.root().node("LivePainter");
    boolean considerPxSizeCfg;

    public LiveOverlay(Sequence s)
    {
        super("Live UI");

        considerPxSizeCfg = prefs.getBoolean("considerPxSizeCfg", true);
        StageMover.setInvertX(prefs.getBoolean("invertX", false));
        StageMover.setInvertY(prefs.getBoolean("invertY", false));

        for (Overlay pa : s.getOverlays())
        {
            if (pa instanceof PainterInfoConfig)
            {
                ((PainterInfoConfig) pa).putData("helpDrag", "Alt + drag: move the stage");
                ((PainterInfoConfig) pa).putData("helpPress", "Ctrl + Shift + click: move to point");
                setInvertX((PainterInfoConfig) pa);
                setInvertY((PainterInfoConfig) pa);
            }
        }
    }

    private void setInvertX(PainterInfoConfig pif)
    {
        if (StageMover.isInvertX())
            pif.putData("helpInvertX", "x: inverted (Alt + X to change)");
        else
            pif.putData("helpInvertX", "x: normal (Alt + X to change)");
    }

    private void setInvertY(PainterInfoConfig pif)
    {
        if (StageMover.isInvertY())
            pif.putData("helpInvertY", "y: inverted (Alt + Y to change)");
        else
            pif.putData("helpInvertY", "y: normal (Alt + Y to change)");
    }

    @Override
    public void keyPressed(KeyEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
        if (e.isConsumed())
            return;

        if (e.isAltDown())
        {
            switch (e.getKeyCode())
            {
                case KeyEvent.VK_X:
                    StageMover.setInvertX(StageMover.isInvertX());
                    for (Overlay pa : canvas.getSequence().getOverlays())
                        if (pa instanceof PainterInfoConfig)
                            setInvertX((PainterInfoConfig) pa);
                    e.consume();
                    break;

                case KeyEvent.VK_Y:
                    StageMover.setInvertY(StageMover.isInvertY());
                    for (Overlay pa : canvas.getSequence().getOverlays())
                        if (pa instanceof PainterInfoConfig)
                            setInvertY((PainterInfoConfig) pa);
                    e.consume();
                    break;
            }
        }
    }

    @Override
    public void mousePressed(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
        if (e.isConsumed())
            return;

        if (e.isAltDown())
        {
            lastImagePoint = imagePoint;
            e.consume();
        }
        else if (e.isShiftDown() && e.isControlDown())
        {
            if (MicroManager.getCore().getAvailablePixelSizeConfigs().size() > 0)
            {
                try
                {
                    StageMover.moveToPoint(canvas.getSequence(), imagePoint.getX(), imagePoint.getY());
                    e.consume();
                }
                catch (Exception e1)
                {
                    e1.printStackTrace();
                }
            }
            else
                MessageDialog
                        .showDialog(
                                "PixelSize configuration",
                                "No pixel size configuration has been found. The pixel size "
                                        + "represents the number of pixels per m, and is often used for high precision when moving the stage. <br/>"
                                        + "It is highly advised to run the Calibrator Manager to get the best value.");
        }
    }

    @Override
    public void mouseReleased(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
        lastImagePoint = null;
    }

    @Override
    public void mouseDrag(MouseEvent e, Point2D imagePoint, IcyCanvas canvas)
    {
        if (e.isConsumed())
            return;

        if (e.isAltDown())
        {
            boolean pxSizeConfigured;

            pxSizeConfigured = MicroManager.getCore().getAvailablePixelSizeConfigs().size() > 0;

            if (!considerPxSizeCfg || (considerPxSizeCfg && pxSizeConfigured))
            {
                if (lastImagePoint != null)
                {
                    Point2D vect;
                    double pxsize;

                    // movement
                    vect = new Point2D.Double(imagePoint.getX() - lastImagePoint.getX(), imagePoint.getY()
                            - lastImagePoint.getY());

                    if (pxSizeConfigured)
                        pxsize = MicroManager.getCore().getPixelSizeUm();
                    else
                        pxsize = 1;

                    try
                    {
                        StageMover.moveXYRelative(vect.getX() * pxsize, vect.getY() * pxsize);
                        e.consume();
                    }
                    catch (Exception e1)
                    {
                        e1.printStackTrace();
                    }
                }
            }
            else
            {
                if (ConfirmDialog
                        .confirm(
                                "PixelSize configuration",
                                "No pixel size configuration was found. The pixel size "
                                        + "represents the number of pixels per m, and is often used for high precision when moving the stage. <br/>"
                                        + "It is highly advised to run the Calibrator Manager to get the best value and set it in Micro-Manager For Icy before continuing."
                                        + "<br/>However, you can use the drag without a pixel size configuration, but the speed of the stage will "
                                        + "not be according to neither your current objective nor your camera.<br/> "
                                        + "Do you want to use the drag without the pixel size configuration ?"))
                {
                    considerPxSizeCfg = true;
                    if (ConfirmDialog.confirm("Save",
                            "Do you want to always be able to use the drag option, without being noticed ?"))
                        prefs.putBoolean("considerPxSizeCfg", considerPxSizeCfg);
                }
            }
        }
    }
}
