/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 *     Daniel Schmitter (daniel.schmitter@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnake.snake;

import icy.util.XMLUtil;

import java.awt.Color;
import java.awt.Polygon;
import java.awt.Rectangle;
import java.awt.geom.Point2D;

import org.w3c.dom.Element;

import plugins.big.bigsnake.core.ImageLUTContainer;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2D;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.bigsnakeutils.icy.snake2D.Snake2DScale;
import plugins.big.bigsnakeutils.process.process1D.BSplineBasis;
import plugins.big.bigsnakeutils.process.process1D.BSplineBasis.BSplineBasisType;
import plugins.big.bigsnakeutils.process.process1D.Filters;
import plugins.big.bigsnakeutils.shape.priorshapes.ShapeProjector;
import plugins.big.bigsnakeutils.shape.priorshapes.ShapeProjectorBuilder;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Banana;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Boomerang;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.BrainWhite;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.BuddingYeast;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Circle;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.CorpusCallosum;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Cross;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Custom;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.ElectricGuitar;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Femur;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Fly;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.MouseOrgan;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.PriorShape;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Rod;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Semicircle;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.Square;
import plugins.big.bigsnakeutils.shape.priorshapes.shapes.UpperLip;
import plugins.big.bigsnakeutils.shape.utils.Geometry2D;
import plugins.kernel.roi.roi2d.ROI2DEllipse;
import plugins.kernel.roi.roi2d.ROI2DPolygon;
import plugins.kernel.roi.roi2d.ROI2DRectangle;

/**
 * Exponential spline snake (E-Snake).
 * 
 * @version November 15, 2014
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Ramtin Madani (ramtin_madani@yahoo.com)
 * @author Masih Nilchian (masih_n85@yahoo.com)
 * @author Daniel Schmitter (daniel.schmitter@epfl.ch)
 */
public class ESnake implements Snake2D {

	/** Snake defining nodes. */
	private Snake2DNode[] coef_ = null;

	// ----------------------------------------------------------------------------
	// SNAKE CONTOUR

	/** Samples of the x coordinates of the snake contour. */
	private double[] xSnakeContour_ = null;
	/** Samples of the y coordinates of the snake contour. */
	private double[] ySnakeContour_ = null;

	/** Samples of the x coordinates of the derivative of the snake contour. */
	private double[] xSnakeTangentVector_ = null;
	/** Samples of the y coordinates of the derivative of the snake contour. */
	private double[] ySnakeTangentVector_ = null;

	/** Bounding box that encloses the outline of the snake. */
	private Rectangle snakeContourBoundingBox_ = null;

	/**
	 * Samples of the x coordinates of the shell that surrounds the snake
	 * contour.
	 */
	private double[] xSnakeShellContour_ = null;
	/**
	 * Samples of the y coordinates of the shell that surrounds the snake
	 * contour.
	 */
	private double[] ySnakeShellContour_ = null;

	/** Bounding box that encloses the outline of the shell of the snake. */
	private Rectangle snakeShellBoundingBox_ = null;

	/** Number of samples at which each segment of the contour is discretized. */
	private final int nSamplesPerSegment_ = 50;

	// ----------------------------------------------------------------------------
	// SNAKE STATUS FIELDS

	/**
	 * If <code>true</code> indicates that the snake is able to keep being
	 * optimized.
	 */
	private boolean alive_ = true;
	/** Signed area of the region enclosed by the snake. */
	private double area_ = 0;
	/**
	 * If <code>true</code>, the snake has gone through the initialization
	 * process.
	 */
	protected boolean isInitialized_ = false;
	/**
	 * Number of iterations left when <code>immortal_</code> is
	 * <code>false</code>.
	 */
	private int life_ = 0;
	/**
	 * If <code>false</code> indicates that the snake takes an ill conditioned
	 * shape.
	 */
	private boolean valid_ = true;

	// ----------------------------------------------------------------------------
	// SNAKE OPTION FIELDS

	/** Indicates the type of features to detect (bright or dark). **/
	private ESnakeTargetType detectType_ = null;
	/** Number of spline vector coefficients. */
	private int M_ = 0;
	/** Snake energy used during the optimization process. */
	private ESnakeEnergyType energyType_ = null;
	/** Energy trade-off factor. */
	private double alpha_ = 0;
	/** Prior-shape information. */
	private ESnakePriorShapeType priorShapeType_ = null;
	/** Shape space information. */
	private ShapeSpaceType shapeSpaceType_ = null;
	/** Weight contribution of the prior-shape energy to the total energy. */
	private double beta_ = 0;
	/**
	 * Maximum number of iterations allowed when the <code>immortal_</code> is
	 * <code>false</code>.
	 */
	private int maxLife_ = 0;
	/**
	 * If <code>true</code> indicates that the snake will keep iterating till
	 * the optimizer decides so.
	 */
	private boolean immortal_ = true;

	/** Initial contour. */
	private Polygon initialContour_ = null;

	// ----------------------------------------------------------------------------
	// SPLINE LUTS

	/** Basis function of the curve. */
	private static final BSplineBasis.BSplineBasisType BASIS_FUNCTION = BSplineBasisType.ESPLINE3;
	/** LUT with the samples of the B-spline basis function. */
	private double[] bSplineLUT_ = null;
	/** LUT with the samples of the derivative of the basis function. */
	private double[] bSplineDerivativeLUT_ = null;
	/** LUT with the samples of the autocorrelation function. */
	private double[] bSplineAutocorrelationLUT_ = null;
	/**
	 * LUT with the values of
	 * $Q(i,j)=\int_{0}^{M}\,\phi_M(t-j)\,\phi'_M(t-i)\,\mathrm{d}t$.
	 */
	private double[][] qLUT_ = null;

	// ----------------------------------------------------------------------------
	// IMAGE FIELDS

	/** Container of the LUTs. */
	private ImageLUTContainer imageLUTs_ = null;

	// ----------------------------------------------------------------------------
	// SELF-INTERSECTION FIELDS

	/** The intersected node in the polygon. */
	private int badNodeIndex_ = 0;
	/** The intersected edge of the polygon. */
	private int badEdgeIndex_ = 0;
	/** Describing the self intersection of the snake polygon. */
	private boolean controlPolygonSelfIntersects_ = false;
	/** Condition of the snake polygon edges with respect to each other. */
	private int[][] edgeVsEdge_ = null;

	// ----------------------------------------------------------------------------
	// PRIOR-SHAPE FIELDS

	/** Encodes the shape-prior that the snake uses within its energy. */
	private PriorShape priorShape_ = null;
	/** Shape-projectors factory. */
	private ShapeProjectorBuilder shapeProjectorBuilder_ = null;
	/** Projector that projects a particular spline curve to a shape space. */
	private ShapeProjector shapeProjectorAffine_ = null;
	private ShapeProjector shapeProjectorSimilarity_ = null;
	private Custom customPriorShape_ = null;

	// ----------------------------------------------------------------------------
	// DISPLAY

	/** Color of the scale that represents the contour of the snake. */
	private final Color SNAKE_CONTOUR_COLOR = Color.RED;
	/** Color of the scale that represents the control polygon of the snake. */
	private final Color CONTROL_POLYGON_COLOR = new Color(255, 255, 0, 128);

	// ----------------------------------------------------------------------------
	// AUXILIARY FIELDS AND LUTS

	/** PI/M. */
	private double PIM_ = 0;
	/** 2*PI/M. */
	private double PI2M_ = 0;
	/**
	 * Support of the basis function multiplied by
	 * <code>nSamplesPerSegment_</code>.
	 */
	private int NR_ = 0;
	/** Number of control points multiplied by <code>nSamplesPerSegment_</code>. */
	private int MR_ = 0;

	/** LUT with samples of one period of a sine. */
	private double[] sinLUT_ = null;
	/** LUT with samples of one period of a cosine. */
	private double[] cosLUT_ = null;

	// ----------------------------------------------------------------------------
	// CONSTANTS

	/** Square root of two. */
	private static final double SQRT2 = Math.sqrt(2.0);

	// ----------------------------------------------------------------------------
	// XML TAGS

	/**
	 * Label of the XML tag containing the list of snake-defining control
	 * points.
	 */
	public static final String ID_CONTROL_POINTS = "control_points";
	/**
	 * Label of the XML tag containing the a single snake-defining control
	 * point.
	 */
	public static final String ID_CONTROL_POINT = "control_point";

	// ============================================================================
	// PUBLIC METHODS

	/** Constructor. */
	public ESnake(ImageLUTContainer imageLUTs, ESnakeParameters parameters,
			Object initialContour, Custom customPriorShape) {

		setCustomPriorShape(customPriorShape);

		if (imageLUTs == null) {
			System.err.println("Image not properly loaded.");
			return;
		}
		imageLUTs_ = imageLUTs;
		if (parameters == null) {
			System.err.println("E-Snake parameters not properly loaded.");
			return;
		}
		M_ = parameters.getM();
		if (M_ < 3) {
			System.err
			.println("The minimum number of knots for this basis function is three.");
			return;
		}

		shapeProjectorBuilder_ = new ShapeProjectorBuilder(M_);

		if (initialContour instanceof Polygon) {
			initialContour_ = (Polygon) initialContour;
		} else if (initialContour instanceof ROI2DPolygon) {
			initialContour_ = ((ROI2DPolygon) initialContour).getPolygon();
		} else if (initialContour instanceof ROI2DEllipse) {
			double x = ((ROI2DEllipse) initialContour).getEllipse()
					.getCenterX();
			double y = ((ROI2DEllipse) initialContour).getEllipse()
					.getCenterY();
			double h = ((ROI2DEllipse) initialContour).getEllipse().getHeight();
			double w = ((ROI2DEllipse) initialContour).getEllipse().getWidth();

			Polygon p = new Polygon();
			for (int i = 0; i < 100; i++) {
				p.addPoint((int) (x + w * Math.cos(2.0 * Math.PI * i / 100.0)
						/ 2.0),
						(int) (y + h * Math.sin(2.0 * Math.PI * i / 100.0)
								/ 2.0));
			}
			initialContour_ = p;
		} else if (initialContour instanceof ROI2DRectangle) {
			Polygon p = new Polygon();
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMinX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMinY());
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMinX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMaxY());
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMaxX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMaxY());
			p.addPoint((int) ((ROI2DRectangle) initialContour).getBounds()
					.getMaxX(), (int) ((ROI2DRectangle) initialContour)
					.getBounds().getMinY());
		} else {
			initialContour_ = null;
		}

		maxLife_ = parameters.getMaxLife();
		alpha_ = parameters.getAlpha();
		beta_ = parameters.getBeta();
		immortal_ = parameters.isImmortal();
		energyType_ = parameters.getEnergyType();
		detectType_ = parameters.getDetectType();
		shapeSpaceType_ = parameters.getShapeSpaceType();
		setPriorShape(parameters.getPriorShapeType());
		initialize(true);
	}

	// ----------------------------------------------------------------------------
	// SNAKE2D METHODS

	/**
	 * The purpose of this method is to compute the energy of the snake. This
	 * energy is usually made of three additive terms: 1) the image energy,
	 * which gives the driving force associated to the data; 2) the internal
	 * energy, which favors smoothness of the snake; and 3) the constraint
	 * energy, which incorporates a priori knowledge. This method is called
	 * repeatedly during the optimization of the snake, but only as long as the
	 * method <code>isAlive()</code> returns <code>true</code>. It is imperative
	 * that this function be everywhere differentiable with respect to the
	 * snake-defining nodes.
	 */
	@Override
	public double energy() {
		if (!immortal_) {
			life_--;
			if (life_ == 0) {
				alive_ = false;
			}
		}

		if (!isSnakeInBoundingBox()) {
			return Double.MAX_VALUE;
		}

		double energyRegularization = Math.pow(computeSelfIntersectionEnergy(),
				6);

		double E = 0;
		switch (energyType_) {
		case CONTOUR:
			E = computeContourEnergy();
			break;
		case REGION:
			E = computeRegionEnergy();
			break;
		case MIXTURE:
			E = alpha_ * computeContourEnergy() + (1 - alpha_)
			* computeRegionEnergy();
			break;
		}

		double priorShape = 0;
		if (!(priorShapeType_ == null || priorShapeType_ == ESnakePriorShapeType.NONE)) {
			priorShape = beta_ * computePriorShapeEnergy();
		}

		if (detectType_ == ESnakeTargetType.DARK) {
			return -E + energyRegularization + priorShape;
		} else if (detectType_ == ESnakeTargetType.BRIGHT) {
			return E + energyRegularization + priorShape;
		} else {
			return Double.MAX_VALUE;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a point with the position of the center of gravity of the scales.
	 */
	@Override
	public Point2D.Double getCentroid() {
		Point2D.Double centroid = new Point2D.Double();
		for (Snake2DNode element : coef_) {
			centroid.x += element.x;
			centroid.y += element.y;
		}
		centroid.x /= coef_.length;
		centroid.y /= coef_.length;
		return centroid;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to compute the gradient of the snake energy
	 * with respect to the snake-defining nodes.
	 */
	@Override
	public Point2D.Double[] getEnergyGradient() {
		return null;
	}

	// ----------------------------------------------------------------------------

	/** This method provides an accessor to the snake-defining nodes. */
	@Override
	public Snake2DNode[] getNodes() {
		return coef_;
	}

	// ----------------------------------------------------------------------------

	/** This method returns the quantity of snake-defining nodes. */
	@Override
	public int getNumNodes() {
		return coef_.length;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to determine what to draw on screen, given
	 * the current configuration of nodes. Collectively, the array of scales
	 * forms the skin of the snake.
	 */
	@Override
	public Snake2DScale[] getScales() {
		int numScales = getNumScales();
		Snake2DScale[] skin = new Snake2DScale[numScales];

		double[] xpoints = new double[M_];
		double[] ypoints = new double[M_];
		for (int k = 0; k < M_; k++) {
			xpoints[k] = coef_[k].x;
			ypoints[k] = coef_[k].y;
		}
		skin[0] = new Snake2DScale(xSnakeContour_, ySnakeContour_, MR_,
				SNAKE_CONTOUR_COLOR, true);

		skin[1] = new Snake2DScale(xpoints, ypoints, M_, CONTROL_POLYGON_COLOR,
				true);

		return skin;
	}

	// ----------------------------------------------------------------------------

	@Override
	public void initialize(boolean initShape) {
		// TODO: Complete with remaining basis functions
		switch (BASIS_FUNCTION) {
		case ESPLINE3:
			NR_ = BSplineBasis.ESPLINE3SUPPORT * nSamplesPerSegment_;
			break;
		case ESPLINE4:
			NR_ = BSplineBasis.ESPLINE4SUPPORT * nSamplesPerSegment_;
			break;
		case LINEARBSPLINE:
			NR_ = BSplineBasis.LINEARBSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case QUADRATICBSPLINE:
			NR_ = BSplineBasis.QUADRATICBSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case CUBICBSPLINE:
			NR_ = BSplineBasis.CUBICBSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case MSPLINE:
			NR_ = BSplineBasis.MSPLINESUPPORT * nSamplesPerSegment_;
			break;
		case ASPLINE:
			break;
		case CERCLE:
			break;
		default:
			break;
		}

		MR_ = M_ * nSamplesPerSegment_;
		PIM_ = Math.PI / M_;
		PI2M_ = 2 * PIM_;

		xSnakeContour_ = new double[MR_];
		ySnakeContour_ = new double[MR_];
		xSnakeTangentVector_ = new double[MR_];
		ySnakeTangentVector_ = new double[MR_];
		xSnakeShellContour_ = new double[MR_];
		ySnakeShellContour_ = new double[MR_];

		qLUT_ = new double[M_][M_];

		snakeContourBoundingBox_ = new Rectangle();
		snakeShellBoundingBox_ = new Rectangle();

		life_ = maxLife_;
		buildLUTs();
		if (initShape) {
			initializeDefaultShape();
		}
		updateSnakeSkin();
		updateArea();
		if (energyType_ == ESnakeEnergyType.REGION
				|| energyType_ == ESnakeEnergyType.MIXTURE) {
			updateSnakeShell();
		}
		isInitialized_ = true;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to monitor the status of the snake.
	 */
	@Override
	public boolean isAlive() {
		return alive_;
	}

	// ----------------------------------------------------------------------------

	/** Returns <code>true</code> if the snake has been initialized. */
	@Override
	public boolean isInitialized() {
		return isInitialized_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sets the status of the snake to alive, and restores the maximum number
	 * iterations to the original one.
	 */
	@Override
	public void reviveSnake() {
		alive_ = true;
		life_ = maxLife_;
	}

	// ----------------------------------------------------------------------------

	/** Saves the snake-defining parameters in an XML file. */
	@Override
	public void saveToXML(Element node) {
		getSnakeParameters().saveToXML(node);
		Element controlPointsElement = XMLUtil.addElement(node,
				ID_CONTROL_POINTS);
		for (Snake2DNode controlPoint : coef_) {
			Element controlPointElement = XMLUtil.addElement(
					controlPointsElement, ID_CONTROL_POINT);
			controlPoint.saveToXML(controlPointElement);
		}
	}

	// ----------------------------------------------------------------------------

	/** This method provides a mutator to the snake-defining nodes. */
	@Override
	public void setNodes(Snake2DNode[] node) {
		for (int i = 0; i < M_; i++) {
			coef_[i].x = node[i].x;
			coef_[i].y = node[i].y;
		}

		updateSnakeSkin();
		updateArea();
		if (energyType_ == ESnakeEnergyType.REGION
				|| energyType_ == ESnakeEnergyType.MIXTURE) {
			updateSnakeShell();
		}
		validateSnake();
	}

	// ----------------------------------------------------------------------------
	// ESNAKE METHODS

	/** Retrieves the area under the curve determined by the snake. */
	public double getArea() {
		return Math.abs(area_);
	}

	// ----------------------------------------------------------------------------

	/** Returns the bounding box of the snake curve. */
	public Rectangle getBounds() {
		return snakeContourBoundingBox_;
	}

	// ----------------------------------------------------------------------------

	/** Returns the custom shape prior (if any). */
	public Custom getCustomPriorShape() {
		return customPriorShape_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns the number of scales provided by the method
	 * <code>getScales()</code>.
	 */
	public int getNumScales() {
		return 2;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a new container with the information of the execution parameters
	 * of the snake.
	 */
	public ESnakeParameters getSnakeParameters() {
		return new ESnakeParameters(detectType_, M_, energyType_, alpha_,
				priorShapeType_, shapeSpaceType_, beta_, maxLife_, immortal_);
	}

	// ----------------------------------------------------------------------------

	/** Retrieves the orientation of the curve determined by the snake. */
	public ESnakeOrientation getOrientation() {
		if (area_ >= 0) {
			return ESnakeOrientation.CLOCKWISE;
		} else {
			return ESnakeOrientation.COUNTERCLOCKWISE;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns <code>false</code> if the snake reaches an invalid configuration.
	 * It can be related to ill-posed configuration of the points or related to
	 * the image boundaries.
	 */
	public boolean isValid() {
		return valid_;
	}

	// ----------------------------------------------------------------------------

	/** Sets a custom prior-shape to the snake. */
	public void setCustomPriorShape(Custom customPriorShape) {
		customPriorShape_ = customPriorShape;
	}

	// ----------------------------------------------------------------------------

	/** Sets the E-Snake execution parameters. */
	public void setSnakeParameters(ESnakeParameters parameters) {
		detectType_ = parameters.getDetectType();
		int M = parameters.getM();
		if (M != M_) {
			setNumNodes(1.5, M);
		}
		energyType_ = parameters.getEnergyType();
		alpha_ = parameters.getAlpha();
		ESnakePriorShapeType priorShapeType = parameters.getPriorShapeType();
		if (priorShapeType != priorShapeType_) {
			setPriorShape(priorShapeType);
		}
		ShapeSpaceType shapeSpaceType = parameters.getShapeSpaceType();
		if (shapeSpaceType != shapeSpaceType_) {
			shapeSpaceType_ = shapeSpaceType;
			// TODO: check wheather if-condition can be removed,
			// i.e. directly call shapeSpaceType_ =
			// parameters.getShapeSpaceType();
		}
		beta_ = parameters.getBeta();
		maxLife_ = parameters.getMaxLife();
		immortal_ = parameters.isImmortal();
		reviveSnake();
	}

	// ----------------------------------------------------------------------------
	// GEOMETRIC TRANSFORMATION METHODS

	/**
	 * Modifies the control points of a snake in order to scale the snake
	 * horizontally by a given factor.
	 */
	public void dilateX(double a) {
		Point2D.Double g = getCentroid();
		Snake2DNode[] coef = getNodes();
		for (int i = 0; i < getNumNodes(); i++) {
			coef[i].x = a * coef[i].x + (1 - a) * g.x;
		}
		setNodes(coef);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Modifies the control points of a snake in order to scale the snake
	 * vertically by a given factor.
	 */
	public void dilateY(double b) {
		Point2D.Double g = getCentroid();
		Snake2DNode[] coef = getNodes();
		for (int i = 0; i < getNumNodes(); i++) {
			coef[i].y = b * coef[i].y + (1 - b) * g.y;
		}
		setNodes(coef);
	}

	// ============================================================================
	// PRIVATE METHODS

	/** Initializes all look-up tables of the class. */
	private void buildLUTs() {
		double currentVal;
		bSplineLUT_ = new double[NR_];
		bSplineDerivativeLUT_ = new double[NR_];
		sinLUT_ = new double[MR_];
		cosLUT_ = new double[MR_];

		double PI2MR = PI2M_ / nSamplesPerSegment_;
		for (int i = 0; i < MR_; i++) {
			sinLUT_[i] = Math.sin(PI2MR * i + 1.5);
			cosLUT_[i] = Math.cos(PI2MR * i + 1.5);
		}

		// TODO: Complete with remaining basis functions
		for (int i = 0; i < NR_; i++) {
			currentVal = (double) i / (double) nSamplesPerSegment_;
			switch (BASIS_FUNCTION) {
			case ESPLINE3:
				bSplineLUT_[i] = BSplineBasis.ESpline3(currentVal, PI2M_);
				break;
			case ESPLINE4:
				bSplineLUT_[i] = BSplineBasis.ESpline4(currentVal, PI2M_);
				break;
			case LINEARBSPLINE:
				bSplineLUT_[i] = BSplineBasis.LinearBSpline(currentVal);
				break;
			case QUADRATICBSPLINE:
				bSplineLUT_[i] = BSplineBasis.QuadraticSpline(currentVal);
				break;
			case CUBICBSPLINE:
				bSplineLUT_[i] = BSplineBasis.CubicBSpline(currentVal);
				break;
			case MSPLINE:
				bSplineLUT_[i] = BSplineBasis.MSpline(currentVal, PI2M_);
				break;
			case ASPLINE:
				break;
			case CERCLE:
				break;
			default:
				break;
			}
			bSplineDerivativeLUT_[i] = BSplineBasis.ESpline3_Prime(currentVal,
					PI2M_);
		}

		int qSize = 2 * BSplineBasis.ESPLINE3SUPPORT - 1;
		bSplineAutocorrelationLUT_ = new double[qSize];
		for (int i = 0; i < qSize; i++) {
			bSplineAutocorrelationLUT_[i] = BSplineBasis.correlationESpline(i
					- BSplineBasis.ESPLINE3SUPPORT + 1, PI2M_);
		}

		for (int i = 0; i < M_; i++) {
			for (int j = 0; j < M_; j++) {
				qLUT_[i][j] = computeQ(i, j);
			}
		}
	}

	// ----------------------------------------------------------------------------

	/** Initializes the snake control points with a predefined shape. */
	private void initializeDefaultShape() {
		coef_ = new Snake2DNode[M_];
		if (initialContour_ != null) {
			if (initialContour_.npoints != 0) {
				Point2D.Double[] resampledContour = Geometry2D
						.arcLengthResampling(initialContour_, M_);
				coef_ = getSplineKnots(resampledContour, M_);
				return;
			}
		}

		coef_ = priorShape_.getNodes(M_);

		double xMin = Double.MAX_VALUE;
		double yMin = Double.MAX_VALUE;

		double xMax = -Double.MAX_VALUE;
		double yMax = -Double.MAX_VALUE;

		double xg = 0;
		double yg = 0;
		for (Snake2DNode element : coef_) {
			double x = element.x;
			double y = element.y;
			xg += x;
			yg += y;
			if (x > xMax) {
				xMax = x;
			}
			if (y > yMax) {
				yMax = y;
			}
			if (x < xMin) {
				xMin = x;
			}
			if (y < yMin) {
				yMin = y;
			}
		}
		xg /= M_;
		yg /= M_;

		double shapeWidth = xMax - xMin;
		double shapeHeight = yMax - yMin;

		double scale = 1.0 / Math.max(shapeWidth / imageLUTs_.getImageWidth(),
				shapeHeight / imageLUTs_.getImageHeight());

		for (int i = 0; i < coef_.length; i++) {
			coef_[i].x = 0.33 * scale * (coef_[i].x - xg)
					+ imageLUTs_.getImageWidth() / 2.0;
			coef_[i].y = 0.33 * scale * (coef_[i].y - yg)
					+ imageLUTs_.getImageHeight() / 2.0;
		}

	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes the points of the contour of the snake from the control points
	 * and stores it in a LUT.
	 */
	private void updateSnakeSkin() {
		int index;

		int xmin = imageLUTs_.getImageWidth() - 1;
		int xmax = 0;
		int ymin = imageLUTs_.getImageHeight() - 1;
		int ymax = 0;

		double aux, aux2, xPrimeVal, yPrimeVal, xPosVal, yPosVal;
		for (int i = 0; i < MR_; i++) {
			xPosVal = 0.0;
			yPosVal = 0.0;
			xPrimeVal = 0.0;
			yPrimeVal = 0.0;
			for (int k = 0; k < M_; k++) {
				index = i - k * nSamplesPerSegment_;

				while (index < 0) {
					index += MR_;
				}
				while (index >= MR_) {
					index -= MR_;
				}
				if (index >= NR_) {
					continue;
				} else {
					aux = bSplineLUT_[index];
					aux2 = bSplineDerivativeLUT_[index];
				}
				xPosVal += coef_[k].x * aux;
				yPosVal += coef_[k].y * aux;
				xPrimeVal += coef_[k].x * aux2;
				yPrimeVal += coef_[k].y * aux2;
			}
			xSnakeContour_[i] = xPosVal;
			ySnakeContour_[i] = yPosVal;

			// update the bounding box
			if ((int) Math.floor(xSnakeContour_[i]) < xmin) {
				xmin = (int) Math.floor(xSnakeContour_[i]);
			}
			if ((int) Math.ceil(xSnakeContour_[i]) > xmax) {
				xmax = (int) Math.ceil(xSnakeContour_[i]);
			}
			if ((int) Math.floor(ySnakeContour_[i]) < ymin) {
				ymin = (int) Math.floor(ySnakeContour_[i]);
			}
			if ((int) Math.ceil(ySnakeContour_[i]) > ymax) {
				ymax = (int) Math.ceil(ySnakeContour_[i]);
			}

			xSnakeTangentVector_[i] = xPrimeVal;
			ySnakeTangentVector_[i] = yPrimeVal;
		}

		snakeContourBoundingBox_.x = xmin;
		snakeContourBoundingBox_.y = ymin;
		snakeContourBoundingBox_.width = xmax - xmin + 1;
		snakeContourBoundingBox_.height = ymax - ymin + 1;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes the points of the snake shell used in the computation of the
	 * region energy and stores it in a LUT. The shell is obtained by an
	 * isotropic scaling of the snake contour over the center of gravity of the
	 * snake such that the shell encloses the two times more area than the
	 * snake.
	 */
	private void updateSnakeShell() {
		Point2D.Double snakeCentroid = getCentroid();

		int xmin = imageLUTs_.getImageWidth() - 1;
		int xmax = 0;
		int ymin = imageLUTs_.getImageHeight() - 1;
		int ymax = 0;

		for (int i = 0; i < MR_; i++) {
			xSnakeShellContour_[i] = SQRT2
					* (xSnakeContour_[i] - snakeCentroid.x) + snakeCentroid.x;
			ySnakeShellContour_[i] = SQRT2
					* (ySnakeContour_[i] - snakeCentroid.y) + snakeCentroid.y;
			if ((int) Math.floor(xSnakeShellContour_[i]) < xmin) {
				xmin = (int) Math.floor(xSnakeShellContour_[i]);
			}
			if ((int) Math.ceil(xSnakeShellContour_[i]) > xmax) {
				xmax = (int) Math.ceil(xSnakeShellContour_[i]);
			}
			if ((int) Math.floor(ySnakeShellContour_[i]) < ymin) {
				ymin = (int) Math.floor(ySnakeShellContour_[i]);
			}
			if ((int) Math.ceil(ySnakeShellContour_[i]) > ymax) {
				ymax = (int) Math.ceil(ySnakeShellContour_[i]);
			}
		}
		snakeShellBoundingBox_.x = xmin;
		snakeShellBoundingBox_.y = ymin;
		snakeShellBoundingBox_.width = xmax - xmin + 1;
		snakeShellBoundingBox_.height = ymax - ymin + 1;
	}

	// ----------------------------------------------------------------------------

	/**
	 * Modifies the value of the <code>valid</code> flag depending if the snake
	 * is in a ill-conditioned configuration or not.
	 */
	private void validateSnake() {
		controlPolygonSelfIntersects_ = false;
		if (!isSnakeInBoundingBox()) {
			valid_ = false;
		} else if (isSnakeIntersected()) {
			valid_ = false;
		} else {
			valid_ = true;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Computes the area of the curve defined by the snake, the enclosing
	 * ellipse and the area ratio.
	 */
	private void updateArea() {
		double area = 0.0;
		int l_p;
		for (int k = 0; k < M_; k++) {
			int kN = k + BSplineBasis.ESPLINE3SUPPORT;
			for (int l = k - BSplineBasis.ESPLINE3SUPPORT + 1; l < kN; l++) {
				l_p = l;
				while (l_p < 0) {
					l_p += M_;
				}

				while (l_p >= M_) {
					l_p -= M_;
				}

				area += coef_[k].y * coef_[l_p].x
						* bSplineAutocorrelationLUT_[kN - l - 1];
			}
		}
		area_ = area;
	}

	// ----------------------------------------------------------------------------
	// ENERGY METHODS

	/** Computes the contour energy. */
	private double computeContourEnergy() {
		double energy = 0.0;
		double fuyLap_val;
		int x1, x2, y1, y2;
		double DeltaX1, DeltaX2, DeltaY1;

		int sign = 0;
		switch (getOrientation()) {
		case CLOCKWISE:
			sign = 1;
			break;
		case COUNTERCLOCKWISE:
			sign = -1;
			break;
		}

		int imageWidth = imageLUTs_.getImageWidth();
		int imageHeight = imageLUTs_.getImageHeight();
		double[] filteredImageDataArray = imageLUTs_
				.getFilteredImageDataArray();
		double[] preintegratedFilteredImageDataArray = imageLUTs_
				.getPreintegratedFilteredImageDataArray();

		for (int i = 0; i < MR_; i++) {
			x1 = (int) Math.floor(xSnakeContour_[i] - 0.5);
			y1 = (int) Math.floor(ySnakeContour_[i] - 0.5);

			if (x1 < 1) {
				x1 = 1;
			} else if (x1 > imageWidth - 2) {
				x1 = imageWidth - 2;
			}

			if (y1 < 1) {
				y1 = 1;
			} else if (y1 > imageHeight - 2) {
				y1 = imageHeight - 2;
			}

			x2 = x1 + 1;
			y2 = y1 + 1;

			DeltaX1 = (xSnakeContour_[i] - 0.5) - x1;
			DeltaY1 = (ySnakeContour_[i] - 0.5) - y1;
			DeltaX2 = x2 - (xSnakeContour_[i] - 0.5);

			fuyLap_val = preintegratedFilteredImageDataArray[x1 + imageWidth
			                                                 * (y1 - 1)]
			                                                		 * DeltaX2
			                                                		 + preintegratedFilteredImageDataArray[x2 + imageWidth
			                                                		                                       * (y1 - 1)] * DeltaX1;
			fuyLap_val += 0.5 * ((filteredImageDataArray[x1 + imageWidth * y1]
					* DeltaX2 + filteredImageDataArray[x2 + imageWidth * y1]
							* DeltaX1) + (DeltaY1 * (((filteredImageDataArray[x1
							                                                  + imageWidth * y1]
							                                                		  * DeltaX2 + filteredImageDataArray[x2 + imageWidth * y1]
							                                                				  * DeltaX1) * (2 - DeltaY1)) + ((filteredImageDataArray[x1
							                                                				                                                         + imageWidth * y2]
							                                                				                                                        		 * DeltaX2 + filteredImageDataArray[x2 + imageWidth * y2]
							                                                				                                                        				 * DeltaX1) * DeltaY1))));

			energy += fuyLap_val * xSnakeTangentVector_[i];
		}
		energy = energy / nSamplesPerSegment_ * sign;
		return energy;
	}

	// ----------------------------------------------------------------------------

	/** Computes the prior shape energy. */
	private double computePriorShapeEnergy() {

		if (shapeSpaceType_ == ShapeSpaceType.AFFINE) {
			return shapeProjectorAffine_.orthoProjectionDistanceAffine(coef_);
		} else if (shapeSpaceType_ == ShapeSpaceType.SIMILARITY) {
			return shapeProjectorSimilarity_
					.orthoProjectionDistanceSimilarity(coef_);
		}
		return 0;
	}

	// ----------------------------------------------------------------------------

	/** Computes the region energy. */
	private double computeRegionEnergy() {
		double[] imageDataArray = imageLUTs_.getImageDataArray();
		double[] preintegratedImageDataArray = imageLUTs_
				.getPreintegratedImageDataArray();
		double internalContribution_ = computeLineIntegral(imageDataArray,
				preintegratedImageDataArray, xSnakeContour_, ySnakeContour_,
				xSnakeTangentVector_);
		double externalContribution_ = SQRT2
				* computeLineIntegral(imageDataArray,
						preintegratedImageDataArray, xSnakeShellContour_,
						ySnakeShellContour_, xSnakeTangentVector_);
		double regionEnergy = ((externalContribution_ - internalContribution_) - internalContribution_)
				/ area_;
		return regionEnergy;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is computing the distance of the intersected
	 * node in the snake polygon w.r.t corresponding edge.
	 */
	private double computeSelfIntersectionEnergy() {
		if (controlPolygonSelfIntersects_) {
			Point2D.Double p = coef_[badNodeIndex_];
			Point2D.Double v = coef_[badEdgeIndex_];
			Point2D.Double w = coef_[(badEdgeIndex_ + 1) % M_];

			double l2 = v.distanceSq(w);
			if (l2 == 0.0) {
				return p.distance(v);
			} else {
				double t = ((p.x - v.x) * (w.x - v.x) + (p.y - v.y)
						* (w.y - v.y))
						/ l2;
				if (t < 0.0) {
					return p.distance(v);
				} else if (t > 1.0) {
					return p.distance(w);
				} else {
					double projectionX = v.x + t * (w.x - v.x);
					double projectionY = v.y + t * (w.y - v.y); // Projection
					// falls
					return p.distance(projectionX, projectionY);
				}
			}
		} else {
			return 0;
		}
	}

	// ----------------------------------------------------------------------------

	/** Computes a line integral over a closed curve. */
	private double computeLineIntegral(double[] f, double[] fy, double[] x,
			double y[], double[] xp) {
		double fy_val;
		int x0, x1, y0, y1;
		double DeltaX1, DeltaX2, DeltaY1, DeltaY2;

		int imageWidth = imageLUTs_.getImageWidth();

		double lineIntegral = 0.0;
		for (int i = 0; i < MR_; i++) {
			x0 = (int) Math.floor(x[i] - 0.5);
			y0 = (int) Math.floor(y[i] - 0.5);

			x1 = x0 + 1;
			y1 = y0 + 1;

			DeltaX1 = (x[i] - 0.5) - x0;
			DeltaY1 = (y[i] - 0.5) - y0;
			DeltaX2 = x1 - (x[i] - 0.5);

			DeltaX1 = (x[i] - 0.5) - x0;
			DeltaY1 = (y[i] - 0.5) - y0;
			DeltaX2 = 1 - DeltaX1;
			DeltaY2 = 1 - DeltaY1;

			fy_val = DeltaX2
					* (fy[x0 + imageWidth * y0] - 0.5 * f[x0 + imageWidth * y0]
							* DeltaY2 * DeltaY2 + 0.5 * f[x0 + imageWidth * y1]
									* DeltaY1 * DeltaY1)
									+ DeltaX1
									* (fy[x1 + imageWidth * y0] - 0.5 * f[x1 + imageWidth * y0]
											* DeltaY2 * DeltaY2 + 0.5 * f[x1 + imageWidth * y1]
													* DeltaY1 * DeltaY1);
			lineIntegral -= fy_val * xp[i];
		}
		return lineIntegral / nSamplesPerSegment_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * $\int_{0}^{M}\,\varphi_{M}(t-k_2)\,\varphi'_{M}(t-k_1)\,\mathrm{d}t$.
	 */
	private double computeQ(int k1, int k2) {
		double val = 0;
		for (int i = 0; i < MR_; i++) {

			int index1 = i - k2 * nSamplesPerSegment_;

			while (index1 < 0) {
				index1 += MR_;
			}
			while (index1 >= MR_) {
				index1 -= MR_;
			}
			int index2 = i - k1 * nSamplesPerSegment_;

			while (index2 < 0) {
				index2 += MR_;
			}
			while (index2 >= MR_) {
				index2 -= MR_;
			}
			val += BSplineBasis.ESpline3((double) index1
					/ (double) nSamplesPerSegment_, PI2M_)
					* BSplineBasis.ESpline3_Prime((double) index2
							/ (double) nSamplesPerSegment_, PI2M_);
		}
		val /= nSamplesPerSegment_;
		return val;
	}

	// ----------------------------------------------------------------------------
	// SELF-INTERSECTION METHODS

	/**
	 * This method check if the snake has been intersected or not. It returns
	 * <code>false</code> when the snake is not bended.
	 */
	private boolean isSnakeIntersected() {
		boolean snakeIntersection = false;
		if (isPolygonIntersected(null)) {
			controlPolygonSelfIntersects_ = true;
			int count = 0;
			for (int i = 0; i < M_; i++) {
				if (!isPolygonIntersected(i)) {
					for (int[] element : edgeVsEdge_) {
						if (element[0] == i) {
							count++;
						}
						if (element[0] == presentPreviousNode(i, M_)) {
							count++;
						}
					}
				}
			}

			int[] IntersectedNode_ = new int[count];
			int[] IntersectedLine_ = new int[count];
			count = 0;
			for (int i = 0; i < M_; i++) {
				if (!isPolygonIntersected(i)) {
					for (int[] element : edgeVsEdge_) {
						if (element[0] == i) {
							IntersectedNode_[count] = i;
							IntersectedLine_[count] = element[1];
							count++;
						}
						if (element[0] == presentPreviousNode(i, M_)) {
							IntersectedNode_[count] = i;
							IntersectedLine_[count] = element[1];
							count++;
						}
					}
				}
			}
			for (int i = 0; i < count && !snakeIntersection; i++) {
				badNodeIndex_ = IntersectedNode_[i];
				badEdgeIndex_ = IntersectedLine_[i];
				double intersectedNodeCondvsLine = ((coef_[IntersectedNode_[i]].y - coef_[IntersectedLine_[i]].y)
						* (coef_[presentNextNode(IntersectedLine_[i], M_)].x - coef_[IntersectedLine_[i]].x) - (coef_[IntersectedNode_[i]].x - coef_[IntersectedLine_[i]].x)
						* (coef_[presentNextNode(IntersectedLine_[i], M_)].y - coef_[IntersectedLine_[i]].y));
				for (int k = 0; k < MR_; k++) {
					if (intersectedNodeCondvsLine
							* ((ySnakeContour_[k] - coef_[IntersectedLine_[i]].y)
									* (coef_[presentNextNode(
											IntersectedLine_[i], M_)].x - coef_[IntersectedLine_[i]].x) - (xSnakeContour_[k] - coef_[IntersectedLine_[i]].x)
											* (coef_[presentNextNode(
													IntersectedLine_[i], M_)].y - coef_[IntersectedLine_[i]].y)) > 0) {

						snakeIntersection = true;
					}
				}
			}
		}
		return snakeIntersection;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to present the next node in the circular
	 * structure.
	 */
	private int presentNextNode(int a, int nodeNumber) {
		int NextNode_ = 0;
		if (a < nodeNumber - 1) {
			NextNode_ = a + 1;
		} else {
			NextNode_ = 0;
		}
		return NextNode_;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to present the previous node in the
	 * circular structure.
	 */
	private int presentPreviousNode(int a, int nodeNumber) {
		int nextNode = 0;
		if (a > 0) {
			nextNode = a - 1;
		} else {
			nextNode = nodeNumber - 1;
		}
		return nextNode;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to check that there is any intersection or
	 * not in the polygon. It returns <code>false</code> whenever there is no
	 * line intersection in the polygon.
	 */
	private boolean isPolygonIntersected(Integer node) {
		double[][] vertices = removeNodePolygon(node);
		int[][] edgevsNode = new int[vertices.length][vertices.length];
		/* Compute the edgevsNode array */
		for (int i = 0; i < vertices.length; i++) {
			for (int j = 0; j < vertices.length; j++) {
				if ((vertices[i][1] - vertices[j][1])
						* (vertices[presentNextNode(j, vertices.length)][0] - vertices[j][0]) > (vertices[i][0] - vertices[j][0])
						* (vertices[presentNextNode(j, vertices.length)][1] - vertices[j][1])) {
					edgevsNode[j][i] = 1;
				} else if ((vertices[i][1] - vertices[j][1])
						* (vertices[presentNextNode(j, vertices.length)][0] - vertices[j][0]) < (vertices[i][0] - vertices[j][0])
						* (vertices[presentNextNode(j, vertices.length)][1] - vertices[j][1])) {
					edgevsNode[j][i] = -1;
				} else {
					edgevsNode[j][i] = 0;
				}
			}
		}

		/*
		 * 2D array which is applied to find the condition of the edges w .r.t.
		 * each other
		 */
		int[][] edgeCond = new int[vertices.length][vertices.length];
		for (int i = 0; i < vertices.length; i++) {
			for (int j = 0; j < vertices.length; j++) {
				edgeCond[i][j] = edgevsNode[i][j]
						* edgevsNode[i][presentNextNode(j, vertices.length)];
			}
		}

		int count = 0;
		for (int i = 0; i < vertices.length; i++) {
			for (int j = 0; j < vertices.length; j++) {
				if (edgeCond[i][j] == -1 && edgeCond[j][i] == -1) {
					count++;
				}
			}
		}
		int[][] edgevsEdge = new int[count][2];
		if (node == null) {
			edgeVsEdge_ = new int[count][2];
		}

		count = 0;
		for (int i = 0; i < vertices.length; i++) {
			for (int j = 0; j < vertices.length; j++) {
				if (edgeCond[i][j] == -1 && edgeCond[j][i] == -1) {
					edgevsEdge[count][0] = i;
					edgevsEdge[count][1] = j;
					if (node == null) {
						edgeVsEdge_[count][0] = i;
						edgeVsEdge_[count][1] = j;
					}
					count++;
				}
			}
		}
		return count != 0;
	}

	// ----------------------------------------------------------------------------

	/**
	 * The purpose of this method is to remove the specific node of the polygon
	 * and return the others as a new polygon
	 */
	private double[][] removeNodePolygon(Integer Indx) {
		double[][] outputPolygon;
		if (Indx == null) {
			outputPolygon = new double[M_][2];
			for (int i = 0; i < M_; i++) {
				outputPolygon[i][0] = coef_[i].x;
				outputPolygon[i][1] = coef_[i].y;
			}
		} else {
			outputPolygon = new double[M_ - 1][2];
			int count = 0;
			int NextNode_ = Indx;
			while (count < M_ - 1) {
				NextNode_ = presentNextNode(NextNode_, M_);
				outputPolygon[count][0] = coef_[NextNode_].x;
				outputPolygon[count][1] = coef_[NextNode_].y;
				count++;
			}
		}
		return outputPolygon;
	}

	// ----------------------------------------------------------------------------

	/**
	 * This method check if the snake is in the bounding box or not. It returns
	 * <code>false</code> when the snake touches the boundaries.
	 */
	private boolean isSnakeInBoundingBox() {
		if (snakeContourBoundingBox_.x < 1) {
			return false;
		} else if (snakeContourBoundingBox_.x + snakeContourBoundingBox_.width > imageLUTs_
				.getImageWidth() - 1) {
			return false;
		} else if (snakeContourBoundingBox_.y < 1) {
			return false;
		} else if (snakeContourBoundingBox_.y + snakeContourBoundingBox_.height > imageLUTs_
				.getImageHeight() - 1) {
			return false;
		} else if (energyType_ == ESnakeEnergyType.REGION
				|| energyType_ == ESnakeEnergyType.MIXTURE) {
			if (snakeShellBoundingBox_.x < 1) {
				return false;
			} else if (snakeShellBoundingBox_.x + snakeShellBoundingBox_.width > imageLUTs_
					.getImageWidth() - 1) {
				return false;
			} else if (snakeShellBoundingBox_.y < 1) {
				return false;
			} else if (snakeShellBoundingBox_.y + snakeShellBoundingBox_.height > imageLUTs_
					.getImageHeight() - 1) {
				return false;
			}
		}
		return true;
	}

	// ----------------------------------------------------------------------------
	// RESAMPLING METHODS

	private Snake2DNode[] getSplineKnots(Point2D.Double[] contour, int M) {
		double[] knotsX = new double[M];
		double[] knotsY = new double[M];

		double b = 2.0 / 3.0;

		for (int i = 0; i < M; i++) {
			knotsX[i] = contour[i].x;
			knotsY[i] = contour[i].y;
		}

		double[] pole = { (-b + Math.sqrt(2 * b - 1)) / (1 - b) };
		knotsX = Filters.prescaledPeriodic(knotsX, pole);
		knotsY = Filters.prescaledPeriodic(knotsY, pole);

		Snake2DNode[] newCoeff = new Snake2DNode[M];
		for (int i = 0; i < M; i++) {
			newCoeff[i] = new Snake2DNode(knotsX[i], knotsY[i]);
		}

		return newCoeff;
	}

	// ----------------------------------------------------------------------------

	/** Assigns a prior-shape to the snake and builds the necessary projectors. */
	private void setPriorShape(ESnakePriorShapeType priorShapeType) {
		priorShapeType_ = priorShapeType;
		switch (priorShapeType_) {
		case NONE:
			priorShape_ = new Circle();
			break;
		case ROD:
			priorShape_ = new Rod();
			break;
		case BUDDINGYEAST:
			priorShape_ = new BuddingYeast();
			break;
		case BANANA:
			priorShape_ = new Banana();
			break;
		case BOOMERANG:
			priorShape_ = new Boomerang();
			break;
		case BRAINWHITE:
			priorShape_ = new BrainWhite();
			break;
		case CIRCLE:
			priorShape_ = new Circle();
			break;
		case CORPUSCALLOSUM:
			priorShape_ = new CorpusCallosum();
			break;
		case CROSS:
			priorShape_ = new Cross();
			break;
		case ELECTRICGUITAR:
			priorShape_ = new ElectricGuitar();
			break;
		case FEMUR:
			priorShape_ = new Femur();
			break;
		case FLY:
			priorShape_ = new Fly();
			break;
		case MOUSEORGAN:
			priorShape_ = new MouseOrgan();
			break;
		case SEMICIRCLE:
			priorShape_ = new Semicircle();
			break;
		case SQUARE:
			priorShape_ = new Square();
			break;
		case UPPERLIP:
			priorShape_ = new UpperLip();
			break;
		case CUSTOM:
			priorShape_ = customPriorShape_;
			break;
		default:
			break;
		}
		shapeProjectorAffine_ = shapeProjectorBuilder_
				.buildShapeProjectorAffine(priorShape_);
		shapeProjectorSimilarity_ = shapeProjectorBuilder_
				.buildShapeProjectorSimilarity(priorShape_);
	}

	// ----------------------------------------------------------------------------

	/** Updates the number of control points. */
	private void setNumNodes(double initialSample, int newM) {
		double c = BSplineBasis.correlationOfTwoESpline(3.0 / newM,
				(2 * Math.PI) / newM, (2 * Math.PI) / newM, newM, newM);
		double b = BSplineBasis.correlationOfTwoESpline(4.0 / newM,
				(2 * Math.PI) / newM, (2 * Math.PI) / newM, newM, newM);
		double a = BSplineBasis.correlationOfTwoESpline(5.0 / newM,
				(2 * Math.PI) / newM, (2 * Math.PI) / newM, newM, newM);

		double alpha1 = (b + Math.sqrt(b * b - 4 * a * c + 8 * a * a))
				/ (2 * a);
		double alpha2 = (b - Math.sqrt(b * b - 4 * a * c + 8 * a * a))
				/ (2 * a);

		double z1 = (-alpha1 + Math.sqrt(alpha1 * alpha1 - 4)) / 2;
		if (Math.abs(z1) > 1) {
			z1 = 1 / z1;
		}

		double z2 = (-alpha2 + Math.sqrt(alpha2 * alpha2 - 4)) / 2;
		if (Math.abs(z2) > 1) {
			z2 = 1 / z2;
		}

		double[] invMatrix = new double[newM];
		double cz1 = 1 / ((1 - Math.pow(z1, newM)) * (z1 - 1 / z1) * a * (z1
				+ 1 / z1 - z2 - 1 / z2));
		double cz1i = -1
				/ ((1 - Math.pow(z1, -newM)) * (z1 - 1 / z1) * a * (z1 + 1 / z1
						- z2 - 1 / z2));
		double cz2 = -1
				/ ((1 - Math.pow(z2, newM)) * (z2 - 1 / z2) * a * (z1 + 1 / z1
						- z2 - 1 / z2));
		double cz2i = 1 / ((1 - Math.pow(z2, -newM)) * (z2 - 1 / z2) * a * (z1
				+ 1 / z1 - z2 - 1 / z2));
		double z1k = 1;
		double z2k = 1;
		for (int k = 0; k < newM; k++) {
			invMatrix[k] = cz1 * z1k + cz1i * (1 / z1k) + cz2 * z2k + cz2i
					* (1 / z2k);
			z1k *= z1;
			z2k *= z2;
		}

		Point2D.Double[] vec = new Point2D.Double[newM];
		double x, y;
		for (int k = 0; k < newM; k++) {
			x = 0.0;
			y = 0.0;
			for (int i = -BSplineBasis.ESPLINE3SUPPORT; i < M_
					+ BSplineBasis.ESPLINE3SUPPORT; i++) {
				x += coef_[(M_ + i) % M_].x
						* BSplineBasis.correlationOfTwoESpline(
								((3.0 + i) / M_ - (double) k / (double) newM),
								(2 * Math.PI) / M_, (2 * Math.PI) / newM, M_,
								newM);
				y += coef_[(M_ + i) % M_].y
						* BSplineBasis.correlationOfTwoESpline(
								((3.0 + i) / M_ - (double) k / (double) newM),
								(2 * Math.PI) / M_, (2 * Math.PI) / newM, M_,
								newM);
			}
			vec[k] = new Point2D.Double(x, y);
		}

		Snake2DNode[] newCoef = new Snake2DNode[newM];
		for (int k = 0; k < newM; k++) {
			x = 0.0;
			y = 0.0;
			for (int i = 0; i < newM; i++) {
				x += vec[i].x * invMatrix[(newM + i - k) % newM];
				y += vec[i].y * invMatrix[(newM + i - k) % newM];
			}
			newCoef[k] = new Snake2DNode(x, y);
		}

		coef_ = newCoef;
		M_ = newM;
		shapeProjectorBuilder_ = new ShapeProjectorBuilder(M_);
		initialize(false);
		alive_ = true;
	}

	@Override
	public Snake2DNode[] getSubdivisionPoints() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public java.awt.geom.Point2D.Double getCentroidRefinement() {
		// TODO Auto-generated method stub
		return null;
	}

	@Override
	public boolean isLocalRefinement() {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public boolean isLocalRefinementAction() {
		// TODO Auto-generated method stub
		return false;
	}

	@Override
	public int getRefinementFactor() {
		// TODO Auto-generated method stub
		return 0;
	}

	@Override
	public int getIndexRefinement() {
		// TODO Auto-generated method stub
		return 0;
	}

	@Override
	public int getFilterSize() {
		// TODO Auto-generated method stub
		return 0;
	}

	@Override
	public void setIndexRefinement(int index) {
		// TODO Auto-generated method stub

	}

	@Override
	public void localRefinementCoef(int index) {
		// TODO Auto-generated method stub

	}

	@Override
	public int getSizeRefinementCoef() {
		// TODO Auto-generated method stub
		return 0;
	}
}
