/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.io;

import icy.file.FileUtil;
import icy.gui.frame.progress.AnnounceFrame;
import icy.roi.ROI;
import icy.sequence.Sequence;
import icy.util.StringUtil;
import icy.util.XMLUtil;

import java.util.ArrayList;

import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.filechooser.FileNameExtensionFilter;

import org.w3c.dom.Document;
import org.w3c.dom.Node;

import plugins.big.bigsnakeutils.icy.snake2D.Snake2DNode;
import plugins.big.shapedesigner.ShapeDesigner;
import plugins.big.shapedesigner.keeper.SplineCurveKeeper;
import plugins.big.shapedesigner.roi.SplineCurve2DROI;
import plugins.big.shapedesigner.splinecurve.SplineCurve;
import plugins.big.shapedesigner.splinecurve.SplineCurveParameters;

/**
 * Class that provides loading and exporting capabilities to the spline curves.
 * 
 * @version May 3, 2014
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class IOXML {

	/**
	 * Label of the XML tag the metadata of the image from which the XML was
	 * generated.
	 */
	private final static String ROOT_META = "meta";
	/**
	 * Label of the XML tag containing the name of the image from which the XML
	 * was generated.
	 */
	private final static String ID_NAME = "name";
	/**
	 * Label of the XML tag containing the the information associated to the ROI
	 * of a collection of spline curves.
	 */
	private final static String ROOT_ROIS = "rois";
	/**
	 * Label of the XML tag containing the the information associated to the ROI
	 * of a single spline curve.
	 */
	private final static String ID_ROI = "roi";

	// ============================================================================
	// PUBLIC METHODS

	/**
	 * Opens a dialog and saves all the spline curves in the active
	 * <code>Sequence</code> in an XML file.
	 */
	public static void saveSplineCurvesToXML(Sequence sequence)
			throws Exception {
		if (sequence == null) {
			throw new Exception("seq is null in saveSplineCurvesToXML.");
		}

		JFileChooser fc = new JFileChooser();
		int returnVal = fc.showSaveDialog(new JFrame());
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String filename = fc.getSelectedFile().getAbsolutePath();
			String xmlFilename = filename + ".xml";
			Document document = XMLUtil.createDocument(true);
			XMLUtil.setElementValue(document.getDocumentElement(), ID_NAME,
					sequence.getName());
			addMetaDataToXML(document.getDocumentElement(), sequence);
			addROIsToXML(document.getDocumentElement(), sequence);
			XMLUtil.saveDocument(document, xmlFilename);
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Opens a dialog to select an XML file, and loads all the spline curves in
	 * the active <code>Sequence</code>.
	 */
	public static ArrayList<SplineCurveKeeper> loadSplineCurvesFromXML(
			Sequence sequence, ShapeDesigner mainPlugin) throws Exception {

		if (sequence == null) {
			throw new Exception("sequence is null in loadSplineCurvesFromXML.");
		}

		JFrame frame = new JFrame();
		JFileChooser fc = new JFileChooser();
		FileNameExtensionFilter filter = new FileNameExtensionFilter(
				"XML Files", "xml");
		fc.setFileFilter(filter);
		int returnVal = fc.showOpenDialog(frame);
		if (returnVal == JFileChooser.APPROVE_OPTION) {
			String xmlFilename = fc.getSelectedFile().getAbsolutePath();
			Document document = null;
			if (xmlFilename != null) {
				if (FileUtil.exists(xmlFilename)) {
					document = XMLUtil.loadDocument(xmlFilename, true);
				} else {
					new AnnounceFrame("Selected file does not exist.");
					throw new Exception(
							"xmlFilename is null in loadSplineCurvesFromXML.");
				}
			} else {
				throw new Exception(
						"xmlFilename is null in loadSplineCurvesFromXML.");
			}

			final String name = XMLUtil.getElementValue(
					document.getDocumentElement(), ID_NAME, "");
			if (!StringUtil.isEmpty(name)) {
				sequence.setName(name);
			}
			loadMetaDataFromXML(document.getDocumentElement(), sequence);

			ArrayList<SplineCurveKeeper> loadedKeepers = new ArrayList<SplineCurveKeeper>();

			final Node nodeROIs = XMLUtil.getElement(
					document.getDocumentElement(), ROOT_ROIS);

			if (nodeROIs != null) {
				ArrayList<Snake2DNode[]> controlPointsList = loadSplineCurveNodesFromXML(
						nodeROIs, sequence);
				ArrayList<SplineCurveParameters> splineCurvesParametersList = loadSplineCurvesParametersFromXML(
						nodeROIs, sequence);

				int detectedSplineCurves = controlPointsList.size();
				for (int i = 0; i < detectedSplineCurves; i++) {
					SplineCurve splineCurve = new SplineCurve(sequence,
							splineCurvesParametersList.get(i), null);
					splineCurve.setNodes(controlPointsList.get(i));
					loadedKeepers.add(new SplineCurveKeeper(sequence,
							splineCurve, mainPlugin));
				}
				return loadedKeepers;
			}
		}
		return null;
	}

	// ============================================================================
	// PRIVATE METHODS

	private static void addMetaDataToXML(Node node, Sequence sequence)
			throws Exception {
		if (node == null) {
			throw new Exception("XML node is null in addMetaDataToXML.");
		}
		if (sequence == null) {
			throw new Exception("sequence is null in addMetaDataToXML.");
		}

		final Node nodeMeta = XMLUtil.setElement(node, ROOT_META);
		if (nodeMeta != null) {
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_PIXEL_SIZE_X,
					sequence.getPixelSizeX());
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_PIXEL_SIZE_Y,
					sequence.getPixelSizeY());
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_PIXEL_SIZE_Z,
					sequence.getPixelSizeZ());
			XMLUtil.setElementDoubleValue(nodeMeta, Sequence.ID_TIME_INTERVAL,
					sequence.getTimeInterval());
			for (int c = 0; c < sequence.getSizeC(); c++) {
				XMLUtil.setElementValue(nodeMeta, Sequence.ID_CHANNEL_NAME + c,
						sequence.getChannelName(c));
			}
		}
	}

	// ----------------------------------------------------------------------------

	private static void addROIsToXML(Node node, Sequence sequence)
			throws Exception {
		if (node == null) {
			throw new Exception("XML node is null in addROIsToXML.");
		}
		if (sequence == null) {
			throw new Exception("sequence is null in addROIsToXML.");
		}

		final Node nodeROIs = XMLUtil.setElement(node, ROOT_ROIS);
		if (nodeROIs != null) {
			XMLUtil.removeAllChildren(nodeROIs);
			for (ROI roi : sequence.getROIs()) {
				final Node nodeROI = XMLUtil.addElement(nodeROIs, ID_ROI);
				if (nodeROI != null) {
					boolean s = roi.saveToXML(nodeROI);
					if (!s) {
						XMLUtil.removeNode(nodeROIs, nodeROI);
					}
				}
			}
		}
	}

	// ----------------------------------------------------------------------------

	private static void loadMetaDataFromXML(Node node, Sequence sequence)
			throws Exception {

		if (node == null) {
			throw new Exception("XML node is null in loadMetaDataFromXML.");
		}

		if (sequence == null) {
			throw new Exception("Sequence is null in loadMetaDataFromXML.");
		}

		final Node nodeMeta = XMLUtil.getElement(node, ROOT_META);
		if (nodeMeta != null) {
			sequence.setPixelSizeX(XMLUtil.getElementDoubleValue(nodeMeta,
					Sequence.ID_PIXEL_SIZE_X, 1d));
			sequence.setPixelSizeY(XMLUtil.getElementDoubleValue(nodeMeta,
					Sequence.ID_PIXEL_SIZE_Y, 1d));
			sequence.setPixelSizeZ(XMLUtil.getElementDoubleValue(nodeMeta,
					Sequence.ID_PIXEL_SIZE_Z, 1d));
			sequence.setTimeInterval(XMLUtil.getElementDoubleValue(nodeMeta,
					Sequence.ID_TIME_INTERVAL, 1d));
			for (int c = 0; c < sequence.getSizeC(); c++) {
				sequence.setChannelName(c, XMLUtil.getElementValue(nodeMeta,
						Sequence.ID_CHANNEL_NAME + c, Sequence.ID_CHANNEL_NAME
								+ c));
			}
		}
	}

	// ----------------------------------------------------------------------------

	private static ArrayList<Snake2DNode[]> loadSplineCurveNodesFromXML(
			Node nodeROIs, Sequence sequence) throws Exception {

		if (nodeROIs == null) {
			throw new Exception(
					"XML node is null in loadSplineCurveNodesFromXML.");
		}
		if (sequence == null) {
			throw new Exception(
					"Sequence is null in loadSplineCurveNodesFromXML.");
		}

		final ArrayList<Node> nodesROI = XMLUtil.getChildren(nodeROIs, ID_ROI);
		ArrayList<Snake2DNode[]> controlPointsList = new ArrayList<Snake2DNode[]>();
		if (nodesROI != null) {
			for (Node n : nodesROI) {
				final Node splineCurveParametersNode = XMLUtil.getElement(n,
						SplineCurve2DROI.ID_SNAKE_PARAMETERS);
				final Node controlPointsNode = XMLUtil.getElement(
						splineCurveParametersNode,
						SplineCurve.ID_CONTROL_POINTS);
				final ArrayList<Node> controlPointNodeList = XMLUtil
						.getChildren(controlPointsNode,
								SplineCurve.ID_CONTROL_POINT);
				int M = controlPointNodeList.size();
				Snake2DNode[] controlPoints = new Snake2DNode[M];
				for (int i = 0; i < M; i++) {
					controlPoints[i] = new Snake2DNode(0, 0);
					controlPoints[i].loadFromXML(controlPointNodeList.get(i));
				}
				controlPointsList.add(controlPoints);
			}
			return controlPointsList;
		}
		return null;
	}

	// ----------------------------------------------------------------------------

	private static ArrayList<SplineCurveParameters> loadSplineCurvesParametersFromXML(
			Node nodeROIs, Sequence sequence) throws Exception {

		if (nodeROIs == null) {
			throw new Exception(
					"XML node is null in loadSplineCurvesParametersFromXML.");
		}
		if (sequence == null) {
			throw new Exception(
					"Sequence is null in loadSplineCurvesParametersFromXML.");
		}

		final ArrayList<Node> nodesROI = XMLUtil.getChildren(nodeROIs, ID_ROI);
		ArrayList<SplineCurveParameters> splineCurvesParametersList = new ArrayList<SplineCurveParameters>();
		if (nodesROI != null) {
			for (Node n : nodesROI) {
				final Node splineCurveParametersNode = XMLUtil.getElement(n,
						SplineCurve2DROI.ID_SNAKE_PARAMETERS);
				SplineCurveParameters splineCurveParameters = new SplineCurveParameters();
				splineCurveParameters.loadFromToXML(splineCurveParametersNode);
				splineCurvesParametersList.add(splineCurveParameters);
			}
			return splineCurvesParametersList;
		}
		return null;
	}
}
