package plugins.ylemontag.complex;

import icy.plugin.abstract_.Plugin;
import icy.plugin.interface_.PluginBundled;
import icy.sequence.Sequence;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.BlockInfo;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.vars.lang.VarEnum;
import plugins.adufour.vars.lang.VarSequence;
import plugins.adufour.vars.util.VarException;

/**
 * 
 * @author Yoann Le Montagner
 *
 * Complex conversion through the 'Block' interface
 */
public class ComplexFeatureBlock
{
	/**
	 * Base class
	 */
	private static abstract class Base extends Plugin implements Block, BlockInfo, PluginBundled
	{
		private VarSequence             _out    ;
		private VarEnum<ComplexFeature> _feature;
		private int                   _nbInputs      ;
		private ComplexRepresentation _representation;
		
		/**
		 * Constructor
		 */
		protected Base(int nbInputs, ComplexRepresentation representation)
		{
			_out     = new VarSequence("Output", null);
			_feature = new VarEnum<ComplexFeature>("Feature", ComplexFeature.REAL_PART);
			_nbInputs       = nbInputs      ;
			_representation = representation;
		}
		
		/**
		 * Retrieve the current representation
		 */
		protected ComplexRepresentation getRepresentation()
		{
			return _representation;
		}
		
		@Override
		public void declareInput(VarList inputMap)
		{
			inputMap.add(_feature);
		}
		
		@Override
		public void declareOutput(VarList outputMap)
		{
			outputMap.add(_out);
		}
		
		@Override
		public String getMainPluginClassName()
		{
			return ComplexFeaturePlugin.class.getName();
		}
		
		@Override
		public String getName()
		{
			return String.format("Complex from %s (%d)", _representation.getSymbol(), _nbInputs);
		}
		
		@Override
		public String getDescription()
		{
			String inputs = _nbInputs>=2 ? String.format("%d sequences", _nbInputs) : "1 sequence";
			return String.format("Extract a complex feature out of %s (%s representation)", inputs,
				_representation.getNameLowCase());
		}
		
		@Override
		public void run()
		{
			try {
				ComplexFeature feature = _feature.getValue();
				if(feature==null) {
					throw new VarException("No complex feature selected");
				}
				_out.setValue(compute(feature));
			}
			catch(InvalidComplexEntry err) {
				throw new VarException("Invalid argument: " + err.getMessage());
			}
		}
		
		/**
		 * Core function to implement
		 */
		protected abstract Sequence compute(ComplexFeature feature) throws InvalidComplexEntry;
	}
	
	/**
	 * Conversion from one complex-valued sequence
	 */
	private static class Unary extends Base
	{
		private VarSequence _in;
		
		/**
		 * Constructor
		 */
		protected Unary(ComplexRepresentation representation, String name)
		{
			super(1, representation);
			_in = new VarSequence(name, null);
		}
		
		@Override
		public void declareInput(VarList inputMap)
		{
			inputMap.add(_in);
			super.declareInput(inputMap);
		}

		@Override
		protected Sequence compute(ComplexFeature feature) throws InvalidComplexEntry
		{
			return feature.get(_in.getValue(), getRepresentation());
		}
	}
	
	/**
	 * Conversion from two real-valued sequences
	 */
	private static class Binary extends Base
	{
		private VarSequence _in1;
		private VarSequence _in2;
		
		/**
		 * Constructor
		 */
		protected Binary(ComplexRepresentation representation, String name1, String name2)
		{
			super(2, representation);
			_in1 = new VarSequence(name1, null);
			_in2 = new VarSequence(name2, null);
		}
		
		@Override
		public void declareInput(VarList inputMap)
		{
			inputMap.add(_in1);
			inputMap.add(_in2);
			super.declareInput(inputMap);
		}

		@Override
		protected Sequence compute(ComplexFeature feature) throws InvalidComplexEntry
		{
			return feature.get(_in1.getValue(), _in2.getValue(), getRepresentation());
		}
	}
	
	/**
	 * Complex-valued Cartesian input
	 */
	public static class UnaryCartesian extends Unary
	{
		/**
		 * Constructor
		 */
		public UnaryCartesian()
		{
			super(ComplexRepresentation.CARTESIAN, "In");
		}
	}
	
	/**
	 * Complex-valued polar input
	 */
	public static class UnaryPolar extends Unary
	{
		/**
		 * Constructor
		 */
		public UnaryPolar()
		{
			super(ComplexRepresentation.POLAR, "In");
		}
	}
	
	/**
	 * Real-valued Cartesian input
	 */
	public static class BinaryCartesian extends Binary
	{
		/**
		 * Constructor
		 */
		public BinaryCartesian()
		{
			super(ComplexRepresentation.CARTESIAN, "Real", "Imaginary");
		}
	}
	
	/**
	 * Real-valued polar input
	 */
	public static class BinaryPolar extends Binary
	{
		/**
		 * Constructor
		 */
		public BinaryPolar()
		{
			super(ComplexRepresentation.POLAR, "Modulus", "Argument");
		}
	}
}
