package plugins.spop.advancefilters;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;
import icy.type.collection.array.Array1DUtil;
import plugins.adufour.blocks.lang.Block;
import plugins.adufour.blocks.util.VarList;
import plugins.adufour.ezplug.EzGroup;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzVarBoolean;
import plugins.adufour.ezplug.EzVarDouble;
import plugins.adufour.ezplug.EzVarInteger;
import plugins.adufour.ezplug.EzVarSequence;
import plugins.adufour.ezplug.EzVarText;
import plugins.adufour.filtering.Convolution1D;
import plugins.adufour.filtering.Kernels1D;


public class ShockFilter extends EzPlug implements Block
{
	/**
	 * PDE Shock Filters
	 * 
	 * @author Sorin POP
	 * 
	 * @see S. Osher, L. Rudin  Feature-oriented image enhancement with shock filters, SIAM Journal on Numerical Analysis, vol.27, no.3, pp. 919-940, 1990
	 * @see G. Gilboa, N. Sochen, Y. Zeevi - Regularized Shock Filters and Complex Diffusion, ECCV 2002, LNCS 2350, pp.399-413, 2002.
	 */

    EzVarSequence input = new EzVarSequence("Input");
    EzVarText type_2D = new EzVarText("Type", new String[] {"2D", "3D"},0, false);
   

    EzVarInteger iterations = new EzVarInteger("Iterations", 5, 1, 10000, 1);
    EzVarDouble step = new EzVarDouble("Discretization step", 0.15, 0.01, 0.25, 0.01);

    EzGroup groupGauss;
    // used in order to reduce the noise influence
    EzVarBoolean gauss = new EzVarBoolean("Gaussian regularization", true);
    //local scale
    EzVarDouble sigmaX = new EzVarDouble("Sigma", 1, 0.1, 10, 0.1);
    
   
    EzGroup groupGilboa;
    EzVarBoolean gilboa = new EzVarBoolean("Gilboa version", false);
    // multiplication coefficient for the argument of atan function
    EzVarDouble gilboa_a = new EzVarDouble("Coefficient", 1, 0, 5, 0.01);
    
    EzVarSequence output = new EzVarSequence("Output");
    
    long time = 0;

    @Override
    public void initialize()
    {
        // TODO Auto-generated method stub
        addEzComponent(input);
        addEzComponent(type_2D);
        
        addEzComponent(iterations);
        addEzComponent(step);

        addEzComponent(gauss);
        groupGauss = new EzGroup("Prefiltering",  sigmaX);
        addEzComponent(groupGauss);
        gauss.addVisibilityTriggerTo(groupGauss, true);

        addEzComponent(gilboa);
        groupGilboa = new EzGroup("Gilboa Version", gilboa_a);
        addEzComponent(groupGilboa);
        gilboa.addVisibilityTriggerTo(groupGilboa, true);

        setTimeDisplay(true);

    }

    @Override
    public void execute()
    {
        // TODO Auto-generated method stub
        Sequence sequence_out=null;
        double dt = step.getValue();
     
        int iter = iterations.getValue();

        double sX = sigmaX.getValue();
       
        Sequence sequence = input.getValue(true);

        if (type_2D.getValue().equalsIgnoreCase("2D"))
            sequence_out = Shock_2D(sequence, iter, dt, gauss.getValue(), sX, gilboa.getValue(), gilboa_a.getValue());
        else
            sequence_out = Shock_3D(sequence, iter, dt, gauss.getValue(), sX, gilboa.getValue(), gilboa_a.getValue());
        
        if(getUI()!=null)
			addSequence(sequence_out);
		output.setValue(sequence_out);

    }

    public Sequence Shock_2D(Sequence sequence, int iter, double dt, boolean gauss, double sigmaX, boolean gilboa, double gilboa_a)
    {
    	int temp = 0;
        double Dx1, Dx2, Dy1, Dy2, Dx1g, Dx2g, Dy1g, Dy2g;
        double uxg, uyg;
        double ux, uy, uxx, uxy, uyy, Lij, in;
        double current, current_g;
               
        Sequence seq_out = new Sequence();       
        int dim_x = sequence.getSizeX();
        int dim_y = sequence.getSizeY();
        int dim_z = sequence.getSizeZ();            
        int dim_t = sequence.getSizeT();
        int dim_c = sequence.getSizeC();
             
        double[] tab_out = new double[dim_x*dim_y];
        
        double[] ker= Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigmaX).getData();
	
		double[][] tab_gauss = new double[1][dim_x*dim_y];
		
        for (int time_slice = 0; time_slice < dim_t; time_slice++)
        {
        	if(gauss)
        	{ // regularized version
        		
	        	for (int z = 0; z < dim_z; z++)
	        	{
	        	
		    		IcyBufferedImage icy_img = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
		        	for (int c = 0; c < dim_c; c++)
		        	{
			            double[] tab = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
			            System.arraycopy(tab, 0, tab_gauss[0], 0, dim_x*dim_y);
			            System.arraycopy(tab, 0, tab_out, 0, dim_x*dim_y);
			           	         
			            for (int t = 0; t < iter; t++)
			            {
			            	try{
			            	Convolution1D.convolve(tab_gauss, dim_x,dim_y,ker, ker,null);
			            	 } catch (Exception e) {
			         			// TODO Auto-generated catch block
			         			e.printStackTrace();
			         		} 
			                for (int x = 1; x < dim_x - 1; x++)
			                    for (int y = 1; y < dim_y - 1; y++)
			                    {
			                    	temp = x + y * dim_x;
	                                current = tab[temp];
	                                Dx1 = tab[temp - 1] - current;
	                                Dx2 = -(tab[temp + 1] - current);
	                                Dy1 = tab[temp - dim_x] - current;
	                                Dy2 = -(tab[temp + dim_x] - current);
	                                current_g = tab_gauss[0][temp];
	                                Dx1g = tab_gauss[0][temp - 1] - current_g;
	                                Dx2g = -(tab_gauss[0][temp + 1] - current_g);
	                                Dy1g = tab_gauss[0][temp - dim_x] - current_g;
	                                Dy2g = -(tab_gauss[0][temp + dim_x] - current_g);
	
	                                ux = DiffusionTools.minmod(Dx1, Dx2);
	                                uy = DiffusionTools.minmod(Dy1, Dy2);
	
	                                uxg = DiffusionTools.minmod(Dx1g, Dx2g);
	                                uyg = DiffusionTools.minmod(Dy1g, Dy2g);
	
	                                uxx = Dx1g - Dx2g;
	                                uyy = Dy1g - Dy2g;
	                                uxy = (tab_gauss[0][temp + 1 + dim_x] + tab_gauss[0][temp - 1 - dim_x]
	                                        - tab_gauss[0][temp - 1 + dim_x] - tab_gauss[0][temp + 1 - dim_x]) * 0.25;
	                                //second order directional derivative
	                                Lij = uxg * uxg * uxx + 2 * uxy * uxg * uyg + uyg * uyg * uyy;
	
	                                if (!gilboa)
	                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign(Lij);
	                                else //Gilboa et al. enhancement
	                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign_Gilboa(gilboa_a * Lij);
	                                tab_out[temp] = current - in;
	                                
			                    }
			                System.arraycopy(tab_out, 0, tab, 0, dim_x * dim_y);
			                System.arraycopy(tab_out, 0, tab_gauss[0], 0, dim_x * dim_y);
			                if(getUI()!=null)
			                	getUI().setProgressBarValue((double) (time_slice*dim_z*dim_c*iter+z*dim_c*iter+c*iter+t+1) / (double) (dim_t*dim_z*dim_c*iter));
				               
			               
			            }
			            Array1DUtil.doubleArrayToSafeArray(tab_out, icy_img.getDataXY(c), icy_img.isSignedDataType());
			            
		        	}
		        	seq_out.setImage(time_slice, z, icy_img);
		        	
	        	}
        	}
        	else
        	{
        		//classical version
        		for (int z = 0; z < dim_z; z++)
	        	{
	        	
		    		IcyBufferedImage icy_img = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
		        	for (int c = 0; c < dim_c; c++)
		        	{
			            double[] tab = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
			            System.arraycopy(tab, 0, tab_out, 0, dim_x*dim_y);
			           	         
			            for (int t = 0; t < iter; t++)
			            {
			            	
			            	for (int x = 1; x < dim_x - 1; x++)
	                            for (int y = 1; y < dim_y - 1; y++)
	                            {
	                                temp = x + y * dim_x;
	                                current = tab[temp];
	                                Dx1 = tab[temp - 1] - current;
	                                Dx2 = -(tab[temp + 1] - current);
	                                Dy1 = tab[temp - dim_x] - current;
	                                Dy2 = -(tab[temp + dim_x] - current);

	                                ux = DiffusionTools.minmod(Dx1, Dx2);
	                                uy = DiffusionTools.minmod(Dy1, Dy2);
	                                uxx = Dx1 - Dx2;
	                                uyy = Dy1 - Dy2;
	                                uxy = (tab[temp + 1 + dim_x] + tab[temp - 1 - dim_x] - tab[temp - 1 + dim_x] - tab[temp
	                                        + 1 - dim_x]) * 0.25;
	                              //second order directional derivative
	                                Lij = ux * ux * uxx + 2 * uxy * ux * uy + uy * uy * uyy;

	                                if (!gilboa)
	                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign(Lij);
	                                else //Gilboa et al. enhancement
	                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign_Gilboa(gilboa_a * Lij);
	                                tab_out[temp] = current - in;

	                            }
			                System.arraycopy(tab_out, 0, tab, 0, dim_x * dim_y);
			                if(getUI()!=null)
			                	getUI().setProgressBarValue((double) (time_slice*dim_z*dim_c*iter+z*dim_c*iter+c*iter+t+1) / (double) (dim_t*dim_z*dim_c*iter));
				               
			               
			            }
			            Array1DUtil.doubleArrayToSafeArray(tab_out, icy_img.getDataXY(c), icy_img.isSignedDataType());
			            
		        	}
		        	seq_out.setImage(time_slice, z, icy_img);
		        	
	        	}
        	}
        	
        }
        seq_out.updateComponentsBounds();
        seq_out.setName("Shock Filter 2D");
        return seq_out;
    }
    
    public Sequence Shock_3D(Sequence sequence, int iter, double dt, boolean gauss, double sigmaX, boolean gilboa, double gilboa_a)
    {
    	int temp = 0;
	    double Dx1, Dx2, Dy1, Dy2, Dx1g, Dx2g, Dy1g, Dy2g;
        double uxg, uyg, uzg;
        double ux, uy, uxx, uxy, uyy, Lij, in;
        double Dz1, Dz2, Dz1g, Dz2g, uz, uzz, uxz, uyz;
        double current, current_g;
             
        Sequence seq_out = new Sequence();       
        int dim_x = sequence.getSizeX();
        int dim_y = sequence.getSizeY();
        int dim_z = sequence.getSizeZ();
        int dim_t = sequence.getSizeT();
        int dim_c = sequence.getSizeC();
        
        double[][] tab     = new double[dim_z][];
        double[][] tab_out = new double[dim_z][dim_x*dim_y];
        
        double[] ker= Kernels1D.CUSTOM_GAUSSIAN.createGaussianKernel1D(sigmaX).getData();
	
		double[][] tab_gauss = new double[dim_z][dim_x*dim_y];
        
        for (int time_slice = 0; time_slice < dim_t; time_slice++)
        {
        		IcyBufferedImage[] icy_img = new IcyBufferedImage[dim_z];
        		for (int z = 0; z < dim_z; z++)
        			icy_img[z] = new IcyBufferedImage(dim_x, dim_y, dim_c, sequence.getDataType());
        		if(gauss)
        		{
        			//regularized version
		        	for (int c = 0; c < dim_c; c++)
		        	{
		        		for (int z = 0; z < dim_z; z++)
		        		{
		        			tab[z] = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
		        			System.arraycopy(tab[z], 0, tab_gauss[z], 0, dim_x*dim_y);
		        			System.arraycopy(tab[z], 0, tab_out[z], 0, dim_x*dim_y);
		        		}
			           	         
			            for (int t = 0; t < iter; t++)
			            {
			            	try {
								Convolution1D.convolve(tab_gauss, dim_x,dim_y,ker, ker,ker);
			            	 } catch (Exception e) {
			         			// TODO Auto-generated catch block
			         			e.printStackTrace();
			         		} 
			            	for (int z = 0; z < dim_z ; z++)
			            	{
			            		if (z == 0 || z == dim_z - 1)
			            		{
			            			for (int x = 1; x < dim_x - 1; x++)
				                    for (int y = 1; y < dim_y - 1; y++)
				                    {
				                    	temp = x + y * dim_x;
		                                current = tab[z][temp];
		                                Dx1 = tab[z][temp - 1] - current;
		                                Dx2 = -(tab[z][temp + 1] - current);
		                                Dy1 = tab[z][temp - dim_x] - current;
		                                Dy2 = -(tab[z][temp + dim_x] - current);
		                                current_g = tab_gauss[z][temp];
		                                Dx1g = tab_gauss[z][temp - 1] - current_g;
		                                Dx2g = -(tab_gauss[z][temp + 1] - current_g);
		                                Dy1g = tab_gauss[z][temp - dim_x] - current_g;
		                                Dy2g = -(tab_gauss[z][temp + dim_x] - current_g);
		
		                                ux = DiffusionTools.minmod(Dx1, Dx2);
		                                uy = DiffusionTools.minmod(Dy1, Dy2);
		
		                                uxg = DiffusionTools.minmod(Dx1g, Dx2g);
		                                uyg = DiffusionTools.minmod(Dy1g, Dy2g);
		
		                                uxx = Dx1g - Dx2g;
		                                uyy = Dy1g - Dy2g;
		                                uxy = (tab_gauss[z][temp + 1 + dim_x] + tab_gauss[z][temp - 1 - dim_x]
		                                        - tab_gauss[z][temp - 1 + dim_x] - tab_gauss[z][temp + 1 - dim_x]) * 0.25;
		                              //second order directional derivative
		                                Lij = uxg * uxg * uxx + 2 * uxy * uxg * uyg + uyg * uyg * uyy;
		
		                                if (!gilboa)
		                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign(Lij);
		                                else //Gilboa et al. enhancement
		                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign_Gilboa(gilboa_a * Lij);
		                                tab_out[z][temp] = current - in; 
				                    }
			            			
			            		}
			            		else
			            		{
					                for (int x = 1; x < dim_x - 1; x++)
				                    for (int y = 1; y < dim_y - 1; y++)
				                    {
				                    	temp = x + y * dim_x;
		                                current = tab[z][temp];
		                                Dx1 = tab[z][temp - 1] - current;
		                                Dx2 = -(tab[z][temp + 1] - current);
		                                Dy1 = tab[z][temp - dim_x] - current;
		                                Dy2 = -(tab[z][temp + dim_x] - current);
		                                Dz1 = tab[z-1][temp] - current;
		                                Dz2 = -(tab[z+1][temp] - current);
		                                current_g = tab_gauss[z][temp];
		                                Dx1g = tab_gauss[z][temp - 1] - current_g;
		                                Dx2g = -(tab_gauss[z][temp + 1] - current_g);
		                                Dy1g = tab_gauss[z][temp - dim_x] - current_g;
		                                Dy2g = -(tab_gauss[z][temp + dim_x] - current_g);
		                                Dz1g = tab_gauss[z-1][temp] - current_g;
		                                Dz2g = -(tab_gauss[z+1][temp] - current_g);

		                                ux = DiffusionTools.minmod(Dx1, Dx2);
		                                uy = DiffusionTools.minmod(Dy1, Dy2);
		                                uz = DiffusionTools.minmod(Dz1, Dz2);

		                                uxg = DiffusionTools.minmod(Dx1g, Dx2g);
		                                uyg = DiffusionTools.minmod(Dy1g, Dy2g);
		                                uzg = DiffusionTools.minmod(Dz1g, Dz2g);

		                                uxx = Dx1g - Dx2g;
		                                uyy = Dy1g - Dy2g;
		                                uzz = Dz1g - Dz2g;

		                                uxy = (tab_gauss[z][temp + 1 + dim_x] + tab_gauss[z][temp - 1 - dim_x]
		                                        - tab_gauss[z][temp - 1 + dim_x] - tab_gauss[z][temp + 1 - dim_x]) * 0.25;
		                                uxz = (tab_gauss[z+1][temp + 1] + tab_gauss[z-1][temp - 1] - tab_gauss[z+1][temp - 1] - tab_gauss[z-1][temp + 1]) * 0.25;
		                                uyz = (tab_gauss[z+1][temp + dim_x] + tab_gauss[z-1][temp - dim_x]
		                                        - tab_gauss[z+1][temp - dim_x] - tab_gauss[z-1][temp + dim_x]) * 0.25;
		                              //second order directional derivative
		                                Lij = uxg * uxg * uxx + uyg * uyg * uyy + uzg * uzg * uzz + 2 * uxy * uxg * uyg + 2 * uxz * uxg * uzg + 2 * uyz * uzg * uyg;

		                                if (!gilboa)
		                                    in = dt * Math.sqrt(ux * ux + uy * uy + uz * uz) * DiffusionTools.sign(Lij);
		                                else //Gilboa et al. enhancement
		                                    in = dt * Math.sqrt(ux * ux + uy * uy + uz * uz) * DiffusionTools.sign_Gilboa(gilboa_a * Lij);
		                                tab_out[z][temp] = current - in;
				                    }
			            		}
			            	}
			            	for (int z = 0; z < dim_z; z++)
			            	{
			            		System.arraycopy(tab_out[z], 0, tab[z], 0, dim_x * dim_y);
			            		System.arraycopy(tab_out[z], 0, tab_gauss[z], 0, dim_x * dim_y);
			            	}
			            	if(getUI()!=null)
			            		getUI().setProgressBarValue((double) (time_slice*dim_c*iter+c*iter+t+1) / (double) (dim_c*dim_t*iter));
			               
			            }
			            for (int z = 0; z < dim_z; z++)
			            	Array1DUtil.doubleArrayToSafeArray(tab_out[z], icy_img[z].getDataXY(c), icy_img[z].isSignedDataType());	
			            System.gc();
		        	}
        		}
        		else
        		{
        			//classical verison
        			for (int c = 0; c < dim_c; c++)
		        	{
		        		for (int z = 0; z < dim_z; z++)
		        		{
		        			tab[z] = Array1DUtil.arrayToDoubleArray(sequence.getDataXY(time_slice, z, c), false);
		        			System.arraycopy(tab[z], 0, tab_out[z], 0, dim_x*dim_y);
		        			
		        		}
			           	         
			            for (int t = 0; t < iter; t++)
			            {
			            	
			            	for (int z = 0; z < dim_z ; z++)
			            	{
			            		if (z == 0 || z == dim_z - 1)
			            		{
			            			for (int x = 1; x < dim_x - 1; x++)
				                    for (int y = 1; y < dim_y - 1; y++)
				                    {
				                    	temp = x + y * dim_x;
		                                current = tab[z][temp];
		                                Dx1 = tab[z][temp - 1] - current;
		                                Dx2 = -(tab[z][temp + 1] - current);
		                                Dy1 = tab[z][temp - dim_x] - current;
		                                Dy2 = -(tab[z][temp + dim_x] - current);
		                               
		                                ux = DiffusionTools.minmod(Dx1, Dx2);
		                                uy = DiffusionTools.minmod(Dy1, Dy2);
		                                uxx = Dx1 - Dx2;
		                                uyy = Dy1 - Dy2;
		                                uxy = (tab[z][temp + 1 + dim_x] + tab[z][temp - 1 - dim_x] - tab[z][temp - 1 + dim_x] - tab[z][temp + 1 - dim_x]) * 0.25;
		                              //second order directional derivative
		                                Lij = ux * ux * uxx + 2 * uxy * ux * uy + uy * uy * uyy;

		                                if (!gilboa)
		                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign(Lij);
		                                else //Gilboa et al. enhancement
		                                    in = dt * Math.sqrt(ux * ux + uy * uy) * DiffusionTools.sign_Gilboa(gilboa_a * Lij);
		                                tab_out[z][temp] = current - in;
				                    }
			            			
			            		}
			            		else
			            		{
					                for (int x = 1; x < dim_x - 1; x++)
				                    for (int y = 1; y < dim_y - 1; y++)
				                    {
				                    	temp = x + y * dim_x;
		                                current = tab[z][temp];
		                                Dx1 = tab[z][temp - 1] - current;
		                                Dx2 = -(tab[z][temp + 1] - current);
		                                Dy1 = tab[z][temp - dim_x] - current;
		                                Dy2 = -(tab[z][temp + dim_x] - current);
		                                Dz1 = tab[z-1][temp] - current;
		                                Dz2 = -(tab[z+1][temp] - current);
		                               
		                                ux = DiffusionTools.minmod(Dx1, Dx2);
		                                uy = DiffusionTools.minmod(Dy1, Dy2);
		                                uz = DiffusionTools.minmod(Dz1, Dz2);
		                                uxx = Dx1 - Dx2;
		                                uyy = Dy1 - Dy2;
		                                uzz = Dz1 - Dz2;

		                                uxy = (tab[z][temp + 1 + dim_x] + tab[z][temp - 1 - dim_x] - tab[z][temp - 1 + dim_x] - tab[z][temp + 1 - dim_x]) * 0.25;

		                                uxz = (tab[z+1][temp + 1] + tab[z-1][temp - 1] - tab[z+1][temp - 1] - tab[z-1][temp + 1]) * 0.25;
		                                uyz = (tab[z+1][temp + dim_x] + tab[z-1][temp - dim_x] - tab[z+1][temp - dim_x] - tab[z-1][temp + dim_x]) * 0.25;
		                              //second order directional derivative
		                                Lij = ux * ux * uxx + uy * uy * uyy + uz * uz * uzz + 2 * uxy * ux * uy + 2 * uxz * ux
		                                        * uz + 2 * uyz * uz * uy;

		                                if (!gilboa)
		                                    in = dt * Math.sqrt(ux * ux + uy * uy + uz * uz) * DiffusionTools.sign(Lij);
		                                else
		                                    in = dt * Math.sqrt(ux * ux + uy * uy + uz * uz) * DiffusionTools.sign_Gilboa(gilboa_a * Lij);
		                                tab_out[z][temp] = current - in;
				                    }
			            		}
			            	}
			            	for (int z = 0; z < dim_z; z++)
			            	{
			            		System.arraycopy(tab_out[z], 0, tab[z], 0, dim_x * dim_y);
			            		
			            	}
			            	if(getUI()!=null)
			            		getUI().setProgressBarValue((double) (time_slice*dim_c*iter+c*iter+t+1) / (double) (dim_c*dim_t*iter));
			               
			            }
			            for (int z = 0; z < dim_z; z++)
			            	Array1DUtil.doubleArrayToSafeArray(tab_out[z], icy_img[z].getDataXY(c), icy_img[z].isSignedDataType());	
			            System.gc();
		        	}
        			
        		}
	        	for (int z = 0; z < dim_z; z++)
	        		seq_out.setImage(time_slice, z, icy_img[z]);
        }
        seq_out.updateComponentsBounds();
        seq_out.setName("Shock Filter 3D");
        return seq_out;

    }

    public void clean()
    {
    }

	@Override
	public void declareInput(VarList inputMap) {
		// TODO Auto-generated method stub
		inputMap.add(input.getVariable());
		inputMap.add(type_2D.getVariable());
		   

		inputMap.add(iterations.getVariable());
		inputMap.add(step.getVariable());		   
	    inputMap.add(gauss.getVariable());	   
	    inputMap.add(sigmaX.getVariable()); 
	    inputMap.add(gilboa.getVariable());
	    inputMap.add(gilboa_a.getVariable());
	}

	@Override
	public void declareOutput(VarList outputMap) {
		// TODO Auto-generated method stub
		outputMap.add(output.getVariable());
		
	}

    

}
/*
 * 

                                */
