/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Philippe Th&#233;venaz (philippe.thevenaz@epfl.ch)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 *     C&#233;dric Vonesch (cedric.vonesch@epfl.ch)
 *     Virginie Uhlmann (virginie.uhlmann@epfl.ch)
 *     Cl&#233;ment Marti (clement.marti@epfl.ch)
 *     Julien Jacquemot (julien.jacquemot@epfl.ch)
 ******************************************************************************/
package plugins.big.bigsnakeutils.icy.ellipsoid;

import icy.roi.ROI;

import java.util.HashMap;

/**
 * Class describing a 2D ellipsoid (i.e. an ellipse)
 * 
 * @version May 3, 2014
 * 
 * @author Julien Jacquemot
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class Ellipsoid2D extends AbstractEllipsoid {
	/** Major axis of the ellipse. */
	public double a = 0;

	/** Minor axis of the ellipse. */
	public double b = 0;

	/** x coordinate of the center of the ellipse. */
	public double x0 = 0;

	/** y coordinate of the center of the ellipse. */
	public double y0 = 0;

	/** Angle of the ellipse. */
	public double alpha = 0;

	/** Default constructor. */
	public Ellipsoid2D() {
		t_ = 0;
	}

	/**
	 * Default constructor.
	 * 
	 * @param t
	 *            Time corresponding to this cell.
	 */
	public Ellipsoid2D(int t) {
		t_ = t;
	}

	/**
	 * Copy constructor.
	 * 
	 * @param other
	 *            Ellipse to copy.
	 */
	public Ellipsoid2D(final Ellipsoid2D other) {
		a = other.a;
		b = other.b;
		x0 = other.x0;
		y0 = other.y0;
		alpha = other.alpha;
		t_ = other.t_;
		properties_ = new HashMap<String, Double>(other.properties_);
	}

	@Override
	public AbstractEllipsoid clone() {
		return new Ellipsoid2D(this);
	}

	@Override
	public boolean isValid() {
		return (!((Double) a).isNaN() && !((Double) b).isNaN()
				&& !((Double) x0).isNaN() && !((Double) y0).isNaN()
				&& !((Double) alpha).isNaN() && b > 0 && a > 0);
	}

	@Override
	public double getMinimalRadius() {
		return Math.min(a, b);
	}

	@Override
	public double getMaximalRadius() {
		return Math.max(a, b);
	}

	@Override
	public double getPerimeter() {
		return 2 * Math.PI * Math.sqrt((a * a + b * b) / 2);
	}

	@Override
	public double getVolume() {
		return Math.PI * a * b;
	}

	@Override
	public double[] getRotationMatrix() {
		double c = Math.cos(alpha);
		double s = Math.sin(alpha);

		double[] R = { c, -s, s, c };

		return R;
	}

	@Override
	public ROI toROI() {
		return new EllipsoidROI2D(this);
	}

	@Override
	public String toString() {
		return "Ellipse (t=" + t_ + ")";
	}

	/** Returns true if the given point is inside the ellipse. */
	public boolean contains(double x, double y) {
		return contains(x, y, getRotationMatrix());
	}

	/**
	 * Returns true if the give point is insid the ellipse.
	 * 
	 * @param R
	 *            Rotation matrix of the ellipse.
	 * */
	public boolean contains(double x, double y, final double R[]) {
		return normalizedPoint(x, y, R) <= 1;
	}

	/** Return true if the given point is on the edge of the ellipse. */
	public boolean isOver(double x, double y) {
		return Math.abs(1 - normalizedPoint(x, y, getRotationMatrix())) < 0.01;
	}

	/**
	 * Return a value indicating the distance between the given point and the
	 * ellipse.
	 * 
	 * @return If the value is < 1, the point is inside the ellipse. If the
	 *         value is 1, the point is on the contour. Otherwise, the point is
	 *         outside the ellipse.
	 */
	private double normalizedPoint(double x, double y, final double R[]) {
		double V[] = new double[] { x - x0, y - y0 };
		V = applyRotation(R, V);

		V[0] /= a;
		V[1] /= b;

		return V[0] * V[0] + V[1] * V[1];
	}
}
