/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Nicolas Chenouard (nicolas.chenouard@gmail.com)
 *     Emrah Bostan (emrah.bostan@gmail.com)
 *     Ulugbek S. Kamilov (kamilov@gmail.com)
 *     Ramtin Madani (ramtin_madani@yahoo.com)
 *     Masih Nilchian (masih_n85@yahoo.com)
 ******************************************************************************/
package plugins.big.shapedesigner.roi;

import icy.roi.ROI;

import java.awt.Polygon;
import java.awt.geom.Ellipse2D;
import java.awt.geom.PathIterator;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;

import plugins.kernel.roi.roi2d.ROI2DEllipse;
import plugins.kernel.roi.roi2d.ROI2DPolygon;
import plugins.kernel.roi.roi2d.ROI2DRectangle;

/**
 * Provides mechanisms to transform <code>ROI</code> objects from Icy into
 * polygons.
 * 
 * @version May 3, 2014
 * 
 * @author Nicolas Chenouard (nicolas.chenouard@gmail.com)
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 */
public class ROIParser {

	/**
	 * Parses a the first 2D ROI of the input list and returns a polygon with
	 * the outline.
	 */
	public static Polygon parseFirstFreeROI(ArrayList<ROI> rois) {
		Polygon initialContour = null;
		for (int i = 0; i < rois.size() && initialContour == null; i++) {
			initialContour = parseROI(rois.get(i));
		}
		return initialContour;
	}

	// ----------------------------------------------------------------------------

	/** Parses a 2D ROI and returns a polygon with the outline. */
	public static Polygon parseROI(ROI roi) {
		Polygon initialContour = null;
		if (!(roi instanceof SplineCurve2DROI)) {
			if (roi instanceof ROI2DPolygon) {
				initialContour = ((ROI2DPolygon) roi).getPolygon();
			} else if (roi instanceof ROI2DEllipse) {
				Ellipse2D ellipse = ((ROI2DEllipse) roi).getEllipse();
				initialContour = new Polygon();
				double[] coords = new double[6];
				for (PathIterator pi = ellipse.getPathIterator(null); !pi
						.isDone(); pi.next()) {
					int type = pi.currentSegment(coords);
					switch (type) {
					case PathIterator.SEG_MOVETO:
						initialContour.addPoint((int) coords[0],
								(int) coords[1]);
						break;
					case PathIterator.SEG_LINETO:
						initialContour.addPoint((int) coords[0],
								(int) coords[1]);
						break;
					case PathIterator.SEG_QUADTO:
						initialContour.addPoint((int) coords[0],
								(int) coords[1]);
						initialContour.addPoint((int) coords[2],
								(int) coords[3]);
						break;
					case PathIterator.SEG_CUBICTO:
						initialContour.addPoint((int) coords[0],
								(int) coords[1]);
						initialContour.addPoint((int) coords[2],
								(int) coords[3]);
						initialContour.addPoint((int) coords[4],
								(int) coords[5]);
						break;
					}
				}
			} else if (roi instanceof ROI2DRectangle) {
				Rectangle2D rectangle = ((ROI2DRectangle) roi).getRectangle();
				initialContour = new Polygon();
				initialContour.addPoint((int) rectangle.getMinX(),
						(int) rectangle.getMinY());
				initialContour.addPoint((int) rectangle.getMaxX(),
						(int) rectangle.getMinY());
				initialContour.addPoint((int) rectangle.getMaxX(),
						(int) rectangle.getMaxY());
				initialContour.addPoint((int) rectangle.getMinX(),
						(int) rectangle.getMaxY());
			}
			roi.remove();
		}
		return initialContour;
	}
}
