/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 ******************************************************************************/
package plugins.big.steerablej.gui;

import icy.gui.component.sequence.SequencePreviewPanel;
import icy.gui.frame.IcyFrame;
import icy.gui.frame.IcyFrameAdapter;
import icy.gui.frame.IcyFrameEvent;
import icy.sequence.Sequence;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import plugins.big.steerablej.FeatureDetector;
import plugins.big.steerablej.core.Settings;

/**
 * Main interface of the plug-in.
 * 
 * @version April 25, 2013
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 */
public class MainDialog extends IcyFrame implements ActionListener,
		ChangeListener, ItemListener {

	/** Main interface panel. */
	private JPanel mainPanel_ = null;

	// ----------------------------------------------------------------------------
	// POP-UP MENUS

	/** Allows the user to select the feature to detect (edge or ridge). */
	private JComboBox chooseType_ = null;
	/**
	 * Allows the user to select the quality of the filter. The higher the
	 * quality is, the higher the order of the steerable filter.
	 */
	private JComboBox chooseQuality_ = null;

	// ----------------------------------------------------------------------------
	// BUTTONS

	/** Initiates the filtering procedure. */
	private JButton runButton_ = null;
	/** Closes the plug-in. */
	private JButton exitButton_ = null;
	/** Initiates the non-maxima removal of the feature map. */
	private JButton nmsButton_ = null;
	/** Displays the local orientation estimation. */
	private JButton orientationButton_ = null;
	/** Displays the response to rotations of the feature template. */
	private JButton rotationButton_ = null;

	// ----------------------------------------------------------------------------
	// OTHER INTERFACE ELEMENTS

	/**
	 * Spinner with the value of the standard deviation of the Gaussian kernel
	 * used to build the template.
	 */
	private final JSpinner sigmaSpinner_ = new JSpinner();

	/** Panel containing the preview image of the steerable filter. */
	private SequencePreviewPanel previewPanel_ = null;
	/**
	 * Class that encapsulates the number crunching to generate a preview of the
	 * steerable filter.
	 */
	private FilterPreview previewGenerator_ = null;

	/** Type of detector. Edge detector (1) or ridge detector (2). */
	private int detectorType_ = 1;
	/**
	 * Quality of the filter. Normal (1), high (2) or highest (3) for the edge
	 * detector, and normal (1) or highest (2) for the ridge detector.
	 */
	private int quality_ = 1;

	/** Reference to the main plug-in object. */
	private FeatureDetector plugin_ = null;

	// ============================================================================
	// PUBLIC METHODS

	/** Constructor. */
	public MainDialog(FeatureDetector plugin) {
		super("", false, true);
		setTitle(Settings.getInstance().getAppName() + " "
				+ Settings.getInstance().getAppVersion());
		plugin_ = plugin;
		previewGenerator_ = new FilterPreview(plugin_.getAlpha().getAlpha(0,
				plugin_.getSigma(), plugin_.getM()), plugin_.getM());
		doDialog();
		refreshPreview();
	}

	// ----------------------------------------------------------------------------

	/** Closes frame and terminates plug-in. */
	@Override
	public void onClosed() {
		plugin_.killDetector();
		super.onClosed();
	}

	// ----------------------------------------------------------------------------

	/**
	 * Sets the interface in the appropriate mode after the detection is
	 * performed.
	 */
	public void detectorFinished() {
		runButton_.setText("Run");
		nmsButton_.setEnabled(true);
		rotationButton_.setEnabled(true);
		orientationButton_.setEnabled(true);
	}

	// ----------------------------------------------------------------------------
	// ACTION LISTENER METHODS

	/**
	 * Implements the <code>actionPerformed</code> for the
	 * <code>ActionListener</code>.
	 */
	@Override
	public synchronized void actionPerformed(ActionEvent e) {
		if (e.getSource() == runButton_) {
			if (!plugin_.isRunning()) {
				runButton_.setText("Abort");
				plugin_.processActiveImage();
			} else {
				plugin_.stopDetector();
				runButton_.setText("Run");
			}
		} else if (e.getSource() == exitButton_) {
			plugin_.killDetector();
			dispose();
		} else if (e.getSource() == rotationButton_) {
			plugin_.showRotations(32);
		} else if (e.getSource() == orientationButton_) {
			plugin_.showOrientations();
		} else if (e.getSource() == nmsButton_) {
			plugin_.showNMS();
		}
	}

	// ----------------------------------------------------------------------------
	// CHANGE LISTENER METHODS

	/**
	 * Implements the <code>stateChanged</code> for the
	 * <code>ChangeListener</code>.
	 */
	@Override
	public void stateChanged(ChangeEvent e) {
		if (e.getSource() == sigmaSpinner_) {
			plugin_.setSigma((Double) sigmaSpinner_.getValue());
		}
	}

	// ----------------------------------------------------------------------------
	// ITEM LISTENER METHODS

	/**
	 * Implements the <code>itemStateChanged</code> for the
	 * <code>ItemListener</code>.
	 */
	@Override
	public void itemStateChanged(ItemEvent e) {
		if (e.getSource() == chooseType_) {
			detectorType_ = chooseType_.getSelectedIndex() + 1;
			quality_ = 1;
			refreshQualityChoice();
		}

		if (e.getSource() == chooseQuality_) {
			quality_ = chooseQuality_.getSelectedIndex() + 1;
		}

		int M = quality_ * 2 - detectorType_ % 2;
		plugin_.setM(M);
		plugin_.getAlpha().setDefault(M, M % 2 == 1 ? 1 : 2,
				plugin_.getParameterSet());
		refreshPreview();
	}

	// ============================================================================
	// PRIVATE METHODS

	/** Configures a <code>JButton</code> with a particular size. */
	private void configButton(JButton jb, int w, int h) {
		jb.setPreferredSize(new Dimension(w, h));
	}

	// ----------------------------------------------------------------------------

	/** Configures a <code>JButton</code> with a particular size and insets. */
	private void configButton(JButton jb, int w, int h, Insets insets) {
		jb.setPreferredSize(new Dimension(w, h));
		jb.setMargin(insets);
	}

	// ----------------------------------------------------------------------------

	/** Builds the main dialog. */
	private void doDialog() {
		JLabel typeLabel = new JLabel("Detector type:");
		JLabel qualityLabel = new JLabel("Quality:");
		JLabel sigmaLabel = new JLabel("Feature width:");

		chooseType_ = new JComboBox();
		chooseType_.addItem("Edge");
		chooseType_.addItem("Ridge");
		chooseType_.setSelectedIndex(0);

		chooseQuality_ = new JComboBox();
		chooseQuality_.addItem("Normal");
		chooseQuality_.addItem("High");
		chooseQuality_.addItem("Highest");
		chooseQuality_.setSelectedIndex(0);

		sigmaSpinner_.setModel(new SpinnerNumberModel(plugin_.getSigma(), 0.0,
				Double.MAX_VALUE, 0.01));
		sigmaSpinner_.setPreferredSize(new Dimension(100, 22));

		// Define global constraints
		GridBagConstraints c = new GridBagConstraints();
		c.ipadx = 0;
		c.ipady = 0;
		c.insets = new Insets(0, 0, 0, 0);
		c.anchor = GridBagConstraints.NORTHWEST;

		JPanel settingsPanel = new JPanel(new GridBagLayout());
		settingsPanel.setPreferredSize(new Dimension(230, 180));
		settingsPanel.setBorder(BorderFactory.createTitledBorder("Settings"));

		c.gridy = 0;
		c.gridx = 0;
		c.insets = new Insets(0, 0, 10, 10);
		settingsPanel.add(typeLabel, c);
		c.gridx = 1;
		c.gridwidth = 2;
		settingsPanel.add(chooseType_, c);

		c.gridy = 1;
		c.gridx = 0;
		c.gridwidth = 1;
		settingsPanel.add(qualityLabel, c);
		c.gridx = 1;
		c.gridwidth = 2;
		c.insets = new Insets(0, 0, 10, 0);
		settingsPanel.add(chooseQuality_, c);

		c.gridy = 2;
		c.gridx = 0;
		c.gridwidth = 1;
		settingsPanel.add(sigmaLabel, c);
		c.gridx = 1;
		settingsPanel.add(sigmaSpinner_, c);
		c.gridx = 2;
		c.insets = new Insets(0, 0, 0, 0);

		JPanel actionsPanel = new JPanel(new GridBagLayout());

		nmsButton_ = new JButton("Narrow features");
		nmsButton_.setEnabled(false);
		configButton(nmsButton_, 160, 29);
		rotationButton_ = new JButton("Display rotation");
		rotationButton_.setEnabled(false);
		configButton(rotationButton_, 160, 29);
		orientationButton_ = new JButton("Show orientation");
		orientationButton_.setEnabled(false);
		configButton(orientationButton_, 160, 29);

		c.gridx = 0;
		c.gridy = 0;
		c.insets = new Insets(0, 0, 10, 0);
		actionsPanel.add(nmsButton_, c);
		c.gridy = 1;
		actionsPanel.add(rotationButton_, c);
		c.gridy = 2;
		actionsPanel.add(orientationButton_, c);

		actionsPanel.setPreferredSize(new Dimension(200, 180));
		actionsPanel.setBorder(BorderFactory.createTitledBorder("Actions"));

		JPanel previewPanel = new JPanel(new GridBagLayout());
		previewPanel.setPreferredSize(new Dimension(181, 180));
		previewPanel.setBorder(BorderFactory.createTitledBorder("Template"));
		c.insets = new Insets(0, 0, 0, 0);
		c.gridx = 0;
		c.gridy = 0;
		previewPanel_ = new SequencePreviewPanel();
		previewPanel_.setPreferredSize(new Dimension(
				FilterPreview.PREVIEW_WIDTH, FilterPreview.PREVIEW_HEIGHT));
		previewPanel.add(previewPanel_, c);

		runButton_ = new JButton("Run");
		configButton(runButton_, 100, 22, new Insets(0, 0, 0, 0));
		exitButton_ = new JButton("Exit");
		configButton(exitButton_, 80, 22, new Insets(0, 0, 0, 0));

		mainPanel_ = new JPanel(new GridBagLayout());

		c.gridx = 0;
		c.gridy = 0;
		c.anchor = GridBagConstraints.NORTH;
		c.insets = new Insets(20, 20, 0, 0);
		mainPanel_.add(settingsPanel, c);

		c.insets = new Insets(10, 20, 10, 0);
		c.gridx = 0;
		c.gridy = 1;
		mainPanel_.add(runButton_, c);

		c.gridx = 1;
		c.gridy = 0;
		c.anchor = GridBagConstraints.CENTER;
		c.insets = new Insets(20, 20, 0, 20);
		mainPanel_.add(previewPanel, c);

		c.gridx = 2;
		c.gridy = 0;
		c.anchor = GridBagConstraints.NORTHEAST;
		c.insets = new Insets(20, 0, 0, 20);
		mainPanel_.add(actionsPanel, c);

		c.gridx = 2;
		c.gridy = 1;
		c.insets = new Insets(10, 0, 10, 20);
		c.anchor = GridBagConstraints.CENTER;
		mainPanel_.add(exitButton_, c);

		// Add Listeners
		runButton_.addActionListener(this);
		exitButton_.addActionListener(this);
		chooseType_.addItemListener(this);
		chooseQuality_.addItemListener(this);
		nmsButton_.addActionListener(this);
		orientationButton_.addActionListener(this);
		rotationButton_.addActionListener(this);
		sigmaSpinner_.addChangeListener(this);

		// sigmaText_.getDocument().addDocumentListener(this);
		// addWindowListener(this);

		add(mainPanel_);
		pack();

		final MainDialog md = this;
		addFrameListener(new IcyFrameAdapter() {
			@Override
			public void icyFrameInternalized(IcyFrameEvent e) {
				md.pack();
			}

			@Override
			public void icyFrameExternalized(IcyFrameEvent e) {
				md.pack();
			}
		});

		setResizable(false);

		plugin_.addIcyFrame(this);
		center();

		setVisible(true);
	}

	// ----------------------------------------------------------------------------

	/** Updates the combo box of the quality. */
	private void refreshQualityChoice() {
		chooseQuality_.removeAllItems();
		chooseQuality_.addItem("Normal");
		if (detectorType_ == 1) {
			chooseQuality_.addItem("High");
		}
		chooseQuality_.addItem("Highest");
		chooseQuality_.setSelectedIndex(quality_ - 1);
	}

	// ----------------------------------------------------------------------------

	/** Updates the preview display of the analysis filter. */
	private void refreshPreview() {
		Sequence preview = previewGenerator_.getPreview(plugin_.getAlpha()
				.getAlpha(0, plugin_.getSigma(), plugin_.getM()), plugin_
				.getM());
		previewPanel_.setModel(preview);

	}
}