package plugins.ylemontag.complex;

import icy.gui.frame.progress.AnnounceFrame;
import icy.sequence.Sequence;
import icy.system.IcyHandledException;
import plugins.adufour.ezplug.EzPlug;
import plugins.adufour.ezplug.EzStoppable;
import plugins.adufour.ezplug.EzVarEnum;
import plugins.adufour.ezplug.EzVarSequence;

/**
 * 
 * @author Yoann Le Montagner
 * 
 * Plugin for extracting components in a complex-valued sequence
 */
public class ComplexFeaturePlugin extends EzPlug implements EzStoppable
{
	/**
	 * Either one or two arguments
	 */
	private static enum Mode
	{
		COMPLEX_CARTESIAN("Complex-valued, Cartesian"),
		COMPLEX_POLAR    ("Complex-valued, polar"    ),
		REAL_CARTESIAN   ("Real-valued, Cartesian"   ),
		REAL_POLAR       ("Real-valued, polar"       );
		
		private String _name;
		
		/**
		 * Constructor
		 */
		private Mode(String name)
		{
			_name = name;
		}
		
		@Override
		public String toString()
		{
			return _name;
		}
	}
	
	private EzVarEnum<Mode> _mode;
	private EzVarSequence _inputCartesian;
	private EzVarSequence _inputPolar    ;
	private EzVarSequence _inputReal     ;
	private EzVarSequence _inputImaginary;
	private EzVarSequence _inputModulus  ;
	private EzVarSequence _inputArgument ;
	private EzVarEnum<ComplexFeature> _feature;
	private Controller _controller;
	
	@Override
	protected void initialize()
	{
		// Create the inputs
		_mode = new EzVarEnum<Mode>("Input mode", Mode.values());
		_inputCartesian = new EzVarSequence("Input - Cartesian");
		_inputPolar     = new EzVarSequence("Input - Polar"    );
		_inputReal      = new EzVarSequence("Input - Real"     );
		_inputImaginary = new EzVarSequence("Input - Imaginary");
		_inputModulus   = new EzVarSequence("Input - Modulus"  );
		_inputArgument  = new EzVarSequence("Input - Argument" );
		_feature = new EzVarEnum<ComplexFeature>("Output feature", ComplexFeature.values());
		_mode.addVisibilityTriggerTo(_inputCartesian, Mode.COMPLEX_CARTESIAN);
		_mode.addVisibilityTriggerTo(_inputPolar    , Mode.COMPLEX_POLAR);
		_mode.addVisibilityTriggerTo(_inputReal     , Mode.REAL_CARTESIAN);
		_mode.addVisibilityTriggerTo(_inputImaginary, Mode.REAL_CARTESIAN);
		_mode.addVisibilityTriggerTo(_inputModulus  , Mode.REAL_POLAR);
		_mode.addVisibilityTriggerTo(_inputArgument , Mode.REAL_POLAR);
		
		// Tooltips
		_mode.setToolTipText(
			"The input can be either two sequences of same size, or one sequence with an even number of channels"
		);
		_inputCartesian.setToolTipText(
			"Complex-valued sequence with an even number of channels. The even channels correspond to the real part, " +
			"the odd channels to the imaginary part."
		);
		_inputPolar.setToolTipText(
			"Complex-valued sequence with an even number of channels. The even channels correspond to the modulus, " +
			"the odd channels to the argument."
		);
		_inputReal.setToolTipText(
			"Real-valued sequence, representing the real part of the targeted sequence. " + 
			"Must have the same size than the imaginary part."
		);
		_inputImaginary.setToolTipText(
			"Real-valued sequence, representing the imaginary part of the targeted sequence. " + 
			"Must have the same size than the real part."
		);
		_inputModulus.setToolTipText(
			"Real-valued sequence, representing the modulus of the targeted sequence. " + 
			"Must have the same size than the argument."
		);
		_inputArgument.setToolTipText(
			"Real-valued sequence, representing the argument of the targeted sequence. " + 
			"Must have the same size than the modulus."
		);
		_feature.setToolTipText("Feature to extract or to generate");
		
		// Layout
		addEzComponent(_mode);
		addEzComponent(_inputCartesian);
		addEzComponent(_inputPolar    );
		addEzComponent(_inputReal     );
		addEzComponent(_inputImaginary);
		addEzComponent(_inputModulus  );
		addEzComponent(_inputArgument );
		addEzComponent(_feature);
	}
	
	@Override
	protected void execute()
	{
		try {
			_controller = new Controller();
			Mode mode = _mode.getValue();
			ComplexFeature feature = _feature.getValue();
			Sequence out = null;
			Sequence in  = null;
			Sequence in1 = null;
			Sequence in2 = null;			
			switch(mode) {
				case COMPLEX_CARTESIAN:
					in  = _inputCartesian.getValue();
					out = feature.get(in, ComplexRepresentation.CARTESIAN, _controller);
					break;
				case COMPLEX_POLAR:
					in  = _inputPolar.getValue();
					out = feature.get(in, ComplexRepresentation.POLAR, _controller);
					break;
				case REAL_CARTESIAN:
					in1 = _inputReal     .getValue();
					in2 = _inputImaginary.getValue();
					out = feature.get(in1, in2, ComplexRepresentation.CARTESIAN, _controller);
					break;
				case REAL_POLAR:
					in1 = _inputModulus .getValue();
					in2 = _inputArgument.getValue();
					out = feature.get(in1, in2, ComplexRepresentation.POLAR, _controller);
					break;
			}
			addSequence(out);
		}
		catch(InvalidComplexEntry err) {
			throw new IcyHandledException("Invalid argument: " + err.getMessage());
		}
		catch(Controller.CanceledByUser e) {
			new AnnounceFrame("The complex conversion operation have been canceled");
		}
		finally {
			_controller = null;
		}
	}
	
	@Override
	public void clean() {}

	@Override
	public void stopExecution()
	{
		Controller controller = _controller;
		if(controller!=null) {
			controller.cancelComputation();
		}
	}
}
