package plugins.ylemontag.sequencecomparator;

import icy.sequence.Sequence;
import icy.sequence.SequenceAdapter;
import icy.sequence.SequenceEvent;
import icy.sequence.SequenceListener;
import icy.system.thread.ThreadUtil;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.WeakHashMap;

/**
 * 
 * @author Yoann Le Montagner
 *
 * Comparison between one reference sequence and any number of test videos
 */
public abstract class ComparisonPool<U, T extends Comparator<U>>
{
	/**
	 * Interface to implement to listen state change events on sequences
	 */
	public interface StateChangeListener
	{
		/**
		 * Event handler
		 * @param seq Sequence concerned by the event
		 * @param state New state of the sequence
		 * @param result New result if state==READY, former result if state==NOT_WATCHED
		 */
		public void stateChanged(Sequence seq, ComparisonState state, Object result);
	}
	
	private Object _poolMutex;
	private Sequence _ref;
	boolean _extendZ;
	boolean _extendT;
	private T _comparator;
	private LinkedList<StateChangeListener> _listeners;
	private InfoMap<U> _infos;
	
	/**
	 * Constructor
	 */
	protected ComparisonPool()
	{
		_poolMutex = new Object();
		_extendZ   = false;
		_extendT   = false;
		_listeners = new LinkedList<StateChangeListener>();
		_infos     = new InfoMap<U>();
	}
	
	/**
	 * Add a new listener
	 */
	public void addStateChangeListener(StateChangeListener l)
	{
		synchronized (_poolMutex)
		{
			_listeners.add(l);
		}
	}
	
	/**
	 * Remove a listener
	 */
	public void removeStateChangeListener(StateChangeListener l)
	{
		synchronized (_poolMutex)
		{
			_listeners.remove(l);
		}
	}
	
	/**
	 * Sequence used as reference
	 */
	public Sequence getReference()
	{
		return _ref;
	}
	
	/**
	 * Whether the reference is extended in the Z dimension
	 */
	public boolean getExtendZ()
	{
		return _extendZ;
	}
	
	/**
	 * Whether the reference is extended in the T dimension
	 */
	public boolean getExtendT()
	{
		return _extendT;
	}
	
	/**
	 * Comparator
	 */
	public T getComparator()
	{
		return _comparator;
	}
	
	/**
	 * Distance name
	 */
	public String getDistanceName()
	{
		synchronized (_poolMutex)
		{
			return _comparator==null ? "" : _comparator.getDistanceName();
		}
	}
	
	/**
	 * Return the comparison state of a sequence
	 */
	public ComparisonState getState(Sequence seq)
	{
		synchronized (_poolMutex)
		{
			Info<U> info = _infos.getInfo(seq);
			return info.state;
		}
	}
	
	/**
	 * Return a comparison result, or null if no comparison is currently available
	 */
	public U getResult(Sequence seq)
	{
		synchronized (_poolMutex)
		{
			Info<U> info = _infos.getInfo(seq);
			return info.result;
		}
	}
	
	/**
	 * Return the list of compared sequences (either in state COMPUTING or READY)
	 */
	public Set<Sequence> getComparedSequences()
	{
		synchronized (_poolMutex)
		{
			return _infos.getComparedSequences();
		}
	}
	
	/**
	 * Change the reference, and specify whether it needs to be extend in the
	 * Z and T dimensions
	 */
	public void setReference(Sequence ref, boolean extendZ, boolean extendT)
	{
		synchronized (_poolMutex)
		{
			if(_ref==ref && extendZ==_extendZ && extendT==_extendT) {
				return;
			}
			_ref = ref;
			_extendZ = extendZ;
			_extendT = extendT;
			Set<Sequence> comparedSequences = getComparedSequences();
			for(Sequence seq : comparedSequences) {
				stopComparison(seq);
			}
		}
	}
	
	/**
	 * Change the comparator
	 */
	public void setComparator(T comparator)
	{
		synchronized (_poolMutex)
		{
			if(_comparator==comparator) {
				return;
			}
			_comparator = comparator;
			refreshAll();
		}
	}
	
	/**
	 * Re-do all the comparisons
	 */
	public void refreshAll()
	{
		synchronized (_poolMutex)
		{
			Set<Sequence> comparedSequences = getComparedSequences();
			for(Sequence seq : comparedSequences) {
				startComparisonAsync(seq);
			}
		}
	}
	
	/**
	 * Start to compare two sequences (or force a refresh)
	 * @remark Asynchronous function
	 */
	public void startComparisonAsync(final Sequence seq)
	{
		ThreadUtil.bgRun(new Runnable()
		{
			@Override
			public void run() {
				startComparison(seq);
			}
		});
	}
	
	/**
	 * Start to compare two sequences (or force a refresh)
	 */
	public void startComparison(Sequence seq)
	{
		// Info associated to the sequence
		// Remark: the Info object is allocated once for a given sequence,
		// therefore the call to getInfo can be done outside the
		// synchronized(_poolMutex) block
		Info<U> info = _infos.getInfo(seq);
		
		// On the contrary, the reference sequence as well as the comparator
		// may change between two synchronized(_poolMutex) blocks
		Sequence ref = null;
		Comparator<U> comparator = null;
		
		// Prepare the comparison by setting the internal objects into a 'COMPUTING',
		// and canceling all previous computations
		synchronized (_poolMutex)
		{
			// Ensure that the compared sequences have proper sizes, and save pointers
			// to the reference and the comparator to use
			if(_ref==null || _comparator==null || !CompareUtil.areComparable(_ref, seq, _extendZ, _extendT)) {
				stopComparison(seq);
				return;
			}
			ref = _ref;
			comparator = _comparator;
			
			// Prepare the computation and set the internals
			info.controller.cancelComputation();
			if(info.listener==null) {
				info.listener = new SequenceAdapter()
				{
					@Override
					public void sequenceChanged(SequenceEvent event) {
						onSequenceChanged(event.getSequence());
					}
				};
				seq.addListener(info.listener);
			}
			if(info.result==null) {
				info.result = allocateResult(seq);
			}
			setState(seq, ComparisonState.COMPUTING);
		}
		
		// Try to execute the comparison
		try {
			comparator.compare(ref, seq, info.result, info.controller);
			
			// Update the internal state
			synchronized (_poolMutex) {
				setState(seq, ComparisonState.READY);
			}
		}
		
		// Computation interrupted (by another call to startComparison()) => abort
		catch(Controller.CanceledByUser err) {
			return;
		}
	}
	
	/**
	 * Stop comparing two sequences
	 */
	public void stopComparison(Sequence seq)
	{
		synchronized(_poolMutex)
		{
			Info<U> info = _infos.getInfo(seq);
			if(info.listener!=null) {
				seq.removeListener(info.listener);
				info.listener = null;
			}
			setState(seq, ComparisonState.NOT_WATCHED);
		}
	}
	
	/**
	 * Allocate a result
	 */
	protected abstract U allocateResult(Sequence seq);
	
	/**
	 * Change the comparison state of a sequence
	 */
	private void setState(final Sequence seq, final ComparisonState newState)
	{
		Info<U> info = _infos.getInfo(seq);
		info.state = newState;
		final U result = info.result;
		for(final StateChangeListener l : _listeners)
		{
			ThreadUtil.invokeLater(new Runnable()
			{	
				@Override
				public void run() {
					l.stateChanged(seq, newState, result);
				}
			});
		}
	}
	
	/**
	 * Handler for sequence change events
	 */
	private void onSequenceChanged(Sequence seq)
	{
		if(seq==_ref) {
			refreshAll();
		}
		else if(getState(seq)!=ComparisonState.NOT_WATCHED) {
			startComparison(seq);
		}
	}
	
	/**
	 * Info associated to a compared sequence
	 */
	private static class Info<U>
	{
		ComparisonState  state     ; // never null
		SequenceListener listener  ;
		Controller       controller; // never null
		U                result    ;
	}
	
	/**
	 * Wrap a weak hash map that holds informations about sequences
	 */
	private static class InfoMap<U>
	{
		private WeakHashMap<Sequence, Info<U>> _map;
		
		public InfoMap()
		{
			_map = new WeakHashMap<Sequence, Info<U>>();
		}
		
		public Info<U> getInfo(Sequence seq)
		{
			synchronized (_map)
			{
				Info<U> info = _map.get(seq);
				if(info==null) {
					info = new Info<U>();
					info.state      = ComparisonState.NOT_WATCHED;
					info.controller = new Controller();
					_map.put(seq, info);
				}
				return info;
			}
		}
		
		public Set<Sequence> getComparedSequences()
		{
			synchronized (_map)
			{
				Set<Sequence> retVal = new HashSet<Sequence>();
				for(Sequence s : _map.keySet()) {
					if(_map.get(s).state!=ComparisonState.NOT_WATCHED) {
						retVal.add(s);
					}
				}
				return retVal;
			}
		}
	}
}
