/*******************************************************************************
 * Copyright (c) 2012-2013 Biomedical Image Group (BIG), EPFL, Switzerland.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Public License v3.0
 * which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/gpl.html
 * 
 * Contributors:
 *     Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 *     Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 ******************************************************************************/
package plugins.big.steerablej.gui;

import icy.image.IcyBufferedImage;
import icy.sequence.Sequence;

/**
 * Class that displays a preview of a steerable filter on a canvas. Based on the
 * <code>Preview</code> package of Francois Aguet.
 * 
 * @version April 25, 2013
 * 
 * @author Ricard Delgado-Gonzalo (ricard.delgado@gmail.com)
 * @author Zsuzsanna Puspoki (zsuzsanna.puspoki@epfl.ch)
 */
public class FilterPreview {

	// ----------------------------------------------------------------------------
	// IMAGE FIELDS

	/** Pixels of the preview image. */
	private double[] pixels_ = null;

	/** Width in pixels of the preview image. */
	public static final int PREVIEW_WIDTH = 141;
	/** Height in pixels of the preview image. */
	public static final int PREVIEW_HEIGHT = 141;

	// ----------------------------------------------------------------------------
	// FILTER PARAMETERS

	/**
	 * Standard deviation of the Gaussian on which the steerable templates are
	 * based.
	 */
	private static final double SIGMA = (PREVIEW_WIDTH - 1) / 6.0;
	/** Order of the feature template. */
	private int M_ = 0;

	// ----------------------------------------------------------------------------
	// AUXILIARY FIELDS

	/** Minimum value of the filter. */
	private double min_ = Double.MAX_VALUE;
	/** Maximum value of the filter. */
	private double max_ = -Double.MAX_VALUE;

	/**
	 * Array containing the first partial derivative of the Gaussian kernel. The
	 * 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gyPixels_ = null;
	/**
	 * Array containing the second partial derivative of the Gaussian kernel.
	 * The 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gxxPixels_ = null;
	/**
	 * Array containing the second partial derivative of the Gaussian kernel.
	 * The 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gyyPixels_ = null;
	/**
	 * Array containing the third partial derivative of the Gaussian kernel. The
	 * 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gxxyPixels_ = null;
	/**
	 * Array containing the third partial derivative of the Gaussian kernel. The
	 * 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gyyyPixels_ = null;
	/**
	 * Array containing the fourth partial derivative of the Gaussian kernel.
	 * The 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gxxxxPixels_ = null;
	/**
	 * Array containing the fourth partial derivative of the Gaussian kernel.
	 * The 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gxxyyPixels_ = null;
	/**
	 * Array containing the fourth partial derivative of the Gaussian kernel.
	 * The 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gyyyyPixels_ = null;
	/**
	 * Array containing the fifth partial derivative of the Gaussian kernel. The
	 * 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gxxxxyPixels_ = null;
	/**
	 * Array containing the fifth partial derivative of the Gaussian kernel. The
	 * 'x' denotes first spatial dimension, and 'y' denotes second spatial
	 * dimension.
	 */
	private double[] gxxyyyPixels_ = null;

	// ============================================================================
	// PUBLIC METHODS

	/** Constructor. */
	public FilterPreview(double[] alpha, int M) {
		M_ = M;
		int size = PREVIEW_WIDTH * PREVIEW_HEIGHT;
		pixels_ = new double[size];
		gyPixels_ = new double[size];
		gxxyPixels_ = new double[size];
		gyyyPixels_ = new double[size];
		gxxxxyPixels_ = new double[size];
		gxxyyyPixels_ = new double[size];
		gxxPixels_ = new double[size];
		gyyPixels_ = new double[size];
		gxxxxPixels_ = new double[size];
		gxxyyPixels_ = new double[size];
		gyyyyPixels_ = new double[size];
		generateTemplates();
		adjustContrast();
		createImage(pixels_);
		updatePixels(alpha);
	}

	// ----------------------------------------------------------------------------

	/**
	 * Returns a <code>Sequence</code> object with the preview of the filter
	 * with new parameters.
	 */
	public Sequence getPreview(double[] alpha, int M) {
		M_ = M;
		updatePixels(alpha);
		adjustContrast();
		IcyBufferedImage image = createImage(pixels_);
		return new Sequence(image);
	}

	// ============================================================================
	// PRIVATE METHODS

	/** Scales the preview pixel array between 0 and 255. */
	private void adjustContrast() {
		refreshMinAndMax();
		double d = max_ - min_;
		for (int i = 0; i < pixels_.length; i++) {
			pixels_[i] = (pixels_[i] - min_) * 255.0 / d;
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Creates the <code>IcyBufferedImage</code> object from an array of pixels.
	 */
	private IcyBufferedImage createImage(double[] pixels) {
		int array[] = new int[pixels.length];
		int value;
		for (int i = 0; i < pixels.length; i++) {
			if (pixels[i] < 0.0) {
				array[i] = 0;
			} else if (pixels[i] > 255.0) {
				array[i] = 0xFFFFFFFF;
			} else {
				value = (int) pixels[i] & 0xFF;
				array[i] = value | (value << 8) | (value << 16) | (255 << 24);
			}
		}
		return new IcyBufferedImage(PREVIEW_WIDTH, PREVIEW_HEIGHT, array, true);
	}

	// ----------------------------------------------------------------------------

	/** Generates the filter templates. */
	private void generateTemplates() {
		int cx = (PREVIEW_WIDTH + 1) / 2;
		int cy = (PREVIEW_HEIGHT + 1) / 2;
		int xi, yi, xi2, yi2;
		int i;
		double g;
		double sigma2 = SIGMA * SIGMA;
		double sigma4 = sigma2 * sigma2;
		for (int x = 0; x < PREVIEW_WIDTH; x++) {
			for (int y = 0; y < PREVIEW_HEIGHT; y++) {
				i = x + y * PREVIEW_WIDTH;
				xi = x - cx;
				yi = y - cy;
				xi2 = xi * xi;
				yi2 = yi * yi;
				g = Math.exp(-(xi2 + yi2) / (2.0 * sigma2))
						/ (2.0 * Math.PI * sigma4);
				gyPixels_[x + y * PREVIEW_WIDTH] = -yi * g;
				g /= sigma2;
				gxxPixels_[i] = g * (xi2 - sigma2);
				gyyPixels_[i] = g * (yi2 - sigma2);
				g /= sigma2;
				gxxyPixels_[i] = g * yi * (sigma2 - xi2);
				gyyyPixels_[i] = g * yi * (3.0 * sigma2 - yi2);
				g /= sigma2;
				gxxxxPixels_[i] = g
						* (xi2 * xi2 - 6.0 * xi2 * sigma2 + 3.0 * sigma4);
				gxxyyPixels_[i] = g * (sigma2 - xi2) * (sigma2 - yi2);
				gyyyyPixels_[i] = g
						* (yi2 * yi2 - 6.0 * yi2 * sigma2 + 3.0 * sigma4);
				gxxxxyPixels_[i] = -yi * gxxxxPixels_[i] / sigma2;
				gxxyyyPixels_[i] = g * yi * (yi2 - 3.0 * sigma2)
						* (sigma2 - xi2) / sigma2;
			}
		}
	}

	// ----------------------------------------------------------------------------

	/** Recomputes the minimum and maximum values within the filter template. */
	private void refreshMinAndMax() {
		min_ = Double.MAX_VALUE;
		max_ = -Double.MAX_VALUE;
		double t;
		for (int i = 0; i < pixels_.length; i++) {
			t = pixels_[i];
			if (t < min_) {
				min_ = t;
			}
			if (t > max_) {
				max_ = t;
			}
		}
	}

	// ----------------------------------------------------------------------------

	/**
	 * Updates the pixel array containing the filter template accordingly to a
	 * given set of weights.
	 */
	private void updatePixels(double[] alpha) {
		int i;
		double a20, a22, a31, a33, a40, a42, a44, a51, a53;
		double sigma2 = SIGMA * SIGMA;
		double sigma3 = SIGMA * sigma2;
		double sigma4 = sigma2 * sigma2;

		switch (M_) {
		case 1:
			for (i = 0; i < pixels_.length; i++) {
				pixels_[i] = alpha[0] * gyPixels_[i];
			}
			break;
		case 2:
			a20 = alpha[0] * SIGMA;
			a22 = alpha[1] * SIGMA;
			for (i = 0; i < pixels_.length; i++) {
				pixels_[i] = a20 * gxxPixels_[i] + a22 * gyyPixels_[i];
			}
			break;
		case 3:
			a31 = alpha[1] * sigma2;
			a33 = alpha[2] * sigma2;
			for (i = 0; i < pixels_.length; i++) {
				pixels_[i] = alpha[0] * gyPixels_[i] + a31 * gxxyPixels_[i]
						+ a33 * gyyyPixels_[i];
			}
			break;
		case 4:
			a20 = alpha[0] * SIGMA;
			a22 = alpha[1] * SIGMA;
			a40 = alpha[2] * sigma3;
			a42 = alpha[3] * sigma3;
			a44 = alpha[4] * sigma3;
			for (i = 0; i < pixels_.length; i++) {
				pixels_[i] = a20 * gxxPixels_[i] + a22 * gyyPixels_[i] + a40
						* gxxxxPixels_[i] + a42 * gxxyyPixels_[i] + a44
						* gyyyyPixels_[i];
			}
			break;
		case 5:
			a31 = alpha[1] * sigma2;
			a33 = alpha[2] * sigma2;
			a51 = alpha[3] * sigma4;
			a53 = alpha[4] * sigma4;
			for (i = 0; i < pixels_.length; i++) {
				pixels_[i] = alpha[0] * gyPixels_[i] + a31 * gxxyPixels_[i]
						+ a33 * gyyyPixels_[i] + a51 * gxxxxyPixels_[i] + a53
						* gxxyyyPixels_[i];
			}
			break;
		default:
			break;
		}
	}
}
